//
// BE memory extension for viewing data obtained by an external program
//
// The cache holds accumulated data (and holes) learn't whilst running.
// Newly obtained data are added to the head of the list.
// Holes found enlarge existing holes, or are appended to the tail of the list.
//
// The cache grows indefinitely, until the program stops, or the user selects
// the 'hard refresh' key (capital R), to invalidate the cache.
//
// Data cached may not be edited.
//

/*...sincludes:0:*/
#ifdef NO_CINCLUDES
  #include <stdio.h>
  #include <ctype.h>
  #include <string.h>
  #include <stddef.h>
  #include <stdlib.h>
  #include <stdarg.h>
#else
  #include <cstdio>
  #include <cctype>
  #include <cstring>
  #include <cstddef>
  #include <cstdlib>
  #include <cstdarg>
#endif

#include <fcntl.h>
#if defined(AIX) || defined(LINUX)
  #include <unistd.h>
#else
  #include <io.h>
#endif

#include "bememext.h"
/*...e*/
/*...sPOSIX\47\ISO:0:*/
#ifdef WIN32
  #define fd_creat _creat
  #define fd_open  _open
  #define fd_close _close
  #define fd_read  _read
  #define fd_write _write
  #define fd_dup   _dup
  #define fd_dup2  _dup2
#else
  #define fd_creat creat
  #define fd_open  open
  #define fd_close close
  #define fd_read  read
  #define fd_write write
  #define fd_dup   dup
  #define fd_dup2  dup2
#endif
/*...e*/

/*...smachine dependent parts:0:*/
#if defined(UNIX)
/*...sget_tmp_fn:0:*/
const char *get_tmp_fn(void)
	{
	static char templ[14];
	strcpy(templ, "/tmp/beXXXXXX");
	return mktemp(templ);
	}
/*...e*/
#endif

#if defined(OS2) || defined(WIN32) || defined(DOS32) || defined(NW)
/*...sget_tmp_fn:0:*/
const char * get_tmp_fn(void)
	{
	return tmpnam(NULL);
	}
/*...e*/
#endif

#if defined(OS2)
/*...sshell:0:*/
#define	FD_STDIN  0
#define	FD_STDOUT 1
#define	FD_STDERR 2

static int fd_stdin, fd_stdout, fd_stderr;

static void take_copies()
	{
	fd_stdin  = fd_dup(FD_STDIN);
	fd_stdout = fd_dup(FD_STDOUT);
	fd_stderr = fd_dup(FD_STDERR);
	}

static void reconnect()
	{
	fd_dup2(fd_stdin,  FD_STDIN);  fd_close(fd_stdin);
	fd_dup2(fd_stdout, FD_STDOUT); fd_close(fd_stdout);
	fd_dup2(fd_stderr, FD_STDERR); fd_close(fd_stderr);
	}

static const char * shell_to_use()
	{
	const char *p;
	return (p = getenv("COMSPEC")) != NULL ? p : "CMD";
	}

Boolean	shell(const char *command, const char *out_fn)
	{
	int tty, result, fd_out;

	/* Connect to console */

	tty = fd_open("con", O_RDWR);	/* Both input and output */
	fd_out = fd_creat(out_fn, 0600);

	take_copies();

	fd_dup2(tty, FD_STDIN);
	fd_dup2(fd_out, FD_STDOUT);
	fd_dup2(tty, FD_STDERR);

	/* Perform command */

	result = system( ( *command ) ? command : shell_to_use() );

	reconnect();

	/* Close connection to console */

	fd_close(tty);
	fd_close(fd_out);

	return result != -1;
	}
/*...e*/
#elif defined(WIN32)
/*...sshell:0:*/
#define	FD_STDIN  0
#define	FD_STDOUT 1
#define	FD_STDERR 2

static int fd_stdin, fd_stdout, fd_stderr;

static void take_copies()
	{
	fd_stdin  = fd_dup(FD_STDIN);
	fd_stdout = fd_dup(FD_STDOUT);
	fd_stderr = fd_dup(FD_STDERR);
	}

static void reconnect()
	{
	fd_dup2(fd_stdin,  FD_STDIN);  fd_close(fd_stdin);
	fd_dup2(fd_stdout, FD_STDOUT); fd_close(fd_stdout);
	fd_dup2(fd_stderr, FD_STDERR); fd_close(fd_stderr);
	}

static const char * shell_to_use()
	{
	const char *p;
	return (p = getenv("COMSPEC")) != NULL ? p : "CMD";
	}

Boolean	shell(const char *command, const char *out_fn)
	{
	int tty, result, fd_out;

	/* Connect to console */

	tty = fd_open("con", O_RDWR);	/* Both input and output */
	fd_out = fd_creat(out_fn, 0600);

	take_copies();

	fd_dup2(tty, FD_STDIN);
	fd_dup2(fd_out, FD_STDOUT);
	fd_dup2(tty, FD_STDERR);

	/* Perform command */

	result = system( ( *command ) ? command : shell_to_use() );

	reconnect();

	/* Close connection to console */

	fd_close(tty);
	fd_close(fd_out);

	return result != -1;
	}
/*...e*/
#elif defined(DOS32) || defined(NW)
/*...sshell:0:*/
#define	FD_STDIN  0
#define	FD_STDOUT 1
#define	FD_STDERR 2

static int fd_stdin, fd_stdout, fd_stderr;

static void take_copies()
	{
	fd_stdin  = fd_dup(FD_STDIN);
	fd_stdout = fd_dup(FD_STDOUT);
	fd_stderr = fd_dup(FD_STDERR);
	}

static void reconnect()
	{
	fd_dup2(fd_stdin,  FD_STDIN);  fd_close(fd_stdin);
	fd_dup2(fd_stdout, FD_STDOUT); fd_close(fd_stdout);
	fd_dup2(fd_stderr, FD_STDERR); fd_close(fd_stderr);
	}

static const char * shell_to_use()
	{
	const char *p;
	return (p = getenv("COMSPEC")) != NULL ? p : "COMMAND";
	}

Boolean	shell(const char *command, const char *out_fn)
	{
	int tty, result, fd_out;

	/* Connect to console */

	tty = fd_open("con", O_RDWR);	/* Both input and output */
	fd_out = fd_creat(out_fn, 0600);

	take_copies();

	fd_dup2(tty, FD_STDIN);
	fd_dup2(fd_out, FD_STDOUT);
	fd_dup2(tty, FD_STDERR);

	/* Perform command */

	result = system( ( *command ) ? command : shell_to_use() );

	reconnect();

	/* Close connection to console */

	fd_close(tty);
	fd_close(fd_out);

	return result != -1;
	}
/*...e*/
#elif defined(UNIX)
/*...sshell:0:*/
#define	FD_STDIN  0
#define	FD_STDOUT 1
#define	FD_STDERR 2

static int fd_stdin, fd_stdout, fd_stderr;

static void take_copies()
	{
	fd_stdin  = fd_dup(FD_STDIN);
	fd_stdout = fd_dup(FD_STDOUT);
	fd_stderr = fd_dup(FD_STDERR);
	}

static void reconnect()
	{
	fd_dup2(fd_stdin,  FD_STDIN);  fd_close(fd_stdin);
	fd_dup2(fd_stdout, FD_STDOUT); fd_close(fd_stdout);
	fd_dup2(fd_stderr, FD_STDERR); fd_close(fd_stderr);
	}

static const char *shell_to_use()
	{
	const char *p;
	return (p = getenv("SHELL")) != NULL ? p : "sh";
	}

Boolean	shell(const char *command, const char *out_fn)
	{
	int tty, result, fd_out;

	tty = fd_open("/dev/tty", O_RDWR);
	fd_out = fd_creat(out_fn, 0600);

	take_copies();

	fd_dup2(tty, FD_STDIN);
	fd_dup2(fd_out, FD_STDOUT);
	fd_dup2(tty, FD_STDERR);

	/* Perform command */

	result = system( ( *command ) ? command : shell_to_use() );

	reconnect();

	/* Close connection to console */

	fd_close(tty);
	fd_close(fd_out);

	return result != -1;
	}
/*...e*/
#endif
/*...e*/
/*...sADDR:0:*/
#ifdef BE64
typedef BEMEMADDR64 ADDR;
#else
typedef BEMEMADDR32 ADDR;
#endif

/*...shex_of:0:*/
static char hexdigits[] = "0123456789abcdef";

static unsigned int hex_of(char c)
	{
	return strchr(hexdigits, tolower(c)) - hexdigits;
	}
/*...e*/
/*...sscan_dec:0:*/
static ADDR scan_dec(const char *buf)
	{
	ADDR i = 0;
	for ( ; *buf >= '0' && *buf <= '9'; buf++ )
		i = i * 10 + (*buf - '0');
	return i;
	}
/*...e*/
/*...sscan_hex:0:*/
static ADDR scan_hex(const char *buf)
	{
	ADDR i = 0;
	for ( ; isxdigit(*buf); buf++ )
		i = (i << 4) + hex_of(*buf);
	return i;
	}
/*...e*/
/*...sscan:0:*/
static ADDR scan(const char *buf)
	{
	if ( buf[0] == '0' && (buf[1] == 'x' || buf[1] == 'X') )
		return scan_hex(buf+2);
	else
		return scan_dec(buf);
	}
/*...e*/
/*...sput_hex_v:0:*/
static char *put_hex_v(char *to, ADDR i)
	{
	*to++ = '0';
	if ( i != (ADDR) 0 )
		{
		char buf[(sizeof(ADDR)*8+3)/4];
		unsigned inx = sizeof(buf);
		while ( i != (ADDR) 0 )
			{
			buf[--inx] = hexdigits[i & 15];
			i >>= 4;
			}
		*to++ = 'x';
		while ( inx < sizeof(buf) )
			*to++ = buf[inx++];
		}
	return to;
	}
/*...e*/
/*...e*/

/*...senum CacheLookup:0:*/
enum CacheLookup { NotInCache, DataInCache, HoleInCache };
/*...e*/
/*...sclass CacheRange:0:*/
// All the internal fields of this class are exposed, at it is manipulated
// by functions other than methods. This is because many of these operate
// upon lists and pairs rather than just single instances.

class CacheRange
	{
public:
	CacheRange *next;
	ADDR low_addr, high_addr;
	unsigned char *data;
	CacheRange(
		ADDR low_addr,
		ADDR high_addr,
		unsigned char *data = 0
		)
		:
		next(0), low_addr(low_addr), high_addr(high_addr), data(data)
		{
		}
	~CacheRange()
		{
		if ( data != 0 )
			delete[] data;
		}
	CacheLookup read(ADDR addr, unsigned char & b) const
		{
		if ( addr < low_addr || addr >= high_addr )
			return NotInCache;
		else if ( data != 0 )
			{
			b = data[addr-low_addr];
			return DataInCache;
			}
		else
			return HoleInCache;
		}
	};
/*...e*/
/*...sclass Cache:0:*/
/*...sdrop:0:*/
static void drop(CacheRange *r)
	{
	if ( r != 0 )
		{
		drop(r->next);
		delete r;
		}
	}
/*...e*/
/*...sexclude:0:*/
// Given a cache chain, drop data/holes in it for a given memory range

static CacheRange *exclude(CacheRange *r, ADDR low_addr, ADDR high_addr)
	{
	if ( r == 0 )
		return 0;
	r->next = exclude(r->next, low_addr, high_addr);
	if ( low_addr >= r->high_addr || high_addr <= r->high_addr )
		// No overlap of this block with exclude zone
		return r;
	else if ( low_addr <= r->low_addr && high_addr >= r->high_addr )
		// Exclude zone completely covers this block, forget it
		{
		delete r;
		return r->next;
		}
	else if ( low_addr <= r->low_addr && high_addr < r->high_addr )
		// Exclude zone covers start of this block
		{
		ADDR lose = high_addr - r->low_addr;
		ADDR keep = r->high_addr - r->low_addr - lose;
		unsigned char *buf = new unsigned char[(size_t) keep];
		memcpy(buf, r->data + lose, (size_t) keep);
		delete[] r->data;
		r->data = buf;
		r->low_addr = high_addr;
		return r;
		}
	else if ( low_addr < r->high_addr && high_addr <= r->high_addr )
		// Exclude zone covers end of this block
		{
		ADDR keep = low_addr - r->low_addr;
		unsigned char *buf = new unsigned char[(size_t) keep];
		memcpy(buf, r->data, (size_t) keep);
		delete[] r->data;
		r->data = buf;
		r->high_addr = low_addr;
		return r;
		}
	else
		// Exclude zone is in the middle of the block
		{
		ADDR keep1 = low_addr - r->low_addr;
		unsigned char *buf1 = new unsigned char[(size_t) keep1];
		memcpy(buf1, r->data, (size_t) keep1);
		CacheRange *r1 = new CacheRange(r->low_addr, low_addr, buf1);
		ADDR keep2 = r->high_addr - high_addr;
		unsigned char *buf2 = new unsigned char[(size_t) keep2];
		memcpy(buf2, r->data + high_addr - r->low_addr, (size_t) keep2);
		CacheRange *r2 = new CacheRange(high_addr, r->high_addr, buf2);
		r2->next = r->next;
		r1->next = r2;
		delete r;
		return r1;
		}
	}
/*...e*/
/*...senlarge:0:*/
// Upon calling this it is known that addr is not in the cache chain already,
// as either data or a hole. Add a record of a hole at the given address.
// Prefer to enlarge existing hole record, above creating a new record.

static CacheRange *enlarge(CacheRange *r, ADDR addr)
	{
	if ( r == 0 )
		return new CacheRange(addr, addr + 1);
	else if ( r->data == 0 && addr == r->high_addr )
		{ ++(r->high_addr); return r; }
	else if ( r->data == 0 && addr == r->low_addr - 1 )
		{ --(r->low_addr); return r; }
	else
		{
		r->next = enlarge(r->next, addr);
		return r;
		}
	}
/*...e*/

class Cache
	{
	CacheRange *ranges;
public:
	Cache() : ranges(0) {}
	~Cache() { refresh(); }
	void refresh() { drop(ranges); ranges = 0; }
	CacheLookup read(ADDR addr, unsigned char & b) const;
	void add_file(ADDR low_addr, const char *fn);
	void add_hole(ADDR addr);
	};

/*...sread:0:*/
CacheLookup Cache::read(ADDR addr, unsigned char & b) const
	{
	for ( CacheRange *r = ranges; r != 0; r = r->next )
		{
		CacheLookup rc;
		if ( (rc = r->read(addr, b)) != NotInCache )
			return rc;
		}
	return NotInCache;
	}
/*...e*/
/*...sadd_file:0:*/
void Cache::add_file(ADDR low_addr, const char *fn)
	{
	FILE *fp;
	if ( (fp = fopen(fn, "rb")) == 0 )
		return;
	fseek(fp, 0L, SEEK_END);
	ADDR length = ftell(fp);
	fseek(fp, 0L, SEEK_SET);
	unsigned char *data;
	if ( (data = new unsigned char[(size_t) length]) == 0 )
		{ fclose(fp); return; }
	fread(data, (size_t) length, 1, fp);
	fclose(fp);
	CacheRange *r = new CacheRange(low_addr, low_addr+length, data);
	r->next = exclude(ranges, r->low_addr, r->high_addr);
	ranges = r;
	}
/*...e*/
/*...sadd_hole:0:*/
void Cache::add_hole(ADDR addr)
	{
	ranges = enlarge(ranges, addr);
	}
/*...e*/
/*...e*/
/*...sclass BeMem:0:*/
class BeMem
	{
	ADDR low_addr, high_addr;
	char command[500+1];
	Cache cache;
public:
	BeMem(ADDR low_addr, ADDR high_addr, const char *command)
		:
		low_addr(low_addr), high_addr(high_addr)
		{
		strcpy(BeMem::command, command);
		}
	~BeMem() { cache.refresh(); }
	Boolean read(ADDR addr, unsigned char & b)
		{
		if ( addr < low_addr || addr >= high_addr )
			return FALSE; // Data not supplied by this instance
		CacheLookup rc = cache.read(addr, b);
		switch ( rc )
			{
			case DataInCache:
				return TRUE;
			case HoleInCache:
				return FALSE;
			case NotInCache:
				break;
			}

		// Determine temporary file name and create it
		char addr_fn[100];
		strcpy(addr_fn, get_tmp_fn());
		FILE *fp;
		if ( (fp = fopen(addr_fn, "w")) == 0 )
			return FALSE;
		fclose(fp);

		// Determine another temporary file name
		char data_fn[100+1];
		strcpy(data_fn, get_tmp_fn());

		char cmd[500+1];
		char *p;
		strcpy(cmd, command);
		p = cmd + strlen(cmd);
		*p++ = ' ';
		p = put_hex_v(p, addr);
		*p++ = ' ';
		strcpy(p, data_fn);
		if ( !shell(cmd, addr_fn) )
			return FALSE; // Couldn't run command

		if ( (fp = fopen(addr_fn, "r")) == 0 )
			return FALSE; // Command didn't generate output file

		char line[100+1];
		if ( fgets(line, 100+1, fp) == NULL )
			// Insufficient or invalid output
			{
			fclose(fp);
			remove(addr_fn);
			remove(data_fn);
			return FALSE;
			}
		fclose(fp);
		remove(addr_fn);

		ADDR r_addr = scan(line);
		cache.add_file(r_addr, data_fn);
		if ( cache.read(addr, b) == DataInCache )
			// Data successfully brought into the cache
			{
			remove(data_fn);
			return TRUE;
			}

		remove(data_fn);
		cache.add_hole(addr);
		return FALSE;
		}
	void refresh()
		{
		cache.refresh();
		}
	};
/*...e*/

/*...sbemem_refresh:0:*/
BEMEMEXPORT void BEMEMENTRY bemem_refresh(void * ptr)
	{
	BeMem *bemem = (BeMem *) ptr;
	bemem->refresh();
	}
/*...e*/
/*...sbemem_read:0:*/
#ifdef BE64
BEMEMEXPORT Boolean BEMEMENTRY bemem_read_64(void * ptr, BEMEMADDR64 addr, unsigned char & b)
#else
BEMEMEXPORT Boolean BEMEMENTRY bemem_read(void * ptr, BEMEMADDR32 addr, unsigned char & b)
#endif
	{
	BeMem *bemem = (BeMem *) ptr;
	return bemem->read(addr, b);
	}
/*...e*/
/*...sbemem_create:0:*/
#ifdef BE64
BEMEMEXPORT void * BEMEMENTRY bemem_create_64(const char *args, BEMEMADDR64 addr, const char *(&err))
#else
BEMEMEXPORT void * BEMEMENTRY bemem_create(const char *args, BEMEMADDR32 addr, const char *(&err))
#endif
	{
	BeMem *bemem;
	char l[100+1];
	char command[500+1];
	if ( sscanf(args, "%[^,],%s", l, command) < 2 )
		{ err = "bad arguments"; return 0; }
	ADDR length = scan(l);
	bemem = new BeMem(addr, (ADDR) (addr + length), command);
	if ( bemem == 0 )
		{ err = "out of memory"; return 0; }
	return (void *) bemem;
	}

/*...e*/
/*...sbemem_delete:0:*/
BEMEMEXPORT void BEMEMENTRY bemem_delete(void * ptr)
	{
	BeMem *bemem = (BeMem *) ptr;
	delete bemem;
	}
/*...e*/

/*...sOS specifics:0:*/
#ifdef DOS32
int main(int term) { term=term; return 0; }
#endif

#ifdef AIX

extern "C" {

BEMEM_EXPORT * __start(void)
	{
	static BEMEM_EXPORT exports[] =
		{
		(BEMEM_EP) bemem_refresh  , "bemem_refresh"  ,
#ifdef BE64
		(BEMEM_EP) bemem_read_64  , "bemem_read_64"  ,
		(BEMEM_EP) bemem_create_64, "bemem_create_64",
#else
		(BEMEM_EP) bemem_read     , "bemem_read"     ,
		(BEMEM_EP) bemem_create   , "bemem_create"   ,
#endif
		(BEMEM_EP) bemem_delete   , "bemem_delete"   ,
		(BEMEM_EP) 0              , 0
		};
	return exports;
	}

}

#endif

#ifdef NW

// Rather ugly mechanism required to make NLMs behave like DLLs under NetWare.

#include <conio.h>
#include <process.h>
#include <advanced.h>

static int tid;

extern "C" BEMEMEXPORT void BEMEMENTRY _bemem_term(void);

BEMEMEXPORT void BEMEMENTRY _bemem_term(void)
	{
	ResumeThread(tid);
	}

int main(int argc, char *argv[])
	{
	argc=argc; argv=argv; // Suppress warnings
	int nid = GetNLMID();
	SetAutoScreenDestructionMode(TRUE);
	SetNLMDontUnloadFlag(nid);
	tid = GetThreadID();
	SuspendThread(tid);
	ClearNLMDontUnloadFlag(nid);
	return 0;
	}

#endif
/*...e*/
