//
// berawio.C - BE memory extension for raw I/O to block devices
//
// This can work in
//   * read-only mode, where no changes can be made
//   * read-write mode, where changes are made immediately
//   * read-write mode with lazy-write, where changes occur when we switch
//     to a new sector, or we flush
//

/*...sincludes:0:*/
#include <cstdio>
#include <cctype>
#include <cstring>
#include <cstddef>
#include <cstdlib>
#include <cstdarg>

#include <windows.h>
#include <winioctl.h>

#include "bememext.h"
/*...e*/

/*...sclass BeMem:0:*/
class BeMem
	{
	HANDLE hdisk;
	unsigned long long base_addr;
	unsigned sector_size;
	unsigned long long sector;
	unsigned char buffer[4096];
	Boolean read_only;
	Boolean lazy_write;
	Boolean empty;
	Boolean dirty; // can only be TRUE when !empty
	Boolean read_sector()
		{
		LARGE_INTEGER li;
		li.QuadPart = sector*sector_size;
		if ( ! SetFilePointerEx(
			hdisk,
			li,
			NULL,
			FILE_BEGIN
			) )
			return FALSE;
		DWORD dwBytes;
		if ( ! ReadFile(
			hdisk,
			buffer,
			(DWORD)sector_size,
			&dwBytes,
			NULL
			) )
			return FALSE;
		if ( dwBytes < sector_size )
			return FALSE;
		return TRUE;
		}
	Boolean write_sector()
		{
		LARGE_INTEGER li;
		li.QuadPart = sector*sector_size;
		if ( ! SetFilePointerEx(
			hdisk,
			li,
			NULL,
			FILE_BEGIN
			) )
			return FALSE;
		DWORD dwBytes;
		if ( ! WriteFile(
			hdisk,
			buffer,
			(DWORD)sector_size,
			&dwBytes,
			NULL
			) )
		if ( dwBytes < sector_size )
			return FALSE;
		return TRUE;
		}
	Boolean flush_sector()
		{
		if ( ! empty && dirty )
			{
			if ( ! write_sector() )
				return FALSE;
			dirty = FALSE;
			}
		return TRUE;
		}
	Boolean select_sector(unsigned long long new_sector)
		{
		if ( new_sector != sector )
			{
			if ( ! flush_sector() )
				// lazy-write has failed
				return FALSE;
			empty = TRUE;
			dirty = FALSE;
			}
		if ( empty )
			{
			sector = new_sector;
			if ( ! read_sector() )
				return FALSE;
			empty = FALSE;
			}
		return TRUE;
		}
public:
	BeMem(
		HANDLE hdisk,
		unsigned long long base_addr,
		unsigned sector_size,
		Boolean read_only,
		Boolean lazy_write
		) :
		hdisk(hdisk),
		base_addr(base_addr),
		sector_size(sector_size),
		read_only(read_only),
		lazy_write(lazy_write),
		empty(TRUE),
		dirty(FALSE)
		{}
	~BeMem()
		{
		DWORD dwResult;
		DeviceIoControl(
			hdisk,
			FSCTL_UNLOCK_VOLUME,
			NULL,
			0,
			NULL,
			0,
			&dwResult,
			NULL
			);
		DeviceIoControl(
			hdisk,
			FSCTL_DISMOUNT_VOLUME,
			NULL,
			0,
			NULL,
			0,
			&dwResult,
			NULL
			);
		CloseHandle(hdisk);
		}
	Boolean read(unsigned long long addr, unsigned char & b)
		{
		addr += base_addr;
		unsigned long long new_sector = addr/sector_size;
		if ( ! select_sector(new_sector) )
			return FALSE;
		b = buffer[addr%sector_size];
		return TRUE;
		}
	Boolean write(unsigned long long addr, unsigned char b)
		{
		if ( read_only )
			return FALSE;
		addr += base_addr;
		unsigned long long new_sector = addr/sector_size;
		if ( ! select_sector(new_sector) )
			return FALSE;
		unsigned char *pbuf = buffer+(addr%sector_size);
		unsigned char b_saved = *pbuf;
		*pbuf = b;
		if ( lazy_write )
			dirty = TRUE; // defer write until later
		else
			{
			if ( ! write_sector() )
				{
				*pbuf = b_saved;
				return FALSE;
				}
			}
		return TRUE;
		}
	void refresh()
		{
		if ( !empty && !dirty )
			empty = TRUE;
		}
	Boolean flush()
		{
		return flush_sector();
		}
	};
/*...e*/

/*...sbemem_read_64:0:*/
BEMEMEXPORT Boolean BEMEMENTRY bemem_read_64(void * ptr, BEMEMADDR64 addr, unsigned char & b)
	{
	BeMem *bemem = (BeMem *) ptr;
	return bemem->read(addr, b);
	}
/*...e*/
/*...sbemem_write_64:0:*/
BEMEMEXPORT Boolean BEMEMENTRY bemem_write_64(void * ptr, BEMEMADDR64 addr, unsigned char b)
	{
	BeMem *bemem = (BeMem *) ptr;
	return bemem->write(addr, b);
	}
/*...e*/
/*...sbemem_create_64:0:*/
/*...svalidBytesPerSector:0:*/
static Boolean validBytesPerSector(int bps)
	{
	return bps ==  128 ||
	       bps ==  256 ||
	       bps ==  512 ||
	       bps == 1024 ||
	       bps == 2048 ||
	       bps == 4096 ;
	}
/*...e*/

BEMEMEXPORT void * BEMEMENTRY bemem_create_64(const char *args, BEMEMADDR64 addr, const char *(&err))
	{
	Boolean read_only  = FALSE;
	Boolean lazy_write = FALSE;
	while ( strchr(args, ':') != NULL )
		if ( !memcmp(args, "RO:", 3) )
			{ read_only = TRUE; args += 3; }
		else if ( !memcmp(args, "LW:", 3) )
			{ lazy_write = TRUE; args += 3; }
		else
			break;

	HANDLE hdisk;
	DWORD dwResult;
	DISK_GEOMETRY dg;
	if ( (hdisk = CreateFileA(
		args,
		read_only ? GENERIC_READ : (GENERIC_READ|GENERIC_WRITE),
		FILE_SHARE_READ|FILE_SHARE_WRITE,
		NULL,
		OPEN_EXISTING,
		0,
		NULL
		)) == INVALID_HANDLE_VALUE )
		{ err = "CreateFile failed"; return 0; }
	if ( ! DeviceIoControl(
		hdisk,
		IOCTL_DISK_GET_DRIVE_GEOMETRY,
		NULL,
		0,
		&dg,
		sizeof(dg),
		&dwResult,
		NULL
		) )
		{
		CloseHandle(hdisk);
		err = "DeviceIoControl IOCTL_DISK_GET_DRIVE_GEOMETRY failed";
		return 0;
		}
	if ( !validBytesPerSector(dg.BytesPerSector) )
		{
		CloseHandle(hdisk);
		err = "invalid BytesPerSector";
		return 0;
		}
	if ( ! DeviceIoControl(
		hdisk,
		FSCTL_LOCK_VOLUME,
		NULL,
		0,
		NULL,
		0,
		&dwResult,
		NULL
		) )
		{
		CloseHandle(hdisk);
		err = "DeviceIoControl FSCTL_LOCK_VOLUME failed";
		return 0;
		}
	BeMem *bemem = new BeMem(
		hdisk,
		addr,
		dg.BytesPerSector,
		read_only,
		lazy_write
		);
	if ( bemem == 0 )
		{
		DeviceIoControl(
			hdisk,
			FSCTL_UNLOCK_VOLUME,
			NULL,
			0,
			NULL,
			0,
			&dwResult,
			NULL
			);
		DeviceIoControl(
			hdisk,
			FSCTL_DISMOUNT_VOLUME,
			NULL,
			0,
			NULL,
			0,
			&dwResult,
			NULL
			);
		CloseHandle(hdisk);
		err = "out of memory";
		return 0;
		}
	return (void *) bemem;
	}
/*...e*/
/*...sbemem_refresh:0:*/
BEMEMEXPORT void BEMEMENTRY bemem_refresh(void * ptr)
	{
	BeMem *bemem = (BeMem *) ptr;
	bemem->refresh();
	}
/*...e*/
/*...sbemem_flush:0:*/
BEMEMEXPORT Boolean BEMEMENTRY bemem_flush(void * ptr)
	{
	BeMem *bemem = (BeMem *) ptr;
	return bemem->flush();
	}
/*...e*/
/*...sbemem_delete:0:*/
BEMEMEXPORT void BEMEMENTRY bemem_delete(void * ptr)
	{
	BeMem *bemem = (BeMem *) ptr;
	delete bemem;
	}
/*...e*/
