//
// bez80.C - BE disassembler extension for Z80
//

/*...sincludes:0:*/
#ifdef NO_CINCLUDES
  #include <stdio.h>
  #include <ctype.h>
  #include <string.h>
  #include <stddef.h>
  #include <stdlib.h>
  #include <stdarg.h>
#else
  #include <cstdio>
  #include <cctype>
  #include <cstring>
  #include <cstddef>
  #include <cstdlib>
  #include <cstdarg>
#endif

#include "bedisext.h"
/*...e*/

/*...stypes:0:*/
typedef   signed char   s8;
typedef unsigned char   u8;
typedef   signed short s16;
typedef unsigned short u16;
/*...e*/
/*...svars:0:*/
static Boolean (*d_read)(BEDISADDR32, unsigned char &);
static Boolean (*d_symbolic)(BEDISADDR32, char *, BEDISADDR32 &);
static Boolean use_syms = TRUE;
static Boolean show_opcode = TRUE;
static Boolean show_ill = TRUE;
static Boolean mtx_exts = FALSE;
/*...e*/
/*...sread8:0:*/
static Boolean read8(u16 addr, u8 & value)
	{
	return d_read(addr, value);
	}
/*...e*/
/*...sread16:0:*/
static Boolean read16(u16 addr, u16 & value)
	{
	unsigned char b0, b1;
	if ( !d_read(addr  ,b0) ||
	     !d_read(addr+1,b1) )
		return FALSE;
	value = (u16) ( b0 + ((u16) b1 << 8) );
	return TRUE;
	}
/*...e*/
/*...ssymbolic:0:*/
#define L_SYM (1+200+1+6+1)

static const char *symbolic_addr(u16 addr, char *buf)
	{
	if ( use_syms )
		{
		BEDISADDR32 disp;
		if ( d_symbolic(addr, buf, disp) && buf[0] != '$' )
			{
			if ( disp != 0 )
				sprintf(buf+strlen(buf), "+0x%04x", disp);
			return buf;
			}		
		}
	sprintf(buf, "0x%04x", addr);
	return buf;
	}

static const char *symbolic_augment(u16 addr, char *buf)
	{
	if ( use_syms )
		{
		BEDISADDR32 disp;
		if ( d_symbolic(addr, buf+1, disp) && buf[1] != '$' )
			{
			buf[0] = '{';
			if ( disp != 0 )
				sprintf(buf+strlen(buf), "+0x%04x", disp);
			strcat(buf, "}");
			return buf;
			}		
		}
	buf[0] = '\0';
	return buf;
	}
/*...e*/

/*...sclass dis:0:*/
// Abstract disassembler class.

class dis
	{
public:
	virtual Boolean disassemble(u16 & a, char *s) = 0;
	virtual Boolean ref_code(u16 a, u16 & r) = 0;
	virtual Boolean ref_data(u16 a, u16 & r) = 0;
	dis() {}
	virtual ~dis() {}
	};
/*...e*/
/*...sclass z80dis:0:*/
class z80dis : public dis
	{
	typedef dis super;
private:
/*...sprivate:8:*/
typedef Boolean (z80dis::*z80disfn)(u8 op, u16 & a, char *s);

z80disfn disfn_table[0x100];

void add_ins(
	int b7, int b6, int b5, int b4, int b3, int b2, int b1, int b0,
	z80disfn disfn
	);

void build_decode_tables();

Boolean rm_fetch_d(u16 & a);
Boolean rm_operand(u8 rm, u16 & a, char *buf);
const char *inx_operand();
const char *ss_operand(u8 ss) const;
const char *dd_operand(u8 dd);
const char *qq_operand(u8 qq);

Boolean d_ld_rm_rm        (u8 op, u16 & a, char *s);
Boolean d_ld_rm_n         (u8 op, u16 & a, char *s);
Boolean d_ld_bc_a         (u8 op, u16 & a, char *s);
Boolean d_ld_a_bc         (u8 op, u16 & a, char *s);
Boolean d_ld_de_a         (u8 op, u16 & a, char *s);
Boolean d_ld_a_de         (u8 op, u16 & a, char *s);
Boolean d_ld_nn_a         (u8 op, u16 & a, char *s);
Boolean d_ld_a_nn         (u8 op, u16 & a, char *s);
Boolean d_ld_dd_nn        (u8 op, u16 & a, char *s);
Boolean d_ld_inx_nn       (u8 op, u16 & a, char *s);
Boolean d_ld_nn_inx       (u8 op, u16 & a, char *s);
Boolean d_ld_sp_inx       (u8 op, u16 & a, char *s);
Boolean d_push_qq         (u8 op, u16 & a, char *s);
Boolean d_pop_qq          (u8 op, u16 & a, char *s);
Boolean d_ex_de_hl        (u8 op, u16 & a, char *s);
Boolean d_ex_af_af        (u8 op, u16 & a, char *s);
Boolean d_exx             (u8 op, u16 & a, char *s);
Boolean d_ex_sp_inx       (u8 op, u16 & a, char *s);
Boolean d_binop8_a_rm     (u8 op, u16 & a, char *s);
Boolean d_binop8_a_n      (u8 op, u16 & a, char *s);
Boolean d_inc8_rm         (u8 op, u16 & a, char *s);
Boolean d_dec8_rm         (u8 op, u16 & a, char *s);
Boolean d_daa             (u8 op, u16 & a, char *s);
Boolean d_cpl             (u8 op, u16 & a, char *s);
Boolean d_ccf             (u8 op, u16 & a, char *s);
Boolean d_scf             (u8 op, u16 & a, char *s);
Boolean d_nop             (u8 op, u16 & a, char *s);
Boolean d_halt            (u8 op, u16 & a, char *s);
Boolean d_di              (u8 op, u16 & a, char *s);
Boolean d_ei              (u8 op, u16 & a, char *s);
Boolean d_add_inx         (u8 op, u16 & a, char *s);
Boolean d_inc_inx         (u8 op, u16 & a, char *s);
Boolean d_dec_inx         (u8 op, u16 & a, char *s);
Boolean d_rlca            (u8 op, u16 & a, char *s);
Boolean d_rla             (u8 op, u16 & a, char *s);
Boolean d_rrca            (u8 op, u16 & a, char *s);
Boolean d_rra             (u8 op, u16 & a, char *s);
Boolean d_bitop           (u8 op, u16 & a, char *s);
Boolean d_jp_nn           (u8 op, u16 & a, char *s);
Boolean d_jp_cc_nn        (u8 op, u16 & a, char *s);
Boolean d_jr              (u8 op, u16 & a, char *s);
Boolean d_jr_c            (u8 op, u16 & a, char *s);
Boolean d_jr_nc           (u8 op, u16 & a, char *s);
Boolean d_jr_z            (u8 op, u16 & a, char *s);
Boolean d_jr_nz           (u8 op, u16 & a, char *s);
Boolean d_jp_inx          (u8 op, u16 & a, char *s);
Boolean d_djnz            (u8 op, u16 & a, char *s);
Boolean d_call_nn         (u8 op, u16 & a, char *s);
Boolean d_call_cc_nn      (u8 op, u16 & a, char *s);
Boolean d_ret             (u8 op, u16 & a, char *s);
Boolean d_ret_cc          (u8 op, u16 & a, char *s);
Boolean d_rst             (u8 op, u16 & a, char *s);
Boolean d_in_a_n          (u8 op, u16 & a, char *s);
Boolean d_out_n_a         (u8 op, u16 & a, char *s);
Boolean d_ed_prefix       (u8 op, u16 & a, char *s);
Boolean d_illegal         (u8 op, u16 & a, char *s);
/*...e*/
protected:
/*...sprotected:8:*/
#define	IX_PREFIX       0x01	// Will be removed if used
#define	IY_PREFIX       0x02	// Will be removed if used
#define	D_FETCHED       0x04	// Present when D fetch, removed when used
#define	ILL_PREFIX      0x10	// Prefix not allowed here
#define	ILL_SHROT       0x20	// Unknown shift/rotate type
#define	ILL_USE_HL_FORM 0x40	// Use a different opcode
#define	ILL_2_MEM_OP    0x80	// 2 memory operands (not allowed)
u8 state;			// Internal state
u8 disp;			// Valid if D_FETCHED

virtual Boolean disassemble_non_prefix(u8 op, u16 & a, char *s);
/*...e*/
public:
	virtual Boolean disassemble(u16 & a, char *s);
	virtual Boolean ref_code(u16 a, u16 & r);
	virtual Boolean ref_data(u16 a, u16 & r);
	z80dis();
	virtual ~z80dis() {};
	};

/*...srm_fetch_d:0:*/
// Fetch d, if the prefix has been given, and we've not yet fetched it

Boolean z80dis::rm_fetch_d(u16 & a)
	{
	if ( (state & (IX_PREFIX|IY_PREFIX)) != 0 &&
	     (state & D_FETCHED)             == 0 )
		{
		if ( !read8(a++, disp) )
			return FALSE;
		state |= D_FETCHED;
		}
	return TRUE;
	}
/*...e*/
/*...srm_operand:0:*/
Boolean z80dis::rm_operand(u8 rm, u16 & a, char *buf)
	{
	if ( rm != 6 )
		{
		buf[0] = "bcdehlMa"[rm];
		buf[1] = '\0';
		}
	else
		{
		if ( !rm_fetch_d(a) )
			return FALSE;
		if ( state & IX_PREFIX )
			{
			state &= ~(IX_PREFIX|D_FETCHED);
			if ( disp == 0 )
				strcpy(buf, "(ix)");
			else if ( (s8) disp > 0 )
				sprintf(buf, "(ix+0x%02x)", (unsigned) disp);
			else
				sprintf(buf, "(ix-0x%02x)", -((int)(s8)disp));
			}
		else if ( state & IY_PREFIX )
			{
			state &= ~(IY_PREFIX|D_FETCHED);
			if ( disp == 0 )
				strcpy(buf, "(iy)");
			else if ( (s8) disp > 0 )
				sprintf(buf, "(iy+0x%02x)", (unsigned) disp);
			else
				sprintf(buf, "(iy-0x%02x)", -((int)(s8)disp));
			}
		else
			strcpy(buf, "(hl)");
		}
	return TRUE;
	}
/*...e*/
/*...sinx_operand:0:*/
const char *z80dis::inx_operand()
	{
	if ( state & IX_PREFIX )
		{
		state &= ~IX_PREFIX;
		return "ix";
		}
	else if ( state & IY_PREFIX )
		{
		state &= ~IY_PREFIX;
		return "iy";
		}
	else
		return "hl";
	}
/*...e*/
/*...sss_operand:0:*/
const char *z80dis::ss_operand(u8 ss) const
	{
	switch ( ss )
		{
		case 0: return "bc";
		case 1: return "de";
		case 2: return "hl";
		case 3: return "sp";
		}
	return 0; // Won't get here
	}
/*...e*/
/*...sdd_operand:0:*/
const char *z80dis::dd_operand(u8 dd)
	{
	switch ( dd )
		{
		case 0: return "bc";
		case 1: return "de";
		case 2: return inx_operand();
		case 3: return "sp";
		}
	return 0; // Won't get here
	}
/*...e*/
/*...sqq_operand:0:*/
const char *z80dis::qq_operand(u8 qq)
	{
	switch ( qq )
		{
		case 0: return "bc";
		case 1: return "de";
		case 2: return inx_operand();
		case 3: return "af";
		}
	return 0; // Won't get here
	}
/*...e*/

/*...sld group:0:*/
/*...sd_ld_rm_rm:0:*/
Boolean z80dis::d_ld_rm_rm(u8 op, u16 & a, char *s)
	{
	char buf1[10+1];
	u8 rm1 = (u8) ((op>>3)&7);
	if ( !rm_operand(rm1, a, buf1) )
		return FALSE;
	char buf2[10+1];
	u8 rm2 = (u8) ( op    &7);
	if ( !rm_operand(rm2, a, buf2) )
		return FALSE;
	sprintf(s, "ld      %s,%s", buf1, buf2);
	if ( rm1 == 6 && rm2 == 6 )
		state |= ILL_2_MEM_OP;
	return TRUE;
	}
/*...e*/
/*...sd_ld_rm_n:0:*/
Boolean z80dis::d_ld_rm_n(u8 op, u16 & a, char *s)
	{
	u8 rm = (u8) ((op>>3)&7);
	char buf[10+1];
	if ( !rm_operand(rm, a, buf) )
		return FALSE;
	u8 n;
	if ( !read8(a++, n) )
		return FALSE;
	sprintf(s, "ld      %s,0x%02x",
		buf,
		(unsigned) n);
	if ( n >= ' ' && n <= '~' )
		sprintf(s+strlen(s), " ; ='%c'", (unsigned) n);
	return TRUE;
	}
/*...e*/
/*...sd_ld_bc_a:0:*/
Boolean z80dis::d_ld_bc_a(u8 op, u16 & a, char *s)
	{
	op=op; a=a; // Suppress warnings
	strcpy(s, "ld      (bc),a");
	return TRUE;
	}
/*...e*/
/*...sd_ld_a_bc:0:*/
Boolean z80dis::d_ld_a_bc(u8 op, u16 & a, char *s)
	{
	op=op; a=a; // Suppress warnings
	strcpy(s, "ld      a,(bc)");
	return TRUE;
	}
/*...e*/
/*...sd_ld_de_a:0:*/
Boolean z80dis::d_ld_de_a(u8 op, u16 & a, char *s)
	{
	op=op; a=a; // Suppress warnings
	strcpy(s, "ld      (de),a");
	return TRUE;
	}
/*...e*/
/*...sd_ld_a_de:0:*/
Boolean z80dis::d_ld_a_de(u8 op, u16 & a, char *s)
	{
	op=op; a=a; // Suppress warnings
	strcpy(s, "ld      a,(de)");
	return TRUE;
	}
/*...e*/
/*...sd_ld_nn_a:0:*/
Boolean z80dis::d_ld_nn_a(u8 op, u16 & a, char *s)
	{
	op=op; // Suppress warning
	u16 nn;
	if ( !read16(a, nn) )
		return FALSE;
	a += 2;
	char buf[L_SYM+1];
	sprintf(s, "ld      (%s),a", symbolic_addr(nn, buf));
	return TRUE;
	}
/*...e*/
/*...sd_ld_a_nn:0:*/
Boolean z80dis::d_ld_a_nn(u8 op, u16 & a, char *s)
	{
	op=op; // Suppress warning
	u16 nn;
	if ( !read16(a, nn) )
		return FALSE;
	a += 2;
	char buf[L_SYM+1];
	sprintf(s, "ld      a,(%s)",
		symbolic_addr(nn, buf));
	u8 n;
	if ( read8(nn, n) )
		{
		sprintf(s+strlen(s), " ; =0x%02x", (unsigned) n);
		if ( n >= ' ' && n <= '~' )
			sprintf(s+strlen(s), " ='%c'", (unsigned) n);
		}
	return TRUE;
	}
/*...e*/
/*...sd_ld_dd_nn:0:*/
Boolean z80dis::d_ld_dd_nn(u8 op, u16 & a, char *s)
	{
	u16 nn;
	if ( !read16(a, nn) )
		return FALSE;
	a += 2;
	u8 dd = (u8) ((op>>4)&3);
	char buf[L_SYM+1];
	sprintf(s, "ld      %s,0x%04x%s",
		dd_operand(dd),
		(unsigned) nn,
		symbolic_augment(nn, buf));
	return TRUE;
	}
/*...e*/
/*...sd_ld_inx_nn:0:*/
Boolean z80dis::d_ld_inx_nn(u8 op, u16 & a, char *s)
	{
	op=op; // Suppress warning
	u16 nn;
	if ( !read16(a, nn) )
		return FALSE;
	a += 2;
	char buf[L_SYM+1];
	sprintf(s, "ld      %s,(%s)",
		inx_operand(),
		symbolic_addr(nn, buf));
	u16 mm;
	if ( read16(nn, mm) )
		sprintf(s+strlen(s), " ; =0x%04x%s",
			(unsigned) mm,
			symbolic_augment(mm, buf));
	return TRUE;
	}
/*...e*/
/*...sd_ld_nn_inx:0:*/
Boolean z80dis::d_ld_nn_inx(u8 op, u16 & a, char *s)
	{
	op=op; // Suppress warning
	u16 nn;
	if ( !read16(a, nn) )
		return FALSE;
	a += 2;
	char buf[L_SYM+1];
	sprintf(s, "ld      (%s),%s",
		symbolic_addr(nn, buf),
		inx_operand());
	return TRUE;
	}
/*...e*/
/*...sd_ld_sp_inx:0:*/
Boolean z80dis::d_ld_sp_inx(u8 op, u16 & a, char *s)
	{
	op=op; a=a; // Suppress warnings
	sprintf(s, "ld      sp,%s", inx_operand());
	return TRUE;
	}
/*...e*/
/*...e*/
/*...spush and pop group:0:*/
/*...sd_push_qq:0:*/
Boolean z80dis::d_push_qq(u8 op, u16 & a, char *s)
	{
	a=a; // Suppress warning
	u8 qq = (u8) ((op>>4)&3);
	sprintf(s, "push    %s", qq_operand(qq));
	return TRUE;
	}
/*...e*/
/*...sd_pop_qq:0:*/
Boolean z80dis::d_pop_qq(u8 op, u16 & a, char *s)
	{
	a=a; // Suppress warning
	u8 qq = (u8) ((op>>4)&3);
	sprintf(s, "pop     %s", qq_operand(qq));
	return TRUE;
	}
/*...e*/
/*...e*/
/*...sexchange group:0:*/
/*...sd_ex_de_hl:0:*/
Boolean z80dis::d_ex_de_hl(u8 op, u16 & a, char *s)
	{
	op=op; a=a; // Suppress warning
	strcpy(s, "ex      de,hl");
	return TRUE;
	}
/*...e*/
/*...sd_ex_af_af:0:*/
Boolean z80dis::d_ex_af_af(u8 op, u16 & a, char *s)
	{
	op=op; a=a; // Suppress warning
	strcpy(s, "ex      af,af'");
	return TRUE;
	}
/*...e*/
/*...sd_exx:0:*/
Boolean z80dis::d_exx(u8 op, u16 & a, char *s)
	{
	op=op; a=a; // Suppress warning
	strcpy(s, "exx");
	return TRUE;
	}
/*...e*/
/*...sd_ex_sp_inx:0:*/
Boolean z80dis::d_ex_sp_inx(u8 op, u16 & a, char *s)
	{
	op=op; a=a; // Suppress warning
	sprintf(s, "ex      (sp),%s", inx_operand());
	return TRUE;
	}
/*...e*/
/*...e*/
/*...sbinop8 group:0:*/
static const char *binop8_name[] =
	{ "add     a,", "adc     a,", "sub     ", "sbc     a,",
	  "and     "  , "xor     "  , "or      ", "cp      "   };

/*...sd_binop8_a_rm:0:*/
Boolean z80dis::d_binop8_a_rm(u8 op, u16 & a, char *s)
	{
	u8 rm = (u8) (op&7);
	char buf[10+1];
	if ( !rm_operand(rm, a, buf) )
		return FALSE;
	sprintf(s, "%s%s",
		binop8_name[(op>>3)&7],
		buf);
	return TRUE;
	}
/*...e*/
/*...sd_binop8_a_n:0:*/
Boolean z80dis::d_binop8_a_n(u8 op, u16 & a, char *s)
	{
	u8 n;
	if ( !read8(a++, n) )
		return FALSE;
	sprintf(s, "%s0x%02x",
		binop8_name[(op>>3)&7],
		(unsigned) n);
	if ( n >= ' ' && n <= '~' )
		sprintf(s+strlen(s), " ; ='%c'", (unsigned) n);
	return TRUE;
	}
/*...e*/
/*...e*/
/*...sinc8 and dec8 group:0:*/
/*...sd_inc8_rm:0:*/
Boolean z80dis::d_inc8_rm(u8 op, u16 & a, char *s)
	{
	u8 rm = (u8) ((op>>3)&7);
	char buf[10+1];
	if ( !rm_operand(rm, a, buf) )
		return FALSE;
	sprintf(s, "inc     %s", buf);
	return TRUE;
	}
/*...e*/
/*...sd_dec8_rm:0:*/
Boolean z80dis::d_dec8_rm(u8 op, u16 & a, char *s)
	{
	u8 rm = (u8) ((op>>3)&7);
	char buf[10+1];
	if ( !rm_operand(rm, a, buf) )
		return FALSE;
	sprintf(s, "dec     %s", buf);
	return TRUE;
	}
/*...e*/
/*...e*/
/*...sinx arith group:0:*/
/*...sd_add_inx:0:*/
// Note: ix or iy prefix, if present, applies to lvalue and rvalue.
// So we avoid inx_operand removing it, and dd_operand not seeing it.

Boolean z80dis::d_add_inx(u8 op, u16 & a, char *s)
	{
	op=op; a=a; // Suppress warning
	u8 prefix = (u8) ( state & (IX_PREFIX|IY_PREFIX) );
	sprintf(s, "add     %s,", inx_operand());
	state |= prefix;
	u8 dd = (u8) ((op>>4)&3);
	strcat(s, dd_operand(dd));
	return TRUE;
	}
/*...e*/
/*...sd_inc_inx:0:*/
Boolean z80dis::d_inc_inx(u8 op, u16 & a, char *s)
	{
	a=a; // Suppress warning
	u8 dd = (u8) ((op>>4)&3);
	sprintf(s, "inc     %s", dd_operand(dd));
	return TRUE;
	}
/*...e*/
/*...sd_dec_inx:0:*/
Boolean z80dis::d_dec_inx(u8 op, u16 & a, char *s)
	{
	a=a; // Suppress warning
	u8 dd = (u8) ((op>>4)&3);
	sprintf(s, "dec     %s", dd_operand(dd));
	return TRUE;
	}
/*...e*/
/*...e*/
/*...sbit group:0:*/
/*...sd_rlca:0:*/
Boolean z80dis::d_rlca(u8 op, u16 & a, char *s)
	{
	op=op; a=a; // Suppress warning
	strcpy(s, "rlca");
	return TRUE;
	}
/*...e*/
/*...sd_rla:0:*/
Boolean z80dis::d_rla(u8 op, u16 & a, char *s)
	{
	op=op; a=a; // Suppress warning
	strcpy(s, "rla");
	return TRUE;
	}
/*...e*/
/*...sd_rrca:0:*/
Boolean z80dis::d_rrca(u8 op, u16 & a, char *s)
	{
	op=op; a=a; // Suppress warning
	strcpy(s, "rrca");
	return TRUE;
	}
/*...e*/
/*...sd_rra:0:*/
Boolean z80dis::d_rra(u8 op, u16 & a, char *s)
	{
	op=op; a=a; // Suppress warning
	strcpy(s, "rra");
	return TRUE;
	}
/*...e*/
/*...sd_bitop:0:*/
static const char *shrot_name[] =
	{ "rlc", "rrc", "rl ", "rr ", "sla", "sra", "sll", "srl" };

static const char *bitop_name[] =
	{ "???", "bit", "res", "set" };

Boolean z80dis::d_bitop(u8 op, u16 & a, char *s)
	{
	op=op; // Suppress warning
	if ( !rm_fetch_d(a) )
		return FALSE;
	u8 op2;
	if ( !read8(a++, op2) )
		return FALSE;
	u8 bitop = (u8) ((op2>>6)&3);
	u8 shrot = (u8) ((op2>>3)&7);
	u8 rm    = (u8) ( op2    &7);
	char buf[10+1];
	rm_operand(rm, a, buf);
	if ( bitop == 0 )
		{
		sprintf(s, "%s     %s",
			shrot_name[shrot],
			buf);
		if ( shrot == 6 )
			state |= ILL_SHROT;
		}
	else
		sprintf(s, "%s     %u,%s",
			bitop_name[bitop],
			(unsigned) shrot,		// bitshift actually
			buf);
	return TRUE;
	}
/*...e*/
/*...e*/
/*...scontrol flow group:0:*/
static const char *cc_name[] =
	{ "nz", "z", "nc", "c", "po", "pe", "p", "m" };

/*...sd_jp_nn:0:*/
Boolean z80dis::d_jp_nn(u8 op, u16 & a, char *s)
	{
	op=op; // Suppress warning
	u16 nn;
	if ( !read16(a, nn) )
		return FALSE;
	a += 2;
	char buf[L_SYM+1];
	sprintf(s, "jp      %s",
		symbolic_addr(nn, buf));
	return TRUE;
	}
/*...e*/
/*...sd_jp_cc_nn:0:*/
Boolean z80dis::d_jp_cc_nn(u8 op, u16 & a, char *s)
	{
	u16 nn;
	if ( !read16(a, nn) )
		return FALSE;
	a += 2;
	char buf[L_SYM+1];
	sprintf(s, "jp      %s,%s",
		cc_name[(op>>3)&7],
		symbolic_addr(nn, buf));
	return TRUE;
	}
/*...e*/
/*...sd_jr:0:*/
Boolean z80dis::d_jr(u8 op, u16 & a, char *s)
	{
	op=op; // Suppress warning
	u8 e;
	if ( !read8(a++, e) )
		return FALSE;
	char buf[L_SYM+1];
	sprintf(s, "jr      %s",
		symbolic_addr((u16) (a+(s16)(s8)e), buf));
	return TRUE;
	}
/*...e*/
/*...sd_jr_c:0:*/
Boolean z80dis::d_jr_c(u8 op, u16 & a, char *s)
	{
	op=op; // Suppress warning
	u8 e;
	if ( !read8(a++, e) )
		return FALSE;
	char buf[L_SYM+1];
	sprintf(s, "jr      c,%s",
		symbolic_addr((u16) (a+(s16)(s8)e), buf));
	return TRUE;
	}
/*...e*/
/*...sd_jr_nc:0:*/
Boolean z80dis::d_jr_nc(u8 op, u16 & a, char *s)
	{
	op=op; // Suppress warning
	u8 e;
	if ( !read8(a++, e) )
		return FALSE;
	char buf[L_SYM+1];
	sprintf(s, "jr      nc,%s",
		symbolic_addr((u16) (a+(s16)(s8)e), buf));
	return TRUE;
	}
/*...e*/
/*...sd_jr_z:0:*/
Boolean z80dis::d_jr_z(u8 op, u16 & a, char *s)
	{
	op=op; // Suppress warning
	u8 e;
	if ( !read8(a++, e) )
		return FALSE;
	char buf[L_SYM+1];
	sprintf(s, "jr      z,%s",
		symbolic_addr((u16) (a+(s16)(s8)e), buf));
	return TRUE;
	}
/*...e*/
/*...sd_jr_nz:0:*/
Boolean z80dis::d_jr_nz(u8 op, u16 & a, char *s)
	{
	op=op; // Suppress warning
	u8 e;
	if ( !read8(a++, e) )
		return FALSE;
	char buf[L_SYM+1];
	sprintf(s, "jr      nz,%s",
		symbolic_addr((u16) (a+(s16)(s8)e), buf));
	return TRUE;
	}
/*...e*/
/*...sd_jp_inx:0:*/
Boolean z80dis::d_jp_inx(u8 op, u16 & a, char *s)
	{
	op=op; a=a; // Suppress warnings
	sprintf(s, "jp      (%s)",
		inx_operand());
	return TRUE;
	}
/*...e*/
/*...sd_djnz:0:*/
Boolean z80dis::d_djnz(u8 op, u16 & a, char *s)
	{
	op=op; // Suppress warning
	u8 e;
	if ( !read8(a++, e) )
		return FALSE;
	char buf[L_SYM+1];
	sprintf(s, "djnz    %s",
		symbolic_addr((u16) (a+(s16)(s8)e), buf));
	return TRUE;
	}
/*...e*/
/*...sd_call_nn:0:*/
Boolean z80dis::d_call_nn(u8 op, u16 & a, char *s)
	{
	op=op; // Suppress warning
	u16 nn;
	if ( !read16(a, nn) )
		return FALSE;
	a += 2;
	char buf[L_SYM+1];
	sprintf(s, "call    %s",
		symbolic_addr(nn, buf));
	return TRUE;
	}
/*...e*/
/*...sd_call_cc_nn:0:*/
Boolean z80dis::d_call_cc_nn(u8 op, u16 & a, char *s)
	{
	u16 nn;
	if ( !read16(a, nn) )
		return FALSE;
	a += 2;
	char buf[L_SYM+1];
	sprintf(s, "call    %s,%s",
		cc_name[(op>>3)&7],
		symbolic_addr(nn, buf));
	return TRUE;
	}
/*...e*/
/*...sd_ret:0:*/
Boolean z80dis::d_ret(u8 op, u16 & a, char *s)
	{
	op=op; a=a; // Suppress warnings
	strcpy(s, "ret");
	return TRUE;
	}
/*...e*/
/*...sd_ret_cc:0:*/
Boolean z80dis::d_ret_cc(u8 op, u16 & a, char *s)
	{
	a=a; // Suppress warnings
	sprintf(s, "ret     %s",
		cc_name[(op>>3)&7]);
	return TRUE;
	}
/*...e*/
/*...sd_rst:0:*/
Boolean z80dis::d_rst(u8 op, u16 & a, char *s)
	{
	a=a; // Suppress warning
	sprintf(s, "rst     0x%02x",
		(unsigned) (op&0x38));
	if ( mtx_exts )
		switch ( op&0x38 )
			{
/*...s0x08 \45\ dehl:24:*/
case 0x08:
	strcat(s, "{dehl}");
	break;
/*...e*/
/*...s0x10 \45\ scn:24:*/
case 0x10:
	strcat(s, "{scn}");
	for ( ;; )
		{
		u8 code;
		if ( !read8(a, code) )
			{
			strcat(s, " ... can't read");
			return TRUE;
			}
		a++;
		sprintf(s+strlen(s), ",0x%02x", code);
		switch ( code & 0xc0 )
			{
			case 0x00:
				strcat(s, "{write}");
				break;
			case 0x40:
				// Virtual screen
				sprintf(s+strlen(s), "{vs%d%s}", code&7, (code&8)?",cls":"");
				break;
			case 0x80:
				// Write string
				{
				strcat(s, "{writestr}");
				u8 i;
				for ( i = 0; i < (code&0x1f); i++ )
					{
					u8 b;
					if ( !read8(a, b) )
						{
						strcat(s, " ... can't read");
						return TRUE;
						}
					a++;
					if ( b < ' ' || b > '~' )
						sprintf(s+strlen(s), ",0x%02x", b);	
					else if ( s[strlen(s)-1] == '\'' )
						sprintf(s+strlen(s)-1, "%c'", b); 
					else
						sprintf(s+strlen(s), ",'%c'", b); 
					}
				}
				break;
			case 0xc0:
				// Write BC
				strcat(s, "{writebc}");
				break;
			}
		if ( (code & 0x20) == 0 )
			break;
		if ( strlen(s) > 300 )
			{
			strcat(s, " ... too long");
			return TRUE;
			}
		}
	break;
/*...e*/
/*...s0x28 \45\ err:24:*/
case 0x28:
	{
	u8 errcode;
	if ( !read8(a, errcode) )
		{
		strcat(s, " ... can't read");
		return TRUE;
		}
	a++;
	sprintf(s+strlen(s), "{err},0x%02x", errcode);
	}
	break;
/*...e*/
			}
	return TRUE;
	}
/*...e*/
/*...e*/
/*...sio group:0:*/
/*...sd_in_a_n:0:*/
Boolean z80dis::d_in_a_n(u8 op, u16 & a, char *s)
	{
	op=op; // Suppress warning
	u8 n;
	if ( !read8(a++, n) )
		return FALSE;
	sprintf(s, "in      a,(0x%02x)", (unsigned) n);
	return TRUE;
	}
/*...e*/
/*...sd_out_n_a:0:*/
Boolean z80dis::d_out_n_a(u8 op, u16 & a, char *s)
	{
	op=op; // Suppress warning
	u8 n;
	if ( !read8(a++, n) )
		return FALSE;
	sprintf(s, "out     (0x%02x),a", (unsigned) n);
	return TRUE;
	}
/*...e*/
/*...e*/
/*...smisc group:0:*/
/*...sd_daa:0:*/
Boolean z80dis::d_daa(u8 op, u16 & a, char *s)
	{
	op=op; a=a; // Suppress warnings
	strcpy(s, "daa");
	return TRUE;
	}
/*...e*/
/*...sd_cpl:0:*/
Boolean z80dis::d_cpl(u8 op, u16 & a, char *s)
	{
	op=op; a=a; // Suppress warnings
	strcpy(s, "cpl");
	return TRUE;
	}
/*...e*/
/*...sd_ccf:0:*/
Boolean z80dis::d_ccf(u8 op, u16 & a, char *s)
	{
	op=op; a=a; // Suppress warnings
	strcpy(s, "ccf");
	return TRUE;
	}
/*...e*/
/*...sd_scf:0:*/
Boolean z80dis::d_scf(u8 op, u16 & a, char *s)
	{
	op=op; a=a; // Suppress warnings
	strcpy(s, "scf");
	return TRUE;
	}
/*...e*/
/*...sd_nop:0:*/
Boolean z80dis::d_nop(u8 op, u16 & a, char *s)
	{
	op=op; a=a; // Suppress warnings
	strcpy(s, "nop");
	return TRUE;
	}
/*...e*/
/*...sd_halt:0:*/
Boolean z80dis::d_halt(u8 op, u16 & a, char *s)
	{
	op=op; a=a; // Suppress warnings
	strcpy(s, "halt");
	return TRUE;
	}
/*...e*/
/*...sd_di:0:*/
Boolean z80dis::d_di(u8 op, u16 & a, char *s)
	{
	op=op; a=a; // Suppress warnings
	strcpy(s, "di");
	return TRUE;
	}
/*...e*/
/*...sd_ei:0:*/
Boolean z80dis::d_ei(u8 op, u16 & a, char *s)
	{
	op=op; a=a; // Suppress warnings
	strcpy(s, "ei");
	return TRUE;
	}
/*...e*/
/*...sd_ed_prefix:0:*/
// opcode 0xed 0x??

Boolean z80dis::d_ed_prefix(u8 op, u16 & a, char *s)
	{
	u8 op2;
	if ( !read8(a++, op2) )
		return FALSE;
	switch ( op2 )
		{
		case 0x44: strcpy(s, "neg"        ); return TRUE;
		case 0x45: strcpy(s, "retn"       ); return TRUE;
		case 0x46: strcpy(s, "im 0"       ); return TRUE;
		case 0x47: strcpy(s, "ld      i,a"); return TRUE;
		case 0x4d: strcpy(s, "reti"       ); return TRUE;
		case 0x4f: strcpy(s, "ld      r,a"); return TRUE;
		case 0x56: strcpy(s, "im 1"       ); return TRUE;
		case 0x57: strcpy(s, "ld      a,i"); return TRUE;
		case 0x5e: strcpy(s, "im 2"       ); return TRUE;
		case 0x5f: strcpy(s, "ld      a,r"); return TRUE;
		case 0x67: strcpy(s, "rrd"        ); return TRUE;
		case 0x6f: strcpy(s, "rld"        ); return TRUE;
		case 0xa0: strcpy(s, "ldi"        ); return TRUE;
		case 0xa1: strcpy(s, "cpi"        ); return TRUE;
		case 0xa2: strcpy(s, "ini"        ); return TRUE;
		case 0xa3: strcpy(s, "outi"       ); return TRUE;
		case 0xa8: strcpy(s, "ldd"        ); return TRUE;
		case 0xa9: strcpy(s, "cpd"        ); return TRUE;
		case 0xaa: strcpy(s, "ind"        ); return TRUE;
		case 0xab: strcpy(s, "outd"       ); return TRUE;
		case 0xb0: strcpy(s, "ldir"       ); return TRUE;
		case 0xb1: strcpy(s, "cpir"       ); return TRUE;
		case 0xb2: strcpy(s, "inir"       ); return TRUE;
		case 0xb3: strcpy(s, "otir"       ); return TRUE;
		case 0xb8: strcpy(s, "lddr"       ); return TRUE;
		case 0xb9: strcpy(s, "cpdr"       ); return TRUE;
		case 0xba: strcpy(s, "indr"       ); return TRUE;
		case 0xbb: strcpy(s, "otdr"       ); return TRUE;
		}
	switch ( op2 & 0xcf )
		{
/*...s0x4b \45\ ld dd\44\\40\nn\41\:16:*/
case 0x4b:
	{
	u16 nn;
	if ( !read16(a, nn) )
		return FALSE;
	a += 2;
	u8 dd = (u8) ((op2>>4)&3);
	char buf[L_SYM+1];
	sprintf(s, "ld      %s,(%s)",
		dd_operand(dd),
		symbolic_addr(nn, buf));
	if ( dd == 2 )
		state |= ILL_USE_HL_FORM;
	u16 mm;
	if ( read16(nn, mm) )
		sprintf(s+strlen(s), " ; =0x%04x%s",
			(unsigned) mm,
			symbolic_augment(mm, buf));
	return TRUE;
	}
/*...e*/
/*...s0x43 \45\ ld \40\nn\41\\44\dd:16:*/
case 0x43:
	{
	u16 nn;
	if ( !read16(a, nn) )
		return FALSE;
	a += 2;
	u8 dd = (u8) ((op2>>4)&3);
	char buf[L_SYM+1];
	sprintf(s, "ld      (%s),%s",
		symbolic_addr(nn, buf),
		dd_operand(dd));
	if ( dd == 2 )
		state |= ILL_USE_HL_FORM;
	return TRUE;
	}
/*...e*/
/*...s0x4a \45\ adc hl\44\ss:16:*/
case 0x4a:
	{
	u8 ss = (u8) ((op2>>4)&3);
	sprintf(s, "adc     hl,%s",
		ss_operand(ss));
	return TRUE;
	}
/*...e*/
/*...s0x42 \45\ sbc hl\44\ss:16:*/
case 0x42:
	{
	u8 ss = (u8) ((op2>>4)&3);
	sprintf(s, "sbc     hl,%s",
		ss_operand(ss));
	return TRUE;
	}
/*...e*/
		}
	switch ( op2 & 0xc7 )
		{
/*...s0x40 \45\ in r\44\\40\c\41\:16:*/
case 0x40:
	{
	if ( state & (IX_PREFIX|IY_PREFIX) )
		state |= ILL_PREFIX;
	u8 rm = (u8) ((op2>>3)&7);
	char buf[10+1];
	if ( !rm_operand(rm, a, buf) )
		return FALSE;
	sprintf(s, "in      %s,(c)", buf);
	return TRUE;
	}
/*...e*/
/*...s0x41 \45\ out \40\c\41\\44\r:16:*/
case 0x41:
	{
	if ( state & (IX_PREFIX|IY_PREFIX) )
		state |= ILL_PREFIX;
	u8 rm = (u8) ((op2>>3)&7);
	char buf[10+1];
	if ( !rm_operand(rm, a, buf) )
		return FALSE;
	sprintf(s, "out     (c),%s", buf);
	return TRUE;
	}
/*...e*/
		}
	sprintf(s, "db      0x%02x,0x%02x", (unsigned) op, (unsigned) op2);
	return TRUE;
	}
/*...e*/
/*...sd_illegal:0:*/
Boolean z80dis::d_illegal(u8 op, u16 & a, char *s)
	{
	a=a; // Suppress warning
	sprintf(s, "db      0x%02x", (unsigned) op);
	if ( op >= ' ' && op <= '~' )
		sprintf(s+strlen(s), " ; ='%c'", (unsigned) op);
	return TRUE;
	}
/*...e*/
/*...e*/

/*...sdecode tables:0:*/
#define	X	2		// Can be 0 or a 1
#define	REG_M	X,X,X		// One of 7 registers or memory
#define	DD_	X,X		// Choice of 4 register pairs
#define	QQ_	X,X		// Choice of 4 register pairs
#define	SS_	X,X		// Choice of 4 register pairs, no prefix
#define	BIN8_	X,X,X		// Choice of 8 2-operand 8 bit arithmetic ops
#define	CC___	X,X,X		// Condition code
#define	T____	X,X,X		// Restart number

/*...sadd_ins:0:*/
void z80dis::add_ins(
	int b7, int b6, int b5, int b4, int b3, int b2, int b1, int b0,
	z80disfn disfn
	)
	{
	for ( int i = 0; i < 0x100; i++ )
		if ( ( b7 == X || b7 == ((i >> 7) & 1) ) &&
		     ( b6 == X || b6 == ((i >> 6) & 1) ) &&
		     ( b5 == X || b5 == ((i >> 5) & 1) ) &&
		     ( b4 == X || b4 == ((i >> 4) & 1) ) &&
		     ( b3 == X || b3 == ((i >> 3) & 1) ) &&
		     ( b2 == X || b2 == ((i >> 2) & 1) ) &&
		     ( b1 == X || b1 == ((i >> 1) & 1) ) &&
		     ( b0 == X || b0 == ( i       & 1) ) )
			disfn_table[i] = disfn;
	}
/*...e*/

void z80dis::build_decode_tables()
	{
	for ( int i = 0; i < 0x100; i++ )
		disfn_table[i] = &z80dis::d_illegal;

/*...sld group:8:*/
add_ins(0,1,REG_M,REG_M, &z80dis::d_ld_rm_rm        );
add_ins(0,0,REG_M,1,1,0, &z80dis::d_ld_rm_n         );
add_ins(0,0,0,0,0,0,1,0, &z80dis::d_ld_bc_a         );
add_ins(0,0,0,0,1,0,1,0, &z80dis::d_ld_a_bc         );
add_ins(0,0,0,1,0,0,1,0, &z80dis::d_ld_de_a         );
add_ins(0,0,0,1,1,0,1,0, &z80dis::d_ld_a_de         );
add_ins(0,0,1,1,0,0,1,0, &z80dis::d_ld_nn_a         );
add_ins(0,0,1,1,1,0,1,0, &z80dis::d_ld_a_nn         );
add_ins(0,0,DD_,0,0,0,1, &z80dis::d_ld_dd_nn        );
add_ins(0,0,1,0,1,0,1,0, &z80dis::d_ld_inx_nn       );
add_ins(0,0,1,0,0,0,1,0, &z80dis::d_ld_nn_inx       );
add_ins(1,1,1,1,1,0,0,1, &z80dis::d_ld_sp_inx       );
/*...e*/
/*...spush and pop group:8:*/
add_ins(1,1,QQ_,0,1,0,1, &z80dis::d_push_qq         );
add_ins(1,1,QQ_,0,0,0,1, &z80dis::d_pop_qq          );
/*...e*/
/*...sexchange group:8:*/
add_ins(1,1,1,0,1,0,1,1, &z80dis::d_ex_de_hl        );
add_ins(0,0,0,0,1,0,0,0, &z80dis::d_ex_af_af        );
add_ins(1,1,0,1,1,0,0,1, &z80dis::d_exx             );
add_ins(1,1,1,0,0,0,1,1, &z80dis::d_ex_sp_inx       );
/*...e*/
/*...sbinop8 group:8:*/
add_ins(1,0,BIN8_,REG_M, &z80dis::d_binop8_a_rm     );
add_ins(1,1,BIN8_,1,1,0, &z80dis::d_binop8_a_n      );
/*...e*/
/*...sinc8 and dec8 group:8:*/
add_ins(0,0,REG_M,1,0,0, &z80dis::d_inc8_rm         );
add_ins(0,0,REG_M,1,0,1, &z80dis::d_dec8_rm         );
/*...e*/
/*...sinx arith group:8:*/
add_ins(0,0,SS_,1,0,0,1, &z80dis::d_add_inx         );
add_ins(0,0,SS_,0,0,1,1, &z80dis::d_inc_inx         );
add_ins(0,0,SS_,1,0,1,1, &z80dis::d_dec_inx         );
/*...e*/
/*...sbit group:8:*/
add_ins(0,0,0,0,0,1,1,1, &z80dis::d_rlca            );
add_ins(0,0,0,1,0,1,1,1, &z80dis::d_rla             );
add_ins(0,0,0,0,1,1,1,1, &z80dis::d_rrca            );
add_ins(0,0,0,1,1,1,1,1, &z80dis::d_rra             );
add_ins(1,1,0,0,1,0,1,1, &z80dis::d_bitop           );
/*...e*/
/*...scontrol flow group:8:*/
add_ins(1,1,0,0,0,0,1,1, &z80dis::d_jp_nn           );
add_ins(1,1,CC___,0,1,0, &z80dis::d_jp_cc_nn        );
add_ins(0,0,0,1,1,0,0,0, &z80dis::d_jr              );
add_ins(0,0,1,1,1,0,0,0, &z80dis::d_jr_c            );
add_ins(0,0,1,1,0,0,0,0, &z80dis::d_jr_nc           );
add_ins(0,0,1,0,1,0,0,0, &z80dis::d_jr_z            );
add_ins(0,0,1,0,0,0,0,0, &z80dis::d_jr_nz           );
add_ins(1,1,1,0,1,0,0,1, &z80dis::d_jp_inx          );
add_ins(0,0,0,1,0,0,0,0, &z80dis::d_djnz            );
add_ins(1,1,0,0,1,1,0,1, &z80dis::d_call_nn         );
add_ins(1,1,CC___,1,0,0, &z80dis::d_call_cc_nn      );
add_ins(1,1,0,0,1,0,0,1, &z80dis::d_ret             );
add_ins(1,1,CC___,0,0,0, &z80dis::d_ret_cc          );
add_ins(1,1,T____,1,1,1, &z80dis::d_rst             );
/*...e*/
/*...sio group:8:*/
add_ins(1,1,0,1,1,0,1,1, &z80dis::d_in_a_n          );
add_ins(1,1,0,1,0,0,1,1, &z80dis::d_out_n_a         );
/*...e*/
/*...smisc:8:*/
add_ins(0,0,1,0,0,1,1,1, &z80dis::d_daa             );
add_ins(0,0,1,0,1,1,1,1, &z80dis::d_cpl             );
add_ins(0,0,1,1,1,1,1,1, &z80dis::d_ccf             );
add_ins(0,0,1,1,0,1,1,1, &z80dis::d_scf             );
add_ins(0,0,0,0,0,0,0,0, &z80dis::d_nop             );
add_ins(0,1,1,1,0,1,1,0, &z80dis::d_halt            );
add_ins(1,1,1,1,0,0,1,1, &z80dis::d_di              );
add_ins(1,1,1,1,1,0,1,1, &z80dis::d_ei              );
add_ins(1,1,1,0,1,1,0,1, &z80dis::d_ed_prefix       );
/*...e*/
	}
/*...e*/

/*...sdisassemble_non_prefix:0:*/
// Can be overridden, if we come to implement son-of-Z80 by inheriting from it.

Boolean z80dis::disassemble_non_prefix(u8 op, u16 & a, char *s)
	{
	z80disfn f = disfn_table[op];
	return(*this .* f)(op, a, s);
	}
/*...e*/
/*...sdisassemble:0:*/
Boolean z80dis::disassemble(u16 & a, char *s)
	{
	u16 a_start = a;
	char *p = s;

	if ( show_opcode )
		p += (1+8+1);

	u8 op;
	if ( !read8(a++, op) )
		return FALSE;

	state = 0; // No prefixes, no illegalities
	switch ( op )
		{
		case 0xdd:
			state |= IX_PREFIX;
			if ( !read8(a++, op) )
				return FALSE;
			break;
		case 0xfd:
			state |= IY_PREFIX;
			if ( !read8(a++, op) )
				return FALSE;
			break;
		}

	if ( ! disassemble_non_prefix(op, a, p) )
		return FALSE;

	if ( show_ill )
		// Display illegal aspects of instruction we've decoded
		{
		if ( state & ILL_PREFIX )
			strcat(s, " illegal-ix/iy-prefix");
		if ( state & ILL_SHROT )
			strcat(s, " illegal-shift/rotate-op");
		if ( state & ILL_USE_HL_FORM )
			strcat(s, " long-form");
		if ( state & ILL_2_MEM_OP )
			strcat(s, " 2-mem-op");
		}

	if ( show_opcode )
		{
		p = s;
		*p++ = '@';
		for ( u16 a2 = a_start; a2 != a && p < s+1+8+1-2; a2++ )
			{
			if ( !read8(a2, op) )
				return FALSE;
			sprintf(p, "%02x", (unsigned) op);
			p += 2;
			}
		while ( p < s+1+8+1 )
			*p++ = ' ';
		}

//	fprintf(stderr, "0x%04x : %s\r\n", (unsigned) a_start, s);

	return TRUE;
	}
/*...e*/
/*...sref_code:0:*/
Boolean z80dis::ref_code(u16 a, u16 & r)
	{
	u8 op;
	if ( !read8(a++, op) )
		return FALSE;

	if (  op       == 0xc3 || // jp nn
	     (op&0xc7) == 0xc2 || // jp cc,nn
	      op       == 0xcd || // call nn
	     (op&0xc7) == 0xc4 )  // call cc,nn
		return read16(a, r);

	if ( op == 0x18 || // jr
	     op == 0x38 || // jr c
	     op == 0x30 || // jr nc
	     op == 0x28 || // jr z
	     op == 0x20 || // jr nz
	     op == 0x10 )  // djnz
		{
		u8 e;
		if ( !read8(a++, e) )
			return FALSE;
		r = (u16) ( a+(s16)(s8)e );
		return TRUE;
		}		

	if ( (op&0xc7) == 0xc7 ) // rst
		{
		r = (u16) (op&0x38);
		return TRUE;
		}

	return FALSE;
	}
/*...e*/
/*...sref_data:0:*/
Boolean z80dis::ref_data(u16 a, u16 & r)
	{
	u8 op;
	if ( !read8(a++, op) )
		return FALSE;

	if ( op == 0xdd || op == 0xfd )
		// Skip IX or IY prefix
		if ( !read8(a++, op) )
			return FALSE;

	if ( (op&0xcf) == 0x01 ) // ld dd,nn
		return read16(a, r);

	if ( op == 0x2a ) // ld inx,(nn)
		{
		u16 nn;
		if ( !read16(a, nn) )
			return FALSE;
		return read16(nn, r);
		}

	if ( op == 0xed )
		{
		u8 op2;
		if ( !read8(a++, op2) )
			return FALSE;
		if ( (op2&0xcf) == 0x4b ) // ld dd,(nn)
			{
			u16 nn;
			if ( !read16(a, nn) )
				return FALSE;
			return read16(nn, r);
			}
		}

	return FALSE;
	}
/*...e*/

/*...sz80dis:0:*/
z80dis::z80dis() { build_decode_tables(); }
/*...e*/
/*...e*/

static dis *d;

/*...sbedis_init:0:*/
BEDISEXPORT Boolean BEDISENTRY bedis_init(
	BEDISREAD32 read,
	BEDISSYMBOLIC32 symbolic,
	const char *(&err)
	)
	{
	err=err; // Suppress warning
	d_read     = read;
	d_symbolic = symbolic;
	d = new z80dis();
	return TRUE;
	}
/*...e*/
/*...sbedis_term:0:*/
BEDISEXPORT void BEDISENTRY bedis_term(void)
	{
	}
/*...e*/
/*...sbedis_disassemble:0:*/
BEDISEXPORT Boolean BEDISENTRY bedis_disassemble(
	BEDISADDR32 addr,
	BEDISADDR32 & bytes,
	char *display
	)
	{
	u16 a       = (u16) addr;
	u16 a_start = (u16) addr;
	if ( !d->disassemble(a, display) )
		return FALSE;
	bytes = (BEDISADDR32) (u16) (a - a_start);
	return TRUE;
	}
/*...e*/
/*...sbedis_ref_code:0:*/
BEDISEXPORT Boolean BEDISENTRY bedis_ref_code(
	BEDISADDR32 addr,
	BEDISADDR32 & ref_addr
	)
	{
	u16 r;
	if ( !d->ref_code((u16) addr, r) )
		return FALSE;
	ref_addr = r;
	return TRUE;
	}
/*...e*/
/*...sbedis_ref_data:0:*/
BEDISEXPORT Boolean BEDISENTRY bedis_ref_data(
	BEDISADDR32 addr,
	BEDISADDR32 & ref_addr
	)
	{
	u16 r;
	if ( !d->ref_data((u16) addr, r) )
		return FALSE;
	ref_addr = r;
	return TRUE;
	}
/*...e*/
/*...sbedis_options:0:*/
BEDISEXPORT Boolean BEDISENTRY bedis_options(
	const char *options,
	const char *(&err)
	)
	{
	char buf[200+1];
	strcpy(buf, options);
	for ( const char *p = strtok(buf, ", ");
	      p != 0;
	      p = strtok(0, ", ") )
		{
		     if ( !strcmp(p, "sym"  ) ) use_syms    = TRUE;
		else if ( !strcmp(p, "nosym") ) use_syms    = FALSE;
		else if ( !strcmp(p, "op"   ) ) show_opcode = TRUE;
		else if ( !strcmp(p, "noop" ) ) show_opcode = FALSE;
		else if ( !strcmp(p, "ill"  ) ) show_ill    = TRUE;
		else if ( !strcmp(p, "noill") ) show_ill    = FALSE;
		else if ( !strcmp(p, "mtx"  ) ) mtx_exts    = TRUE;
		else if ( !strcmp(p, "nomtx") ) mtx_exts    = FALSE;
		else
			{
			err = "must be one of: sym, nosym, op, noop, ill, noill, mtx, nomtx";
			return FALSE;
			}
		}
	return TRUE;
	}
/*...e*/

/*...sOS specific bits:0:*/
#ifdef DOS32
int main(int term) { term=term; return 0; }
#endif

#ifdef AIX

extern "C" {

BEDIS_EXPORT * __start(void)
	{
	static BEDIS_EXPORT exports[] =
		{
		(BEDIS_EP) bedis_init       , "bedis_init"       ,
		(BEDIS_EP) bedis_term       , "bedis_term"       ,
		(BEDIS_EP) bedis_disassemble, "bedis_disassemble",
		(BEDIS_EP) bedis_ref_code   , "bedis_ref_code"   ,
		(BEDIS_EP) bedis_ref_data   , "bedis_ref_data"   ,
		(BEDIS_EP) bedis_options    , "bedis_options"    ,
		(BEDIS_EP) 0                , 0
		};
	return exports;
	}

}

#endif

#ifdef NW

#include <conio.h>
#include <process.h>
#include <advanced.h>

static int tid;

extern "C" BEDISEXPORT void BEDISENTRY _bez80_term(void);

BEDISEXPORT void BEDISENTRY _bez80_term(void)
	{
	ResumeThread(tid);
	}

int main(int argc, char *argv[])
	{
	argc=argc; argv=argv; // Suppress warnings
	int nid = GetNLMID();
	SetAutoScreenDestructionMode(TRUE);
	SetNLMDontUnloadFlag(nid);
	tid = GetThreadID();
	SuspendThread(tid);
	ClearNLMDontUnloadFlag(nid);
	return 0;
	}

#endif
/*...e*/
