/*

cpmfuse.c - CP/M filesystem in userspace

http://www.moria.de/~michael/cpmtools/
http://fuse.sourceforge.net/

*/

/*...sincludes:0:*/
#define	FUSE_USE_VERSION 26

#include <fuse.h>
#include <stdio.h>
#include <ctype.h>
#include <stdlib.h>
#include <stddef.h>
#include <stdarg.h>
#include <string.h>
#include <errno.h>
#include <fcntl.h>

#include "config.h"
#include "cpmfs.h"

/*...vconfig\46\h:0:*/
/*...vcpmfs\46\h:0:*/
/*...e*/

/*...sBOOLEAN:0:*/
typedef int BOOLEAN;
#define	TRUE  1
#define	FALSE 0
/*...e*/

const char cmd[] = "cpmfuse"; /* needed by cpmfs.c */

/*...sfatal:0:*/
static void fatal(const char *fmt, ...)
	{
	va_list	vars;
	char s[256+1];
	va_start(vars, fmt);
	vsprintf(s, fmt, vars);
	va_end(vars);
	fprintf(stderr, "%s: %s\n", cmd, s);
	exit(1);
	}
/*...e*/

/*...svars:0:*/
static struct cpmSuperBlock drive;
static struct cpmInode      root;
static int user;
static BOOLEAN invert;
/*...e*/

/*...scpmXxxxSafe:0:*/
/* I have observed that cpmtools cpmcp only copies in 4096 byte blocks,
   and when I try to write larger blocks in one go, the file size wraps
   modulo 16384 bytes. So these wrappers avoid this problem. */

static int cpmReadSafe(struct cpmFile *file, char *buffer, int count)
	{
	int total = 0;
	while ( count > 0 )
		{
		int thisgo = ( count < 4096 ) ? count : 4096;
		int n = cpmRead(file, buffer, thisgo);
		total += n;
		if ( n < thisgo )
			break;
		buffer += n;
		count -= n;
		}
	return total;
	}

static int cpmWriteSafe(struct cpmFile *file, const char *buffer, int count)
	{
	int total = 0;
	while ( count > 0 )
		{
		int thisgo = ( count < 4096 ) ? count : 4096;
		int n = cpmWrite(file, buffer, thisgo);
		total += n;
		if ( n < thisgo )
			break;
		buffer += n;
		count -= n;
		}
	return total;
	}
/*...e*/

/*...sinvert_char:0:*/
static char invert_char(char ch)
	{
	if ( invert )
		{
		if ( islower(ch) )
			return toupper(ch);
		if ( isupper(ch) )
			return tolower(ch);
		}
	return ch;
	}
/*...e*/
/*...scpm2unix:0:*/
static BOOLEAN cpm2unix(const char *c, char *u)
	{
	for ( ; *c != '\0'; c++, u++ )
		{
		if ( *c == '/' )
			return FALSE;
		*u = invert_char(*c);
		}
	*u = '\0';
	return TRUE;
	}
/*...e*/
/*...sunix2cpm:0:*/
/* Valid filenames on the UNIX side, include "FILE" and "FILE.EXT".
   "", ".EXT" or "FILE." are not valid. */

/*...svalid_filename_char:0:*/
/* cpmtools considers the files to be in lower case.
   Of course, in reality, they're not, thats the game it plays. */

static BOOLEAN valid_filename_char(char ch)
	{
	if ( ch >= '0' && ch <= '9' ) return TRUE;
	if ( ch >= 'a' && ch <= 'z' ) return TRUE;
	if ( ch >= 'A' && ch <= 'Z' ) return FALSE;
	if ( ch <= ' ' || ch >  '~' ) return FALSE;
	if ( ch == '<' ||
	     ch == '>' ||
	     ch == '.' ||
	     ch == ',' ||
	     ch == ';' ||
	     ch == ':' ||
	     ch == '=' ||
	     ch == '[' ||
	     ch == ']' ||
	     ch == '%' ||
	     ch == '|' ||
	     ch == '(' ||
	     ch == ')' ||
	     ch == '/' ||
	     ch == '\\' ||
	     ch == '?' ||
	     ch == '*' )
		return FALSE;
	/* Assuming its some ASCII punctuation we allow */
	return TRUE;
	}
/*...e*/

static BOOLEAN unix2cpm(const char *u, char *c)
	{
	int n = 0;
	for ( ; *u != '\0'; u++, c++ )
		{
		char u2 = invert_char(*u);
		BOOLEAN u2v = valid_filename_char(u2);
		if ( n < 8 && u2v )
			{
			*c = u2;
			++n;
			}
		else if ( n >=1 && n <= 8 && u2 == '.' )
			{
			*c = '.';
			n = 9;
			}
		else if ( n >= 9 && n < 12 && u2v )
			{
			*c = u2;
			++n;
			}
		else
			return FALSE;
		}
	if ( n == 0 || n == 9 )
		return FALSE;
	*c = '\0';
	return TRUE;
	}
/*...e*/

/*...scpmfuse_readdir:0:*/
/* We'll ignore the offset and always pass 0 to the filler. */

static int cpmfuse_readdir(
	const char *path,
	void *buf, fuse_fill_dir_t filler,
	off_t offset, struct fuse_file_info *fi
	)
	{
	struct cpmFile file;
	struct cpmDirent dirent;
	if ( strcmp(path, "/") )
		return -ENOENT;
	filler(buf, ".", NULL, 0);
	filler(buf, "..", NULL, 0);
	cpmOpendir(&root, &file);
	while ( cpmReaddir(&file, &dirent) )
		{
		const char *p = dirent.name;
		if ( p[0] == (user/10)+'0' &&
		     p[1] == (user%10)+'0' )
			{
			char name[8+1+3+1];
			if ( cpm2unix(p+2, name) )
				filler(buf, name, NULL, 0);
			}
		}
	return 0;
	}
/*...e*/
/*...scpmfuse_getattr:0:*/
static int cpmfuse_getattr(const char *path, struct stat *stbuf)
	{
	memset(stbuf, 0, sizeof(struct stat));
	if ( !strcmp(path, "/") )
		{
		stbuf->st_mode  = S_IFDIR | 0755;
		stbuf->st_nlink = 2;
		}
	else
		{
		char dirent[2+8+1+3+1];
		struct cpmInode ino;
		struct cpmStat cstbuf;
		dirent[0] = (user/10)+'0';
		dirent[1] = (user%10)+'0';
		if ( ! unix2cpm(path+1, dirent+2) )
			return -ENOENT;
		if ( cpmNamei(&root, dirent, &ino) != 0 )
			return -ENOENT;
		cpmStat(&ino, &cstbuf);
		stbuf->st_mode = S_IFREG | 0444;
		if ( cstbuf.mode & 0222 )
			stbuf->st_mode |= 0222;
		stbuf->st_nlink = 1;
		stbuf->st_size  = cstbuf.size;
		stbuf->st_ctime = cstbuf.ctime;
		stbuf->st_atime = cstbuf.atime;
		stbuf->st_mtime = cstbuf.mtime;
		}
	return 0;
	}
/*...e*/
/*...scpmfuse_creat:0:*/
static int cpmfuse_create(
	const char *path,
	mode_t mode,
	struct fuse_file_info *fi
	)
	{
	char dirent[2+8+1+3+1];
	struct cpmInode ino;
	dirent[0] = (user/10)+'0';
	dirent[1] = (user%10)+'0';
	if ( ! unix2cpm(path+1, dirent+2) )
		return -ENOENT;
	mode = ( mode & 0200 ) ? 0666 : 0444;
	if ( cpmCreat(&root, dirent, &ino, mode) != 0 )
		return -EIO;
	return 0;
	}
/*...e*/
/*...scpmfuse_open:0:*/
static int cpmfuse_open(const char *path, struct fuse_file_info *fi)
	{
	char dirent[2+8+1+3+1];
	struct cpmInode ino;
	struct cpmStat cstbuf;
	dirent[0] = (user/10)+'0';
	dirent[1] = (user%10)+'0';
	if ( ! unix2cpm(path+1, dirent+2) )
		return -ENOENT;
	if ( cpmNamei(&root, dirent, &ino) != 0 )
		return -ENOENT;
	cpmStat(&ino, &cstbuf);
	if ( (fi->flags & O_ACCMODE) != O_RDONLY &&
	     (cstbuf.mode & 0200) == 0 )
		/* User wants to write, but file is not writeable */
		return -EACCES;
	return 0;
	}
/*...e*/
/*...scpmfuse_unlink:0:*/
static int cpmfuse_unlink(const char *path)
	{
	char dirent[2+8+1+3+1];
	struct cpmInode ino;
	struct cpmStat cstbuf;
	struct cpmFile file;
	dirent[0] = (user/10)+'0';
	dirent[1] = (user%10)+'0';
	if ( ! unix2cpm(path+1, dirent+2) )
		return -ENOENT;
	if ( cpmUnlink(&root, dirent) != 0 )
		return -ENOENT;
	return 0;
	}
/*...e*/
/*...scpmfuse_read:0:*/
static int cpmfuse_read(
	const char *path,
	char *buf, size_t size, off_t offset,
	struct fuse_file_info *fi
	)
	{
	char dirent[2+8+1+3+1];
	struct cpmInode ino;
	struct cpmStat cstbuf;
	struct cpmFile file;
	dirent[0] = (user/10)+'0';
	dirent[1] = (user%10)+'0';
	if ( ! unix2cpm(path+1, dirent+2) )
		return -ENOENT;
	if ( cpmNamei(&root, dirent, &ino) != 0 )
		return -ENOENT;
	cpmStat(&ino, &cstbuf);
	if ( offset > cstbuf.size )
		size = 0;
	else if ( offset + size > cstbuf.size )
		size = cstbuf.size - offset;
	cpmOpen(&ino, &file, O_RDONLY);
	while ( offset > 0 )
		{
		unsigned char dump_buf[4096];
		off_t thisgo = ( offset<sizeof(dump_buf) ) ? offset : sizeof(dump_buf);
		cpmReadSafe(&file, dump_buf, thisgo);
		offset -= thisgo;
		}
	size = cpmReadSafe(&file, buf, size);
	cpmClose(&file);
	return size;
	}
/*...e*/
/*...scpmfuse_write:0:*/
static int cpmfuse_write(
	const char *path,
	const char *buf, size_t size, off_t offset,
	struct fuse_file_info *fi
	)
	{
	char dirent[2+8+1+3+1];
	struct cpmInode ino;
	struct cpmStat cstbuf;
	struct cpmFile file;
	dirent[0] = (user/10)+'0';
	dirent[1] = (user%10)+'0';
	if ( ! unix2cpm(path+1, dirent+2) )
		return -ENOENT;
	if ( cpmNamei(&root, dirent, &ino) != 0 )
		return -ENOENT;
	cpmStat(&ino, &cstbuf);
	if ( offset > cstbuf.size )
		return -EIO;
	cpmOpen(&ino, &file, O_RDWR);
	while ( offset > 0 )
		{
		unsigned char dump_buf[4096];
		off_t thisgo = ( offset<sizeof(dump_buf) ) ? offset : sizeof(dump_buf);
		cpmReadSafe(&file, dump_buf, thisgo);
		offset -= thisgo;
		}
	size = cpmWriteSafe(&file, buf, size);
	cpmClose(&file);
	return size;
	}
/*...e*/
/*...scpmfuse_rename:0:*/
static int cpmfuse_rename(const char *path_old, const char *path_new)
	{
	char dirent_old[2+8+1+3+1];
	char dirent_new[2+8+1+3+1];
	dirent_old[0] = (user/10)+'0';
	dirent_old[1] = (user%10)+'0';
	dirent_new[0] = (user/10)+'0';
	dirent_new[1] = (user%10)+'0';
	if ( ! unix2cpm(path_old+1, dirent_old+2) )
		return -ENOENT;
	if ( ! unix2cpm(path_new+1, dirent_new+2) )
		return -ENOENT;
	if ( cpmRename(&root, dirent_old, dirent_new) != 0 )
		return -EIO;
	return 0;
	}
/*...e*/
/*...scpmfuse_chmod:0:*/
static int cpmfuse_chmod(const char *path, mode_t mode)
	{
	char dirent[2+8+1+3+1];
	struct cpmInode ino;
	struct cpmStat cstbuf;
	struct utimbuf times;
	dirent[0] = (user/10)+'0';
	dirent[1] = (user%10)+'0';
	if ( ! unix2cpm(path+1, dirent+2) )
		return -ENOENT;
	if ( cpmNamei(&root, dirent, &ino) != 0 )
		return -ENOENT;
	mode = ( mode & 0200 ) ? 0666 : 0444;
	cpmChmod(&ino, mode);
	return 0;
	}
/*...e*/
/*...scpmfuse_utimens:0:*/
/* We're supposed to honor the nanoseconds. */
static int cpmfuse_utimens(const char *path, const struct timespec tv[2])
	{
	char dirent[2+8+1+3+1];
	struct cpmInode ino;
	struct cpmStat cstbuf;
	struct utimbuf times;
	dirent[0] = (user/10)+'0';
	dirent[1] = (user%10)+'0';
	if ( ! unix2cpm(path+1, dirent+2) )
		return -ENOENT;
	if ( cpmNamei(&root, dirent, &ino) != 0 )
		return -ENOENT;
	times.actime  = tv[0].tv_sec;
	times.modtime = tv[1].tv_sec;
	cpmUtime(&ino, &times);
	return 0;
	}
/*...e*/
/*...scpmfuse_truncate:0:*/
static int cpmfuse_truncate(const char *path, off_t size)
	{
	char dirent[2+8+1+3+1];
	struct cpmInode ino;
	struct cpmStat cstbuf;
	struct cpmFile file;
	unsigned char *buf;
	dirent[0] = (user/10)+'0';
	dirent[1] = (user%10)+'0';
	if ( ! unix2cpm(path+1, dirent+2) )
		return -ENOENT;
	if ( cpmNamei(&root, dirent, &ino) != 0 )
		return -ENOENT;
	cpmStat(&ino, &cstbuf);
	if ( size > cstbuf.size )
		/* Truncate can't make the file longer */
		return -EIO;
	/* CP/M doesn't support filesystems beyond 8MB,
	   so we can catch silly arguments like this. */
	if ( size > 8 * 1024 * 1024 )
		return -EIO;
	/* We should easily be able to allocate 8MB of temporary buffer.
	   +1, so that we never malloc(0), which can return NULL */
	if ( (buf = malloc(size+1)) == NULL )
		return -EIO;
	cpmOpen(&ino, &file, O_RDONLY);
	cpmReadSafe(&file, buf, size);
	cpmClose(&file);
	if ( cpmUnlink(&root, dirent) != 0 )
		{
		free(buf);
		return -EIO;
		}
	/* cpmCreat works because the cpmUnlink left a directory space */
	cpmCreat(&root, dirent, &ino, cstbuf.mode);
	cpmOpen(&ino, &file, O_WRONLY);
	/* cpmWrite works because new size <= old size */
	cpmWriteSafe(&file, buf, size);
	cpmClose(&file);
	free(buf);
	return 0;
	}
/*...e*/
/*...scpmfuse_statfs:0:*/
static int cpmfuse_statfs(const char *path, struct statvfs *stvbuf)
	{
	struct cpmStatFS cs;
	cpmStatFS(&root, &cs);
	stvbuf->f_bsize   = cs.f_bsize;
	stvbuf->f_frsize  = 0;			/* ignored */
	stvbuf->f_blocks  = cs.f_blocks;
	stvbuf->f_bfree   = cs.f_bfree;
	stvbuf->f_bavail  = cs.f_bavail;
	stvbuf->f_files   = cs.f_files;
	stvbuf->f_ffree   = cs.f_ffree;		/* free directory entries */
	stvbuf->f_favail  = 0;			/* ignored */
	stvbuf->f_fsid    = 0;			/* ignored */
	stvbuf->f_flag    = 0;			/* ignored */
	stvbuf->f_namemax = 8+1+3;		/* cs.f_namelen returns 11 */
	return 0;
	}
/*...e*/
/*...scpmfuse_destroy:0:*/
static void cpmfuse_destroy(void *p)
	{
	cpmSync(&drive);
	cpmUmount(&drive);
	Device_close(&drive.dev);
	}
/*...e*/
/*...scpmfuse_oper:0:*/
static struct fuse_operations cpmfuse_oper =
	{
	.getattr  = cpmfuse_getattr,
	.readdir  = cpmfuse_readdir,
	.create   = cpmfuse_create,
	.open     = cpmfuse_open,
	.unlink   = cpmfuse_unlink,
	.read     = cpmfuse_read,
	.write    = cpmfuse_write,
	.rename   = cpmfuse_rename,
	.chmod    = cpmfuse_chmod,
	.utimens  = cpmfuse_utimens,
	.truncate = cpmfuse_truncate,
	.statfs   = cpmfuse_statfs,
	.destroy  = cpmfuse_destroy,
	};
/*...e*/
/*...smain:0:*/
/* Jump through a few hoops to fit in with the way fuse likes you to
   do command line arguments. */

struct cpmfuse_config
	{
	const char *format;
	const char *image;
	int user;
	BOOLEAN invert;
	};
enum
	{
	KEY_VERSION,
	KEY_HELP,
	};
#define	CPMFUSE_OPT(t, p, v) { t, offsetof(struct cpmfuse_config, p), v }
static struct fuse_opt cpmfuse_opts[] =
	{
	CPMFUSE_OPT("-f %s", format, 0),
	CPMFUSE_OPT("--format %s", format, 0),
	CPMFUSE_OPT("-i %s", image, 0),
	CPMFUSE_OPT("--image %s", image, 0),
	CPMFUSE_OPT("-u %d", user, 0),
	CPMFUSE_OPT("--user %d", user, 0),
	CPMFUSE_OPT("-v", invert, TRUE),
	CPMFUSE_OPT("--invert-case", invert, TRUE),
	FUSE_OPT_KEY("-V", KEY_VERSION),
	FUSE_OPT_KEY("--version", KEY_VERSION),
	FUSE_OPT_KEY("-h", KEY_HELP),
	FUSE_OPT_KEY("--help", KEY_HELP),
	FUSE_OPT_END
	};

/*...scpmfuse_opt_proc:0:*/
static int cpmfuse_opt_proc(
	void *data, const char *arg, int key,
	struct fuse_args *outargs
	)
	{
	switch ( key )
		{
		case KEY_HELP:
			fprintf(stderr, "usage: %s mountpoint [flags]\n", cmd);
			fprintf(stderr, "flags: -f,--format format  f/s format (default $CPMTOOLSFMT or %s)\n", FORMAT);
			fprintf(stderr, "       -i,--image image    f/s image (must be specified)\n");
			fprintf(stderr, "       -u,--user user      CP/M user (default 0)\n");
			fprintf(stderr, "       -v,--invert-case    toggle case to match what CP/M uses\n");
			fprintf(stderr, "       -h,--help           print help\n");
			fprintf(stderr, "       -V,--version        print version\n");
			fprintf(stderr, "       ...                 other fuse related options\n");
			exit(1);
		case KEY_VERSION:
			fprintf(stderr, "%s version 1.2\n", cmd);
			exit(0);
		}
	return 1;
	}
/*...e*/

int main(int argc, char *argv[])
	{
	const char *err;
	struct fuse_args args = FUSE_ARGS_INIT(argc, argv);
	struct cpmfuse_config config;

	memset(&config, 0, sizeof(struct cpmfuse_config));
	config.format = getenv("CPMTOOLSFMT");
	if ( config.format == NULL )
		config.format = FORMAT;
	config.user   = 0;
	config.invert = FALSE;

	if ( fuse_opt_parse(&args, &config, cpmfuse_opts, cpmfuse_opt_proc) == -1 )
		return -1;

	fuse_opt_add_arg(&args, "-s");

	if ( config.image == NULL )
		fatal("CP/M filesystem image must be specified with -i");

	if ( (err = Device_open(&drive.dev, config.image, O_RDWR, NULL)) != NULL )
		fatal("can't open %s (%s)", config.image, err);

	if ( cpmReadSuper(&drive,&root,config.format) != 0 )
		fatal("can't read super block on %s of format %s\n", config.image, config.format);

	user   = config.user;
	invert = config.invert;

	return fuse_main(args.argc, args.argv, &cpmfuse_oper, NULL);
	}
/*...e*/
