//
// cube.C - Rubiks Cube
//

/*...sincludes:0:*/
#ifdef NO_CINCLUDES
  #include <stdio.h>
  #include <ctype.h>
  #include <stdlib.h>
  #include <stdarg.h>
  #include <string.h>
  #include <float.h>
#else
  #include <cstdio>
  #include <cctype>
  #include <cstdlib>
  #include <cstdarg>
  #include <cstring>
  #include <cfloat>
#endif

#include "mr.h"
/*...e*/

/*...sprogname:0:*/
static char progname[] = "cube";
/*...e*/
/*...sfatal:0:*/
static void fatal(const char *fmt, ...)
	{
	va_list	vars;
	char s[256+1];
	va_start(vars, fmt);
	vsprintf(s, fmt, vars);
	va_end(vars);
	fprintf(stderr, "%s: %s\n", progname, s);
	exit(1);
	}
/*...e*/
/*...sconstants:0:*/
#ifndef ORDER
#define	ORDER 3
#endif

#define	CUBIE_SIZE     0.8
#define	CUBIE_SIZE_GAP 1.0

#ifdef VIEWVIO
#define	BMP_W 200
#define	BMP_H 200
#else
#define	BMP_W 512
#define	BMP_H 512
#endif
/*...e*/
/*...spalette:0:*/
// 7 colours, 0x10 evenly spaced shades of each

static void palette(Bitmap & b8)
	{
	//                           b w  r g b y  o
	static double rtab[0x10] = { 0,1, 1,0,0,1, 1   };
	static double gtab[0x10] = { 0,1, 0,1,0,1, 0.7 };
	static double btab[0x10] = { 0,1, 0,0,1,0, 0   };
	int c, s;
	for ( c = 0; c < 7; c++ )
		for ( s = 0; s < 0x10; s++ )
			{
			double shade = s / 15.0;
			b8.set_pal(c*0x10+s, Rgb(rtab[c] * shade,
						 gtab[c] * shade,
						 btab[c] * shade));
			}
	}
/*...e*/
/*...senum FaceIndex:0:*/
enum FaceIndex
	{
	FI_Xmin, FI_Xmax,
	FI_Ymin, FI_Ymax,
	FI_Zmin, FI_Zmax
	};
/*...e*/
/*...senum FaceColour:0:*/
enum FaceColour
	{
	FC_Black,
	FC_Red,
	FC_Green,
	FC_Blue,
	FC_Orange,
	FC_Yellow,
	FC_White
	};

static int colours[] = { 0x00, 0x20, 0x30, 0x40, 0x60, 0x50, 0x10 };
/*...e*/
/*...senum Rotation:0:*/
enum Rotation
	{
	RO_Xaxis,
	RO_Yaxis,
	RO_Zaxis
	};
/*...e*/
/*...sclass Cubie:0:*/
class Cubie
	{
	void cycle4(FaceIndex fi0, FaceIndex fi1, FaceIndex fi2, FaceIndex fi3);
	void cycle2(FaceIndex fi0, FaceIndex fi1, FaceIndex fi2, FaceIndex fi3);
public:
	FaceColour faces[6];
	Cubie();
	void render(double x, double y, double z, PolygonOpList & ol) const;
	void rot_x_1();
	void rot_x_2();
	void rot_x_3();
	void rot_y_1();
	void rot_y_2();
	void rot_y_3();
	void rot_z_1();
	void rot_z_2();
	void rot_z_3();
	};

/*...sCubie:0:*/
Cubie::Cubie()
	{
	for ( FaceIndex fi = FI_Xmin; fi <= FI_Zmax; fi = (FaceIndex) (fi+1) )
		faces[fi] = FC_Black;
	}
/*...e*/
/*...srender:0:*/
void Cubie::render(
	double x, double y, double z,
	PolygonOpList & ol
	) const
	{
	ol.vertex(Xyz(x+CUBIE_SIZE,y+CUBIE_SIZE,z+CUBIE_SIZE))	// #7
	  .vertex(Xyz(x+CUBIE_SIZE,y+CUBIE_SIZE,z-CUBIE_SIZE))
	  .vertex(Xyz(x+CUBIE_SIZE,y-CUBIE_SIZE,z+CUBIE_SIZE))
	  .vertex(Xyz(x+CUBIE_SIZE,y-CUBIE_SIZE,z-CUBIE_SIZE))
	  .vertex(Xyz(x-CUBIE_SIZE,y+CUBIE_SIZE,z+CUBIE_SIZE))
	  .vertex(Xyz(x-CUBIE_SIZE,y+CUBIE_SIZE,z-CUBIE_SIZE))
	  .vertex(Xyz(x-CUBIE_SIZE,y-CUBIE_SIZE,z+CUBIE_SIZE))
	  .vertex(Xyz(x-CUBIE_SIZE,y-CUBIE_SIZE,z-CUBIE_SIZE))	// #0
	  .col(colours[faces[FI_Xmin]]) .polygon(0,1,3,2) 
	  .col(colours[faces[FI_Xmax]]) .polygon(4,6,7,5)
	  .col(colours[faces[FI_Ymin]]) .polygon(0,4,5,1)
	  .col(colours[faces[FI_Ymax]]) .polygon(2,3,7,6)
	  .col(colours[faces[FI_Zmin]]) .polygon(0,2,6,4)
	  .col(colours[faces[FI_Zmax]]) .polygon(1,5,7,3);
	}
/*...e*/
/*...scycle4:0:*/
void Cubie::cycle4(FaceIndex fi0, FaceIndex fi1, FaceIndex fi2, FaceIndex fi3)
	{
	FaceColour t = faces[fi3];
	faces[fi3] = faces[fi2];
	faces[fi2] = faces[fi1];
	faces[fi1] = faces[fi0];
	faces[fi0] = t;
	}
/*...e*/
/*...scycle2:0:*/
void Cubie::cycle2(FaceIndex fi0, FaceIndex fi1, FaceIndex fi2, FaceIndex fi3)
	{
	FaceColour t = faces[fi0];
	faces[fi0] = faces[fi1];
	faces[fi1] = t;
	t = faces[fi2];
	faces[fi2] = faces[fi3];
	faces[fi3] = t;
	}
/*...e*/
/*...srot_ functions:0:*/
void Cubie::rot_x_1() { cycle4(FI_Ymin, FI_Zmin, FI_Ymax, FI_Zmax); }
void Cubie::rot_x_2() { cycle2(FI_Ymin, FI_Ymax, FI_Zmin, FI_Zmax); }
void Cubie::rot_x_3() { cycle4(FI_Ymin, FI_Zmax, FI_Ymax, FI_Zmin); }

void Cubie::rot_y_1() { cycle4(FI_Zmin, FI_Xmin, FI_Zmax, FI_Xmax); }
void Cubie::rot_y_2() { cycle2(FI_Zmin, FI_Zmax, FI_Xmin, FI_Xmax); }
void Cubie::rot_y_3() { cycle4(FI_Zmin, FI_Xmax, FI_Zmax, FI_Xmin); }

void Cubie::rot_z_1() { cycle4(FI_Xmin, FI_Ymin, FI_Xmax, FI_Ymax); }
void Cubie::rot_z_2() { cycle2(FI_Xmin, FI_Xmax, FI_Ymin, FI_Ymax); }
void Cubie::rot_z_3() { cycle4(FI_Xmin, FI_Ymax, FI_Xmax, FI_Ymin); }
/*...e*/
/*...e*/
/*...sclass Cube:0:*/
class Cube
	{
public:
	Cubie cubies[ORDER][ORDER][ORDER];
	Cube();
	void start();
	void render(PolygonOpList & ol, Rotation ro, const double angles[]) const;
	void render(PolygonOpList & ol) const
		{
		double angles[ORDER];
		for ( int i = 0; i < ORDER; i++ )
			angles[i] = 0.0;
		render(ol, RO_Xaxis, angles);
		}
	void rotate(Rotation ro, int slice, int turns);
	};

/*...sCube:0:*/
Cube::Cube()
	{
	start();
	}
/*...e*/
/*...sstart:0:*/
// Standard left-handed Rubiks Cube colouring

void Cube::start()
	{
	int x, y, z;
	for ( y = 0; y < ORDER; y++ )
		for ( z = 0; z < ORDER; z++ )
			{
			cubies[0      ][y][z].faces[FI_Xmin] = FC_Green ;
			cubies[ORDER-1][y][z].faces[FI_Xmax] = FC_Blue  ;
			}
	for ( z = 0; z < ORDER; z++ )
		for ( x = 0; x < ORDER; x++ )
			{
			cubies[x][0      ][z].faces[FI_Ymin] = FC_Orange;
			cubies[x][ORDER-1][z].faces[FI_Ymax] = FC_Red   ;
			}
	for ( x = 0; x < ORDER; x++ )
		for ( y = 0; y < ORDER; y++ )
			{
			cubies[x][y][0      ].faces[FI_Zmin] = FC_White ;
			cubies[x][y][ORDER-1].faces[FI_Zmax] = FC_Yellow;
			}
	}
/*...e*/
/*...srender:0:*/
static double coord[ORDER];

class Coords
	{
public:
	Coords()
		{
		for ( int x = 0; x < ORDER; x++ )
#if ORDER & 1
			coord[x] =  (x - ORDER/2) * 2      * CUBIE_SIZE_GAP;
#else
			coord[x] = ((x - ORDER/2) * 2 + 1) * CUBIE_SIZE_GAP;
#endif		
		}
	};

static Coords coords; // Ensure coord gets initialised

void Cube::render(PolygonOpList & ol, Rotation ro, const double angles[]) const
	{
	int x, y, z;
	switch ( ro )
		{
		case RO_Xaxis:
			for ( x = 0; x < ORDER; x++ )
				{
				PolygonOpList ol2;
				for ( y = 0; y < ORDER; y++ )
					for ( z = 0; z < ORDER; z++ )
						cubies[x][y][z].render(
							coord[x],
							coord[y],
							coord[z],
							ol2);
				if ( angles[x] == 0.0 )
					ol += ol2;
				else
					ol += ol2.rot_x(angles[x]);
				}
			break;
		case RO_Yaxis:
			for ( y = 0; y < ORDER; y++ )
				{
				PolygonOpList ol2;
				for ( z = 0; z < ORDER; z++ )
					for ( x = 0; x < ORDER; x++ )
						cubies[x][y][z].render(
							coord[x],
							coord[y],
							coord[z],
							ol2);
				if ( angles[y] == 0.0 )
					ol += ol2;
				else
					ol += ol2.rot_y(angles[y]);
				}
			break;
		case RO_Zaxis:
			for ( z = 0; z < ORDER; z++ )
				{
				PolygonOpList ol2;
				for ( x = 0; x < ORDER; x++ )
					for ( y = 0; y < ORDER; y++ )
						cubies[x][y][z].render(
							coord[x],
							coord[y],
							coord[z],
							ol2);
				if ( angles[z] == 0.0 )
					ol += ol2;
				else
					ol += ol2.rot_z(angles[z]);
				}
			break;
		}
	}
/*...e*/
/*...srotate:0:*/
void Cube::rotate(Rotation ro, int slice, int turns)
	{
	Cubie tmp[ORDER][ORDER];
	int x, y, z;
	turns = ( turns + 4 ) % 4;
	if ( turns == 0 )
		return;
	switch ( ro )
		{
/*...sRO_Xaxis:16:*/
case RO_Xaxis:
	for ( y = 0; y < ORDER; y++ )
		for ( z = 0; z < ORDER; z++ )
			tmp[y][z] = cubies[slice][y][z];
	switch ( turns )
		{
		case 1:
			for ( y = 0; y < ORDER; y++ )
				for ( z = 0; z < ORDER; z++ )
					{
					cubies[slice][y][z] = tmp[z][ORDER-1-y];
					cubies[slice][y][z].rot_x_1();
					}
			break;
		case 2:
			for ( y = 0; y < ORDER; y++ )
				for ( z = 0; z < ORDER; z++ )
					{
					cubies[slice][y][z] = tmp[ORDER-1-y][ORDER-1-z];
					cubies[slice][y][z].rot_x_2();
					}
			break;
		case 3:
			for ( y = 0; y < ORDER; y++ )
				for ( z = 0; z < ORDER; z++ )
					{
					cubies[slice][y][z] = tmp[ORDER-1-z][y];
					cubies[slice][y][z].rot_x_3();
					}
			break;
		}
	break;
/*...e*/
/*...sRO_Yaxis:16:*/
case RO_Yaxis:
	for ( z = 0; z < ORDER; z++ )
		for ( x = 0; x < ORDER; x++ )
			tmp[z][x] = cubies[x][slice][z];
	switch ( turns )
		{
		case 1:
			for ( z = 0; z < ORDER; z++ )
				for ( x = 0; x < ORDER; x++ )
					{
					cubies[x][slice][z] = tmp[x][ORDER-1-z];
					cubies[x][slice][z].rot_y_1();
					}
			break;
		case 2:
			for ( z = 0; z < ORDER; z++ )
				for ( x = 0; x < ORDER; x++ )
					{
					cubies[x][slice][z] = tmp[ORDER-1-z][ORDER-1-x];
					cubies[x][slice][z].rot_y_2();
					}
			break;
		case 3:
			for ( z = 0; z < ORDER; z++ )
				for ( x = 0; x < ORDER; x++ )
					{
					cubies[x][slice][z] = tmp[ORDER-1-x][z];
					cubies[x][slice][z].rot_y_3();
					}
			break;
		}
	break;
/*...e*/
/*...sRO_Zaxis:16:*/
case RO_Zaxis:
	for ( x = 0; x < ORDER; x++ )
		for ( y = 0; y < ORDER; y++ )
			tmp[x][y] = cubies[x][y][slice];
	switch ( turns )
		{
		case 1:
			for ( x = 0; x < ORDER; x++ )
				for ( y = 0; y < ORDER; y++ )
					{
					cubies[x][y][slice] = tmp[y][ORDER-1-x];
					cubies[x][y][slice].rot_z_1();
					}
			break;
		case 2:
			for ( x = 0; x < ORDER; x++ )
				for ( y = 0; y < ORDER; y++ )
					{
					cubies[x][y][slice] = tmp[ORDER-1-x][ORDER-1-y];
					cubies[x][y][slice].rot_z_2();
					}
			break;
		case 3:
			for ( x = 0; x < ORDER; x++ )
				for ( y = 0; y < ORDER; y++ )
					{
					cubies[x][y][slice] = tmp[ORDER-1-y][x];
					cubies[x][y][slice].rot_z_3();
					}
			break;
		}
	break;
/*...e*/
		}
	}
/*...e*/
/*...e*/
/*...srandom_turn:0:*/
static void random_turn(Rotation & ro, int & slice, int & turns)
	{
	static int turns_tab[] = { -1, 1, 2 };
	switch ( rand() % 3 )
		{
		case 0:	ro = RO_Xaxis;	break;
		case 1:	ro = RO_Yaxis;	break;
		case 2:	ro = RO_Zaxis;	break;
		}
	turns = turns_tab[rand()%3];
	slice = ( rand() % ORDER );
	}
/*...e*/
/*...srandom_double:0:*/
static double random_double()
	{
	return (double) ( rand()%2000-1000 ) / 1000.0;
	}
/*...e*/
/*...sinterp:0:*/
static double interp(int i, int j, double p, double q)
	{
	return p + ( (double) i / (double) j ) * ( q - p );
	}
/*...e*/
/*...smain:0:*/
int main(void)
	{
	Boolean loop = TRUE;

#ifdef OS2
	/* Prevent numeric exceptions from terminating this program */
	_control87(EM_UNDERFLOW|EM_DENORMAL, EM_UNDERFLOW|EM_DENORMAL);
#endif

	Bitmap b8(BMP_W, BMP_H, 8);
	palette(b8);
	b8.clear(0x18); // Medium white == Grey

#if defined(VIEWVIO)
	ViewerVio viewer(b8, 7*0x10);
#elif defined(VIEWWIN)
	ViewerWin viewer(b8, 7*0x10, "Rubiks Cube");
#elif defined(VIEWXWIN)
	ViewerXWin viewer(b8, 7*0x10, "Rubiks Cube");
#else
	ViewerBitmapFile viewer(b8, "cube%04d.gif");
#endif

	Cube cube;

	double r0 = 1.0, lat0 = 0.0, lon0 = 0.0;

	while ( loop )
		{
		Rotation ro; int slice, turns;
		random_turn(ro, slice, turns);
		double angles[ORDER];
		for ( int i = 0; i < ORDER; i++ )
			angles[i] = 0.0;
		int maxstep = ( turns == 2 ) ? 80 : 40;
		// Compute r1 between 0.75 and 1.25
		double r1   = 1.0  + random_double() * 0.25      ;
		// Compute lat1 between +/- 70 degrees
		double lat1 =        random_double() * rad(80.0) ;
		// Compute lon1 as lon0 +/- 100 degrees
		double lon1 = lon0 + random_double() * rad(120.0);
		for ( int step = 0; step < maxstep; step++ )
			{
			double angle = (double) step / (double) maxstep * turns * 90.0;
			angles[slice] = rad( angle );
			PolygonOpList ol;
			cube.render(ol, ro, angles);
			b8.clear(0x18); // Medium white == Grey

/*...smake view:24:*/
double r   = interp(step,maxstep,r0  ,r1  );
double lat = interp(step,maxstep,lat0,lat1);
double lon = interp(step,maxstep,lon0,lon1);
View view;
view.eye     = Xyz(0.0,0.0,r*ORDER*2.5).rot_x(lat).rot_y(lon);
view.forward = Xyz(0.0,0.0,-1.0)       .rot_x(lat).rot_y(lon);
view.up      = Xyz(0.0,1.0,0.0)        .rot_x(lat).rot_y(lon);
/*...e*/

			ol.render(view, b8, view.eye);
#ifdef PROFILING
{ static int n = 0; if ( ++n == 1000 ) exit(0); } 
#else
			viewer.refresh();
#endif
			}
		cube.rotate(ro, slice, turns);
		r0 = r1; lat0 = lat1; lon0 = lon1;
		switch ( viewer.keypress() )
			{
			case 'S': case 's':
				cube.start();
				break;
			case 'Q': case 'q': case 0x1b: case -1:
				loop = FALSE;
				break;
			}
		}
	return 0;
	}
/*...e*/
