//
// bitmap.C - Bitmap
//

/*...sincludes:0:*/
#ifdef NO_CINCLUDES
  #include <stdio.h>
  #include <ctype.h>
  #include <stdlib.h>
  #include <stddef.h>
  #include <stdarg.h>
  #include <string.h>
  #include <iostream.h>
#else
  #include <cstdio>
  #include <cctype>
  #include <cstdlib>
  #include <cstddef>
  #include <cstdarg>
  #include <cstring>
  #include <iostream>
#endif

#ifndef NO_STDNS
  using namespace std;
#endif

extern "C" {
#if defined(AIX) || defined(LINUX) || defined(SUN) || defined(MACOSX) || defined(IPHONE)
  #include <unistd.h>
#else
  #include <io.h>
#endif
#include <fcntl.h>
#include <sys/types.h>
#include <sys/stat.h>
#ifndef O_BINARY
  #define O_BINARY 0
#endif
}

#include "insane.h"
#include "font.h"
#include "bitmap.h"

/*...vinsane\46\h:0:*/
/*...vfont\46\h:0:*/
/*...vbitmap\46\h:0:*/
/*...e*/

/*...sGbmSupport:0:*/
// Ensure gbm initialised before program runs
// and is terminated after program terminates.

class GbmSupport
	{
public:
	GbmSupport() { gbm_init(); }
	~GbmSupport() { gbm_deinit(); }
	};

static GbmSupport gbmsupport;
/*...e*/
/*...sPrimBitmap:0:*/
PrimBitmap * PrimBitmap::head = 0;

/*...sPrimBitmapListWithout:0:*/
PrimBitmap *PrimBitmapListWithout(PrimBitmap *pbl, PrimBitmap *pb)
	{
	if ( pbl == pb )
		return pbl->next;
	pbl->next = PrimBitmapListWithout(pbl->next, pb);
	return pbl;
	}
/*...e*/

/*...sPrimBitmap \40\blank\41\:0:*/
PrimBitmap::PrimBitmap(int w, int h, int bpp, Byte *data)
	:
	n(1)
	{
	gbm.w   = w;
	gbm.h   = h;
	gbm.bpp = bpp;
	stride  = ((w * bpp + 31)/32)*4;

	if ( data != 0 )
		{
		PrimBitmap::data = data;
		data_free = 0;
		}
	else
		PrimBitmap::data = data_free = new Byte[stride * h];

	// Initialse surface to black
	memset(PrimBitmap::data, 0, stride * h);

	// Initialise palette to black also
	if ( bpp != 24 )
		memset(gbmrgb, 0, sizeof(gbmrgb));

	// Keep this bitmap in the list
	fn_origin[0] = '\0';
	next = head; head = this;
	}
/*...e*/
/*...sPrimBitmap \40\file\41\:0:*/
PrimBitmap::PrimBitmap(const char *fn)
	:
	n(1)
	{
	char *opt;
	int fd, ft;

	strcpy(fn_origin, fn);

	if ( (opt = strchr(fn_origin, ',')) != NULL )
		*opt++ = '\0';
	else
		opt = (char *) "";

	if ( gbm_guess_filetype(fn_origin, &ft) != GBM_ERR_OK )
		{ cerr << "can't guess type of " << fn << endl; exit(1); }

	if ( (fd = gbm_io_open(fn_origin, O_RDONLY | O_BINARY)) == -1 )
		{ cerr << "can't open " << fn << endl; exit(1); }

	if ( gbm_read_header(fn, fd, ft, &gbm, opt) != GBM_ERR_OK )
		{ cerr << "can't read header of " << fn << endl; exit(1); }

	if ( gbm_read_palette(fd, ft, &gbm, gbmrgb) != GBM_ERR_OK )
		{ cerr << "can't read palette of " << fn << endl; exit(1); }

	stride = ((gbm.w * gbm.bpp + 31)/32)*4;
	if ( (data = data_free = new Byte[stride * gbm.h]) == 0 )
		{ cerr << "out of memory loading bitmap file " << fn << endl; exit(1); }

	if ( gbm_read_data(fd, ft, &gbm, data) )
		{ cerr << "can't read data of " << fn << endl; exit(1); }

	gbm_io_close(fd);

	// Keep this bitmap in the list
	next = head; head = this;
	}
/*...e*/
/*...s\126\PrimBitmap:0:*/
PrimBitmap::~PrimBitmap()
	{
	// Remove this bitmap from the list
	head = PrimBitmapListWithout(head, this);

	if ( data_free != 0 )
		delete[] data_free;
	}
/*...e*/
/*...swrite:0:*/
Boolean PrimBitmap::write(const char *fn) const
	{
	char *opt;
	int fd, ft;
	GBMFT gbmft;
	char fn2[200+1];

	strcpy(fn2, fn);
	if ( (opt = strchr(fn2, ',')) != NULL )
		*opt++ = '\0';
	else
		opt = (char *) "";

	if ( gbm_guess_filetype(fn2, &ft) != GBM_ERR_OK )
		return FALSE;

	gbm_query_filetype(ft, &gbmft);

	int wf;
	switch ( gbm.bpp )
		{
		case 1:		wf = GBM_FT_W1;		break;
		case 4:		wf = GBM_FT_W4;		break;
		case 8:		wf = GBM_FT_W8;		break;
		case 24:	wf = GBM_FT_W24;	break;
		}

	if ( (gbmft.flags & wf) == 0 )
		return FALSE;

	if ( (fd = gbm_io_create(fn2, O_WRONLY|O_BINARY)) == -1 )
		return FALSE;

	if ( gbm_write(fn2, fd, ft, &gbm, gbmrgb, data, opt) != GBM_ERR_OK )
		{
		gbm_io_close(fd);
		remove(fn2);
		return FALSE;
		}

	gbm_io_close(fd);

	return TRUE;
	}
/*...e*/
/*...sget_pixel:0:*/
Rgb PrimBitmap::get_pixel(int x, int y) const
	{
	Byte inx, r, g, b, *line = data + y * stride;

	switch ( gbm.bpp )
		{
		case 1:
			inx = line[x >> 3];
			inx >>= ( 7 - (x & 7) );
			inx &= 1;
			b = gbmrgb[inx].b;
			g = gbmrgb[inx].g;
			r = gbmrgb[inx].r;
			break;
		case 4:
			inx = line[x >> 1];
			if ( x & 1 )
				inx &= 0x0f;
			else
				inx >>= 4;
			b = gbmrgb[inx].b;
			g = gbmrgb[inx].g;
			r = gbmrgb[inx].r;
			break;
		case 8:
			inx = line[x];
			b = gbmrgb[inx].b;
			g = gbmrgb[inx].g;
			r = gbmrgb[inx].r;
			break;
		case 24:
			line += (x * 3);
			b = *line++;
			g = *line++;
			r = *line;
			break;
		}
	return Rgb((double) r/255.0, (double) g/255.0, (double) b/255.0);
	}
/*...e*/
/*...sset_pixel \40\rgb\41\:0:*/
// Note: We will only call these methods on a 24bpp bitmap, promise!

void PrimBitmap::set_pixel(int x, int y, const Rgb & rgb)
	{
	Byte *p = data + y * stride + x * 3;
	*p++ = (Byte) (rgb.b * 255.0);
	*p++ = (Byte) (rgb.g * 255.0);
	*p   = (Byte) (rgb.r * 255.0);
	}

void PrimBitmap::set_pixel(int x, int y, int w, int h, const Rgb & rgb)
	{
	Byte *p = data + y * stride + x * 3;
	Byte b = (Byte) (rgb.b * 255.0);
	Byte g = (Byte) (rgb.g * 255.0);
	Byte r = (Byte) (rgb.r * 255.0);
	int step = stride - w*3;
	for ( int j = 0; j < h; j++, p += step )
		for ( int i = 0; i < w; i++ )
			{ *p++ = b; *p++ = g; *p++ = r; }
	}
/*...e*/
/*...sset_pixel \40\index\41\:0:*/
// Note: We will only call this method on a non 24bpp bitmap, promise!

void PrimBitmap::set_pixel(int x, int y, int index)
	{
	Byte *p = data + y * stride;

	switch ( gbm.bpp )
		{
		case 1:
			{
			p += (x >> 3);
			Byte bit = (0x80 >> (x & 7));
			if ( index )
				*p |= bit;
			else
				*p &= ~bit;
			}
			break;
		case 4:
			p += (x >> 1);
			if ( x & 1 )
				*p = ((*p & 0xf0) | index);
			else
				*p = ((*p & 0x0f) | (index << 4));
			break;
		case 8:
			p[x] = index;
			break;
		}
	}
/*...e*/
/*...sset_pal:0:*/
void PrimBitmap::set_pal(int index, const Rgb & rgb)
	{
	gbmrgb[index].b = (Byte) (rgb.b * 255.0);
	gbmrgb[index].g = (Byte) (rgb.g * 255.0);
	gbmrgb[index].r = (Byte) (rgb.r * 255.0);
	}
/*...e*/
/*...splot:0:*/
void PrimBitmap::plot(int x, int y, int index)
	{
/*...ssafety:8:*/
#ifdef SAFETY
if ( x < 0 || x >= gbm.w ||
     y < 0 || y >= gbm.h )
	insane("unclipped coord passed to PrimBitmap::plot");
#endif
/*...e*/
	data[y*stride+x] = (Byte) index;
	}
/*...e*/
/*...splot_clipped:0:*/
void PrimBitmap::plot_clipped(int x, int y, int index)
	{
	if ( x >= 0 && x < gbm.w &&
	     y >= 0 && y < gbm.h )
		data[y*stride+x] = (Byte) index;
	}
/*...e*/
/*...srect:0:*/
void PrimBitmap::rect(int x1, int y1, int x2, int y2, int index)
	{
/*...ssafety:8:*/
#ifdef SAFETY
if ( x1 < 0 || x1 > gbm.w ||
     y1 < 0 || y1 > gbm.h ||
     x2 < 0 || x2 > gbm.w ||
     y2 < 0 || y2 > gbm.h )
	insane("unclipped coord(s) passed to PrimBitmap::rect");
#endif
/*...e*/
	Byte *p = data+(y1*stride)+x1;
	for ( ; y1 < y2; y1++, p += stride )
		memset(p, (Byte) index, x2-x1);
	}
/*...e*/
/*...srect_clipped:0:*/
void PrimBitmap::rect_clipped(int x1, int y1, int x2, int y2, int index)
	{
	if ( x1 < gbm.w && x2 > 0 &&
	     y1 < gbm.h && y2 > 0 )
		{
		if ( x1 < 0 ) x1 = 0;
		if ( y1 < 0 ) y1 = 0;
		if ( x2 > gbm.w ) x2 = gbm.w;
		if ( y2 > gbm.h ) y2 = gbm.h;
		rect(x1, y1, x2, y2, index);
		}
	}
/*...e*/
/*...sline:0:*/
// Use the traditional Bresenham line drawing algorithm.
// This code has to go fast, so try to optimise it a bit.
// There are 8 octants the line can be travelling in.
// So use a quick tree style of nested if decode to find which one.
// Then use generic Bresenham stepper.

/*...sbres:0:*/
// Do the bresenham line stepping algorithm code across a range of i values,
// with occasional steps in the j direction too.

inline void bres(
	int di, int dj,
	Byte *ptr, int ptradd_i, int ptradd_ij,
	int index
	)
	{
	*ptr = (Byte) index;
	int v   = 2 * dj - di;
	int vi  = 2 * dj;
	int vij = 2 * (dj - di);
	while ( di-- > 0 )
		{
		if ( v <= 0 )
			{ v += vi ; ptr += ptradd_i ; }
		else
			{ v += vij; ptr += ptradd_ij; }
		*ptr = (Byte) index;
		}
	}
/*...e*/

void PrimBitmap::line(
	int x1, int y1, int x2, int y2,
	int index
	)
	{
/*...ssafety:8:*/
#ifdef SAFETY
if ( x1 < 0 || x1 >= gbm.w ||
     y1 < 0 || y1 >= gbm.h ||
     x2 < 0 || x2 >= gbm.w ||
     y2 < 0 || y2 >= gbm.h )
	insane("unclipped coords passed to PrimBitmap::line");
#endif
/*...e*/
	Byte *ptr = data + ((y1*stride)+x1);
	if ( x1 < x2 )
		if ( y1 < y2 )
			if ( y2-y1 <= x2-x1 )
				// l->r, b->t, shallow
				bres(x2-x1, y2-y1, ptr, 1, stride+1, index);
			else
				// l->r, b->t, steep
				bres(y2-y1, x2-x1, ptr, stride, stride+1, index);
		else
			if ( y1-y2 <= x2-x1 )
				// l->r, t->b, shallow
				bres(x2-x1, y1-y2, ptr, 1, -stride+1, index);
			else
				// l->r, t->b, steep
				bres(y1-y2, x2-x1, ptr, -stride, -stride+1, index);
	else
		if ( y1 < y2 )
			if ( y2-y1 <= x1-x2 )
				// r->l, b->t, shallow
				bres(x1-x2, y2-y1, ptr, -1, stride-1, index);
			else
				// r->l, b->t, steep
				bres(y2-y1, x1-x2, ptr, stride, stride-1, index);
		else
			if ( y1-y2 <= x1-x2 )
				// r->l, t->b, shallow
				bres(x1-x2, y1-y2, ptr, -1, -stride-1, index);
			else
				// r->l, t->b, steep
				bres(y1-y2, x1-x2, ptr, -stride, -stride-1, index);
	}
/*...e*/
/*...sline_clipped:0:*/
// Use the traditional Cohen Sutherland

#define	ABOVE 0x01
#define BELOW 0x02
#define	LEFT  0x04
#define	RIGHT 0x08

/*...sclassify:0:*/
inline unsigned classify(double x, double y, int w, int h)
	{
	unsigned c = 0;
	if ( x < 0.0 ) c |= LEFT ; else if ( x > (double) w ) c |= RIGHT;
	if ( y < 0.0 ) c |= BELOW; else if ( y > (double) h ) c |= ABOVE;
	return c;
	}
/*...e*/

void PrimBitmap::line_clipped(
	double x1, double y1,
	double x2, double y2,
	int index
	)
	{
	int w = gbm.w - 1;
	int h = gbm.h - 1;

	unsigned c1 = classify(x1, y1, w, h);
	unsigned c2 = classify(x2, y2, w, h);
	if ( c1 & c2 )
		return;
	while ( c1 | c2 )
		if ( c1 )
			{
			if ( c1&LEFT )
				{ y1 -= x1*(y2-y1)/(x2-x1); x1 = 0.0; }
			else if ( c1&RIGHT )
				{ y1 -= (x1-w)*(y2-y1)/(x2-x1); x1 = w; }
			else if ( c1&BELOW )
				{ x1 -= y1*(x2-x1)/(y2-y1); y1 = 0.0; }
			else
				{ x1 -= (y1-w)*(x2-x1)/(y2-y1); y1 = w; }
			c1 = classify(x1, y1, w, h);
			}
		else
			{
			if ( c2&LEFT )
				{ y2 -= x2*(y2-y1)/(x2-x1); x2 = 0.0; }
			else if ( c2&RIGHT )
				{ y2 -= (x2-w)*(y2-y1)/(x2-x1); x2 = w; }
			else if ( c2&BELOW )
				{ x2 -= y2*(x2-x1)/(y2-y1); y2 = 0.0; }
			else
				{ x2 -= (y2-w)*(x2-x1)/(y2-y1); y2 = w; }
			c2 = classify(x2, y2, w, h);
			}
	// Now clipped to bitmap dimensions
	line((int)x1,(int)y1, (int)x2,(int)y2, index);
	}
/*...e*/
/*...spolygon:0:*/
// Draw a polygon, given the vertex coordinates and a colour.
// Polygon is convex.
// Polygon is already clipped to bitmap size.
// This code will explicitly fail to draw a 0 vertex polygon.
// A 1 vertex polygon ends up being drawn as a single pixel.
// A 2 vertex polygon ends up as a line from A to B and one from B to A.

/*...sclass PolyContext:0:*/
// This is used to accumulate the area of the screen enclosed by the polygon,
// over the whole bitmap.

/*...sclass Span:0:*/
// This is used to accumulate the area of the screen enclosed by the polygon,
// over a single scan line of the image.

class Span
	{
public:
	int minx, maxx;
	void init() { minx = -1; }
	void x(int x)
		{
		if ( minx == -1 )
			minx = maxx = x;
		else if ( x < minx )
			minx = x;
		else if ( x > maxx )
			maxx = x;
		}
	};

// Before invoking the x method on a span, you need to have init()ed it.
// We shall plan to do this just for those spans on the bitmap we use.
/*...e*/

class PolyContext
	{
public:
	int miny, maxy;
	Span *spans;
	PolyContext(int ix, int iy, int h)
		:
		miny(iy), maxy(iy)
		{
		spans = new Span[h];
		spans[iy].init();
		spans[iy].x(ix);
		}
	~PolyContext()
		{
		delete[] spans;
		}
	void line(int x1, int y1, int x2, int y2);
	};

/*...sbres_span_x \45\ shallow variant:0:*/
inline void bres_span_x(
	int dx, int dy,
	Span *span, int x, 
	int addx, int addy
	)
	{
	span->x(x);
	int v   = 2 * dy - dx;
	int vx  = 2 * dy;
	int vxy = 2 * (dy - dx);
	while ( dx-- > 0 )
		{
		if ( v <= 0 )
			{ v += vx ; x += addx; }
		else
			{ v += vxy; x += addx; span += addy; }
		span->x(x);
		}
	}
/*...e*/
/*...sbres_span_y \45\ steep variant:0:*/
inline void bres_span_y(
	int dx, int dy,
	Span *span, int x, 
	int addx, int addy
	)
	{
	span->x(x);
	int v   = 2 * dx - dy;
	int vy  = 2 * dx;
	int vxy = 2 * (dx - dy);
	while ( dy-- > 0 )
		{
		if ( v <= 0 )
			{ v += vy ;            span += addy; }
		else
			{ v += vxy; x += addx; span += addy; }
		span->x(x);
		}
	}
/*...e*/

void PolyContext::line(int x1, int y1, int x2, int y2)
	{
	// Ensure enough spans are initialised
	while ( miny > y1 ) spans[--miny].init();
	while ( miny > y2 ) spans[--miny].init();
	while ( maxy < y1 ) spans[++maxy].init();
	while ( maxy < y2 ) spans[++maxy].init();
	// Do 8-way octant split
	Span *span = &(spans[y1]);
	if ( x1 < x2 )
		if ( y1 < y2 )
			if ( y2-y1 <= x2-x1 )
				// l->r, b->t, shallow
				bres_span_x(x2-x1, y2-y1, span, x1, 1, 1);
			else
				// l->r, b->t, steep
				bres_span_y(x2-x1, y2-y1, span, x1, 1, 1);
		else
			if ( y1-y2 <= x2-x1 )
				// l->r, t->b, shallow
				bres_span_x(x2-x1, y1-y2, span, x1, 1, -1);
			else
				// l->r, t->b, steep
				bres_span_y(x2-x1, y1-y2, span, x1, 1, -1);
	else
		if ( y1 < y2 )
			if ( y2-y1 <= x1-x2 )
				// r->l, b->t, shallow
				bres_span_x(x1-x2, y2-y1, span, x1, -1, 1);
			else
				// r->l, b->t, steep
				bres_span_y(x1-x2, y2-y1, span, x1, -1, 1);
		else
			if ( y1-y2 <= x1-x2 )
				// r->l, t->b, shallow
				bres_span_x(x1-x2, y1-y2, span, x1, -1, -1);
			else
				// r->l, t->b, steep
				bres_span_y(x1-x2, y1-y2, span, x1, -1, -1);
	}
/*...e*/

void PrimBitmap::polygon(const int xs[], const int ys[], int n, int index)
	{
	if ( n == 0 )
		return;
/*...ssafety:8:*/
#ifdef SAFETY
for ( int j = 0; j < n; j++ )
	if ( xs[j] < 0 || xs[j] >= gbm.w ||
	     ys[j] < 0 || ys[j] >= gbm.h )
		insane("unclipped coords passed to PrimBitmap::polygon");
#endif
/*...e*/
	PolyContext pc(xs[n-1], ys[n-1], gbm.h);
	pc.line(xs[n-1], ys[n-1], xs[0], ys[0]);
	for ( int i = 0; i < n-1; i++ )
		pc.line(xs[i], ys[i], xs[i+1], ys[i+1]);
	for ( int y = pc.miny; y <= pc.maxy; y++ )
		{
		Span *s = &(pc.spans[y]);
		memset(data+y*stride+s->minx, (Byte) index, s->maxx - s->minx + 1);
		}
	}
/*...e*/
/*...spolygon_clipped:0:*/
// Draw convex polygon, clipping against bitmap size.
// When clipping to an edge, the number of vertexes can only go up by 1
// So we only need space for n+4 elements in our temporary arrays.

/*...stest_x_ge\44\ test_y_ge\44\ clip_x:0:*/
static Boolean test_x_ge(double x, double y, double v) { y=y; return x >= v; }
static Boolean test_x_le(double x, double y, double v) { y=y; return x <= v; }

static void clip_x(
	double   x1, double   y1,
	double   x2, double   y2,
	double & x , double & y ,
	double   v
	)
	{
	x = v;
	y = y1 + (x-x1)/(x2-x1) * (y2-y1);
	}
/*...e*/
/*...stest_y_ge\44\ test_y_le\44\ clip_y:0:*/
static Boolean test_y_ge(double x, double y, double v) { x=x; return y >= v; }
static Boolean test_y_le(double x, double y, double v) { x=x; return y <= v; }

static void clip_y(
	double   x1, double   y1,
	double   x2, double   y2,
	double & x , double & y ,
	double   v
	)
	{
	y = v;
	x = x1 + (y-y1)/(y2-y1) * (x2-x1);
	}
/*...e*/
/*...sedge_clip:0:*/
static void edge_clip(
	const double xs [], const double ys [], int   n ,
	      double xs2[],       double ys2[], int & n2,
	Boolean (*test)(double x, double y, double value),
	void    (*clip)(double   x1, double   y1,
			double   x2, double   y2,
			double & x , double & y ,
			double value),
	double value
	)
	{
	n2 = 0;
	if ( n == 0 )
		return;
	int i, prev_i = n-1;
	Boolean in, prev_in = test(xs[prev_i], ys[prev_i], value);
	for ( i	= 0; i < n; prev_in = in, prev_i = i++ )
		// Consider the line segment from vertex prev_i to i
		{
		in = test(xs[i], ys[i], value);
		if ( prev_in )
			{
			xs2[n2] = xs[prev_i];
			ys2[n2] = ys[prev_i];
			n2++;
			if ( !in )
				{
				clip(xs[prev_i], ys[prev_i], xs[i], ys[i],
				     xs2[n2], ys2[n2], value);
				n2++;
				}
			}
		else
			{
			if ( in )
				{
				clip(xs[prev_i], ys[prev_i], xs[i], ys[i],
				     xs2[n2], ys2[n2], value);
				n2++;
				}
			}	
		}
	}
/*...e*/

void PrimBitmap::polygon_clipped(
	const double xs[], const double ys[], int n, int index
	)
	{
	double *xs2  = new double[n+4];
	double *ys2  = new double[n+4];
	double *xs3  = new double[n+4];
	double *ys3  = new double[n+4];
	int    *xs3i = new int[n+4];
	int    *ys3i = new int[n+4];
	int n2, n3;
	edge_clip(xs , ys , n , xs2, ys2, n2, test_x_ge, clip_x, 0);
	edge_clip(xs2, ys2, n2, xs3, ys3, n3, test_x_le, clip_x, gbm.w-1);
	edge_clip(xs3, ys3, n3, xs2, ys2, n2, test_y_ge, clip_y, 0);
	edge_clip(xs2, ys2, n2, xs3, ys3, n3, test_y_le, clip_y, gbm.h-1);
	for ( int i = 0; i < n3; i++ )
		{
		xs3i[i] = (int) xs3[i];
		ys3i[i] = (int) ys3[i];
		}
	polygon(xs3i, ys3i, n3, index);
	delete[] ys3i;
	delete[] xs3i;
	delete[] ys3;
	delete[] xs3;
	delete[] ys2;
	delete[] xs2;
	}
/*...e*/
/*...sprint:0:*/
void PrimBitmap::print(int x, int y, char ch, int index, int index_bg)
	{
	for ( int j = 0; j < 8; j++ )
		{
		Byte *p = data+(y+j)*stride+x;
		for ( int i = 0; i < 8; i++ )
			if ( mr_font[ch-' '][7-j][i] )
				*p++ = index;
			else if ( index_bg != -1 )
				*p++ = index_bg;
			else
				p++;
		}
	}

void PrimBitmap::print(int x, int y, const char *str, int index, int index_bg)
	{
	for ( ; *str != '\0'; str++, x += 8 )
		print(x, y, *str, index, index_bg);
	}
/*...e*/
/*...sprint_clipped:0:*/
void PrimBitmap::print_clipped(int x, int y, char ch, int index, int index_bg)
	{
	for ( int j = 0; j < 8; j++ )
		for ( int i = 0; i < 8; i++ )
			if ( mr_font[ch-' '][7-j][i] )
				plot_clipped(x+i, y+j, index);
			else if ( index_bg != -1 )
				plot_clipped(x+i, y+j, index_bg);
	}

void PrimBitmap::print_clipped(int x, int y, const char *str, int index, int index_bg)
	{
	for ( ; *str != '\0'; str++, x += 8 )
		print_clipped(x, y, *str, index, index_bg);
	}
/*...e*/
/*...sclear:0:*/
void PrimBitmap::clear(int index)
	{
	memset(data, (Byte) index, stride * gbm.h);
	}
/*...e*/
/*...e*/
/*...sBitmap:0:*/
Bitmap::Bitmap(PrimBitmap *p) : p(p) {}

Bitmap::Bitmap() { p = new PrimBitmap(1,1,24); }

Bitmap::Bitmap(int w, int h, int bpp, Byte *data)
	{ p = new PrimBitmap(w, h, bpp, data); }

Bitmap::Bitmap(const char *fn, Boolean check_cache)
	{
	if ( check_cache )
		for ( PrimBitmap *pb = PrimBitmap::head; pb != 0; pb = pb->next )
#ifdef UNIX
			// UNIX filenames are case sensitive
			if ( !strcmp(fn, pb->fn_origin) )
#elif defined(WIN32)
			// Windows filenames are case insensitive
			if ( !_strcmpi(fn, pb->fn_origin) )
#else
			// OS/2 filenames are case insensitive
			if ( !strcmpi(fn, pb->fn_origin) )
#endif
				{
				p = pb;
				++(p->n);
				return;
				}

	p = new PrimBitmap(fn);
	}

Bitmap::Bitmap(const Bitmap & b) { p = b.p; (p->n)++; }

Bitmap & Bitmap::operator=(const Bitmap & b)
	{
	++(b.p->n);
	if ( --(p->n) == 0 )
		delete p;
	p = b.p;
	return *this;
	}

Bitmap::~Bitmap()
	{
	if ( --(p->n) == 0 )
		delete p;
	}

// This last entrypoint is to allow users of the Bitmap class to get
// a the data it contains, for looking at only.

void Bitmap::get_gbm_internals(
	GBM    *(&gbm),
	GBMRGB *(&gbmrgb),
	Byte   *(&data)
	) const
	{
	gbm    = &(p->gbm);
	gbmrgb = p->gbmrgb;
	data   = p->data;
	}
/*...e*/
