//
// bitmap.h - Bitmap
//
// I go to special lengths to handle the fact that bitmaps may consume large
// amounts of memory (often megabytes).
//
// Each PrimBitmap includes a reference count. Therefore, passing Bitmaps by
// value does not cause multiple copies of the bitmap data to exist.
//
// All PrimBitmaps are kept in a list, and remember the filename they
// originated from. Then it becomes possible to detect that we are loading a
// bitmap that we have previously loaded, and simply re-reference it.
// This 'check_cache' feature may be disabled. eg:
//
//	Bitmap b1("backdrop.gif");		// Loads bitmap
//	Bitmap b2("backdrop.gif");		// Points to b1 internally
//	Bitmap b3("backdrop.gif", FALSE);	// Does load another copy
//

#ifndef BITMAP_H
#define	BITMAP_H

/*...sincludes:0:*/
extern "C" {
#include "gbm.h"
}

#include "types.h"
#include "rgb.h"

/*...vtypes\46\h:0:*/
/*...vrgb\46\h:0:*/
/*...e*/

/*...sprivate PrimBitmap:0:*/
class PrimBitmap
	{
	void print(int x, int y, char ch, int index, int index_bg);
	void print_clipped(int x, int y, char ch, int index, int index_bg);
public:
	GBM gbm;
	GBMRGB gbmrgb[0x100];
	Byte *data;
	Byte *data_free;
	int stride;

	static PrimBitmap *head;	// Head of list of PrimBitmaps
	PrimBitmap *next;		// Next PrimBitmap in list
	char fn_origin[500+1];		// Filename constructed from or ""
	int n;				// Reference count (initially 1)

	PrimBitmap(int w, int h, int bpp = 24, Byte *data = 0);
	PrimBitmap(const char *fn);
	~PrimBitmap();
	Boolean write(const char *fn) const;
	int width () const { return gbm.w; }
	int height() const { return gbm.h; }
	int bpp   () const { return gbm.bpp; }
	Rgb  get_pixel(int x, int y) const;
	void set_pixel(int x, int y, const Rgb & rgb);
	void set_pixel(int x, int y, int w, int h, const Rgb & rgb);
	void set_pixel(int x, int y, int index);
	void set_pal(int index, const Rgb & rgb);
	void plot(int x, int y, int index);
	void plot_clipped(int x, int y, int index);
	void line(int x1, int y1, int x2, int y2, int index);
	void line_clipped(
		double x1, double y1, double x2, double y2, int index
		);
	void rect(int x1, int y1, int x2, int y2, int index);
	void rect_clipped(int x1, int y1, int x2, int y2, int index);
	void polygon(const int xs[], const int ys[], int n, int index);
	void polygon_clipped(
		const double xs[], const double ys[], int n, int index
		);
	void print(int x, int y, const char *str, int index, int index_bg);
	void print_clipped(int x, int y, const char *str, int index, int index_bg);
	void clear(int index = 0);
	};
/*...e*/

class Bitmap
	{
/*...sprivate:8:*/
PrimBitmap *p;
Bitmap(PrimBitmap *p);
/*...e*/
public:
	Bitmap();
	Bitmap(int w, int h, int bpp = 24, Byte *data = 0);
	Bitmap(const char *fn, Boolean check_cache = TRUE);
	Bitmap(const Bitmap & b);
	Bitmap & operator=(const Bitmap & b);
	~Bitmap();
	Boolean write(const char *fn) const { return p->write(fn); }
	int width () const { return p->width (); }
	int height() const { return p->height(); }
	int bpp   () const { return p->bpp   (); }
	Rgb get_pixel(int x, int y) const
		{ return p->get_pixel(x,y); }
	void set_pixel(int x, int y, const Rgb & rgb)	// Use if bpp==24
		{ p->set_pixel(x,y,rgb); }
	void set_pixel(int x, int y, int w, int h, const Rgb & rgb)
		{ p->set_pixel(x,y,w,h,rgb); }
	void set_pixel(int x, int y, int index)		// Use if bpp!=24
		{ p->set_pixel(x,y,index); }
	void set_pal(int index, const Rgb & rgb)	// Use if bpp!=24
		{ p->set_pal(index, rgb); }

	// This next group of methods are only for when bpp==8
	void plot(int x, int y, int index)
		{ p->plot(x, y, index); }
	void plot_clipped(int x, int y, int index)
		{ p->plot_clipped(x, y, index); }
	void line(int x1, int y1, int x2, int y2, int index)
		{ p->line(x1, y1, x2, y2, index); }
	void line_clipped(
		double x1, double y1, double x2, double y2, int index
		)
		{ p->line_clipped(x1, y1, x2, y2, index); }
	void rect(int x1, int y1, int x2, int y2, int index)
		{ p->rect(x1, y1, x2, y2, index); }
	void rect_clipped(int x1, int y1, int x2, int y2, int index)
		{ p->rect_clipped(x1, y1, x2, y2, index); }
	void polygon(const int xs[], const int ys[], int n, int index)
		{ p->polygon(xs, ys, n, index); }
	void polygon_clipped(
		const double xs[], const double ys[], int n, int index
		)
		{ p->polygon_clipped(xs, ys, n, index); }
	void print(
		int x, int y, const char *str, int index, int index_bg = -1
		)
		{ p->print(x, y, str, index, index_bg); }
	void print_clipped(
		int x, int y, const char *str, int index, int index_bg = -1
		)
		{ p->print_clipped(x, y, str, index, index_bg); }
	void clear(int index = 0)
		{ p->clear(index); }

	// For other tools
	void get_gbm_internals(
		GBM    *(&gbm),
		GBMRGB *(&gbmrgb),
		Byte   *(&data)
		) const;
	};

#endif
