//
// col.C - Colours
//

/*...sincludes:0:*/
#include "matrix.h"
#include "col.h"

/*...vmatrix\46\h:0:*/
/*...vcol\46\h:0:*/
/*...e*/

/*...spos_mod:0:*/
inline int pos_mod(int num, int denom)
	{
	if ( num >= 0 )
		return num % denom;
	else
		{
		int mod = num % denom;
		if ( mod < 0 )
			mod += denom;
		return mod;
		}
	}
/*...e*/
/*...spos_fmod:0:*/
inline double pos_fmod(double num, double denom)
	{
	if ( num >= 0.0 )
		return fmod(num, denom);
	else
		{
		double mod = fmod(num, denom);
		if ( mod < 0.0 )
			mod += denom;
		return mod;
		}
	}
/*...e*/
/*...sinterp_rgb:0:*/
inline Rgb interp_rgb(const Rgb & rgb0, const Rgb & rgb1, double f)
	{ return rgb0 * (1.0-f) + rgb1 * f; }
/*...e*/

/*...sPrimColRgb:0:*/
class PrimColRgb : public PrimCol
	{
	Rgb rgb;
public:
	PrimColRgb(Rgb rgb) : rgb(rgb) {}
	virtual ~PrimColRgb() {}
	virtual PrimCol *copy() const
		{ return new PrimColRgb(rgb); }
	virtual PrimCol *trans  (Xyz    t) const
		{ t=t; return copy(); }
	virtual PrimCol *trans_x(double t) const
		{ t=t; return copy(); }
	virtual PrimCol *trans_y(double t) const
		{ t=t; return copy(); }
	virtual PrimCol *trans_z(double t) const
		{ t=t; return copy(); }
	virtual PrimCol *rot_x(double angle) const
		{ angle=angle; return copy(); }
	virtual PrimCol *rot_y(double angle) const
		{ angle=angle; return copy(); }
	virtual PrimCol *rot_z(double angle) const
		{ angle=angle; return copy(); }
	virtual PrimCol *scale  (Xyz    factor) const
		{ factor=factor; return copy(); } 
	virtual PrimCol *scale  (double factor) const
		{ factor=factor; return copy(); } 
	virtual PrimCol *scale_x(double factor) const
		{ factor=factor; return copy(); } 
	virtual PrimCol *scale_y(double factor) const
		{ factor=factor; return copy(); } 
	virtual PrimCol *scale_z(double factor) const
		{ factor=factor; return copy(); }
	virtual Rgb evaluate(double p0, double p1, double p2) const
		{ p0=p0; p1=p1; p2=p2; return rgb; }
	};
/*...e*/
/*...sPrimColRgbBlend:0:*/
class PrimColRgbBlend : public PrimCol
	{
	Rgb rgb0, rgb1;
public:
	PrimColRgbBlend(Rgb rgb0, Rgb rgb1) : rgb0(rgb0), rgb1(rgb1) {}
	virtual ~PrimColRgbBlend() {}
	virtual PrimCol *copy() const
		{ return new PrimColRgbBlend(rgb0, rgb1); }
	virtual PrimCol *trans  (Xyz    t) const
		{ t=t; return copy(); }
	virtual PrimCol *trans_x(double t) const
		{ t=t; return copy(); }
	virtual PrimCol *trans_y(double t) const
		{ t=t; return copy(); }
	virtual PrimCol *trans_z(double t) const
		{ t=t; return copy(); }
	virtual PrimCol *rot_x(double angle) const
		{ angle=angle; return copy(); }
	virtual PrimCol *rot_y(double angle) const
		{ angle=angle; return copy(); }
	virtual PrimCol *rot_z(double angle) const
		{ angle=angle; return copy(); }
	virtual PrimCol *scale  (Xyz    factor) const
		{ factor=factor; return copy(); } 
	virtual PrimCol *scale  (double factor) const
		{ factor=factor; return copy(); } 
	virtual PrimCol *scale_x(double factor) const
		{ factor=factor; return copy(); } 
	virtual PrimCol *scale_y(double factor) const
		{ factor=factor; return copy(); } 
	virtual PrimCol *scale_z(double factor) const
		{ factor=factor; return copy(); }
	virtual Rgb evaluate(double p0, double p1, double p2) const
		{
		if ( p0 < 0.0 )
			return rgb0;
		if ( p0 > 1.0 )
			return rgb1;
		else
			{
			double fp0 = floor(p0);
			double fra = p0-fp0;
			return rgb0*(1.0-fra)+rgb1*fra;
			}
		}
	};
/*...e*/
/*...sPrimColNoMove:0:*/
class PrimColNoMove : public PrimCol
	{
	PrimCol *col;
public:
	PrimColNoMove(PrimCol *col) : col(col) {}
	~PrimColNoMove() { delete col; }
	virtual PrimCol *copy() const
		{ return new PrimColNoMove(col->copy()); }
	virtual PrimCol *trans  (Xyz    t) const
		{ t=t; return copy(); }
	virtual PrimCol *trans_x(double t) const
		{ t=t; return copy(); }
	virtual PrimCol *trans_y(double t) const
		{ t=t; return copy(); }
	virtual PrimCol *trans_z(double t) const
		{ t=t; return copy(); }
	virtual PrimCol *rot_x(double angle) const
		{ angle=angle; return copy(); }
	virtual PrimCol *rot_y(double angle) const
		{ angle=angle; return copy(); }
	virtual PrimCol *rot_z(double angle) const
		{ angle=angle; return copy(); }
	virtual PrimCol *scale  (Xyz    factor) const
		{ factor=factor; return copy(); }
	virtual PrimCol *scale  (double factor) const
		{ factor=factor; return copy(); } 
	virtual PrimCol *scale_x(double factor) const
		{ factor=factor; return copy(); } 
	virtual PrimCol *scale_y(double factor) const
		{ factor=factor; return copy(); } 
	virtual PrimCol *scale_z(double factor) const
		{ factor=factor; return copy(); }
	virtual Rgb evaluate(double p0, double p1, double p2) const
		{ return col->evaluate(p0, p1, p2); }
	};
/*...e*/
/*...sPrimColInterp0:0:*/
class PrimColInterp0 : public PrimCol
	{
	PrimCol *col;
public:
	PrimColInterp0(PrimCol *col) : col(col) {}
	~PrimColInterp0() { delete col; }
	virtual PrimCol *copy() const
		{ return new PrimColInterp0(col->copy()); }
	virtual PrimCol *trans  (Xyz t) const
		{ return new PrimColInterp0(col->trans  (t)); }
	virtual PrimCol *trans_x(double t) const
		{ return new PrimColInterp0(col->trans_x(t)); }
	virtual PrimCol *trans_y(double t) const
		{ return new PrimColInterp0(col->trans_y(t)); }
	virtual PrimCol *trans_z(double t) const
		{ return new PrimColInterp0(col->trans_z(t)); }
	virtual PrimCol *rot_x(double angle) const
		{ return new PrimColInterp0(col->rot_x(angle)); }
	virtual PrimCol *rot_y(double angle) const
		{ return new PrimColInterp0(col->rot_y(angle)); }
	virtual PrimCol *rot_z(double angle) const
		{ return new PrimColInterp0(col->rot_z(angle)); }
	virtual PrimCol *scale  (Xyz factor) const
		{ return new PrimColInterp0(col->scale  (factor)); }
	virtual PrimCol *scale  (double factor) const
		{ return new PrimColInterp0(col->scale  (factor)); }
	virtual PrimCol *scale_x(double factor) const
		{ return new PrimColInterp0(col->scale_x(factor)); }
	virtual PrimCol *scale_y(double factor) const
		{ return new PrimColInterp0(col->scale_y(factor)); }
	virtual PrimCol *scale_z(double factor) const
		{ return new PrimColInterp0(col->scale_z(factor)); }
	virtual Rgb evaluate(double p0, double p1, double p2) const;
	};

Rgb PrimColInterp0::evaluate(double p0, double p1, double p2) const
	{
	double fp0 = floor(p0);
//	double fra = pos_fmod(p0, 1.0);
	double fra = p0-fp0;
	Rgb rgb0 = col->evaluate(fp0      , p1, p2);
	Rgb rgb1 = col->evaluate(fp0 + 1.0, p1, p2);
	return rgb0*(1.0-fra)+rgb1*fra;
	}
/*...e*/
/*...sPrimColInterp1:0:*/
class PrimColInterp1 : public PrimCol
	{
	PrimCol *col;
public:
	PrimColInterp1(PrimCol *col) : col(col) {}
	~PrimColInterp1() { delete col; }
	virtual PrimCol *copy() const
		{ return new PrimColInterp1(col->copy()); }
	virtual PrimCol *trans  (Xyz t) const
		{ return new PrimColInterp1(col->trans  (t)); }
	virtual PrimCol *trans_x(double t) const
		{ return new PrimColInterp1(col->trans_x(t)); }
	virtual PrimCol *trans_y(double t) const
		{ return new PrimColInterp1(col->trans_y(t)); }
	virtual PrimCol *trans_z(double t) const
		{ return new PrimColInterp1(col->trans_z(t)); }
	virtual PrimCol *rot_x(double angle) const
		{ return new PrimColInterp1(col->rot_x(angle)); }
	virtual PrimCol *rot_y(double angle) const
		{ return new PrimColInterp1(col->rot_y(angle)); }
	virtual PrimCol *rot_z(double angle) const
		{ return new PrimColInterp1(col->rot_z(angle)); }
	virtual PrimCol *scale  (Xyz factor) const
		{ return new PrimColInterp1(col->scale  (factor)); }
	virtual PrimCol *scale  (double factor) const
		{ return new PrimColInterp1(col->scale  (factor)); }
	virtual PrimCol *scale_x(double factor) const
		{ return new PrimColInterp1(col->scale_x(factor)); }
	virtual PrimCol *scale_y(double factor) const
		{ return new PrimColInterp1(col->scale_y(factor)); }
	virtual PrimCol *scale_z(double factor) const
		{ return new PrimColInterp1(col->scale_z(factor)); }
	virtual Rgb evaluate(double p0, double p1, double p2) const;
	};

Rgb PrimColInterp1::evaluate(double p0, double p1, double p2) const
	{
	p2=p2; // Suppress warning
	double fp1 = floor(p1);
//	double fra = pos_fmod(p1, 1.0);
	double fra = p1-fp1;
	Rgb rgb0 = col->evaluate(p0, fp1      , p1);
	Rgb rgb1 = col->evaluate(p0, fp1 + 1.0, p1);
	return rgb0*(1.0-fra)+rgb1*fra;
	}
/*...e*/
/*...sPrimColInterp2:0:*/
class PrimColInterp2 : public PrimCol
	{
	PrimCol *col;
public:
	PrimColInterp2(PrimCol *col) : col(col) {}
	~PrimColInterp2() { delete col; }
	virtual PrimCol *copy() const
		{ return new PrimColInterp2(col->copy()); }
	virtual PrimCol *trans  (Xyz t) const
		{ return new PrimColInterp2(col->trans  (t)); }
	virtual PrimCol *trans_x(double t) const
		{ return new PrimColInterp2(col->trans_x(t)); }
	virtual PrimCol *trans_y(double t) const
		{ return new PrimColInterp2(col->trans_y(t)); }
	virtual PrimCol *trans_z(double t) const
		{ return new PrimColInterp2(col->trans_z(t)); }
	virtual PrimCol *rot_x(double angle) const
		{ return new PrimColInterp2(col->rot_x(angle)); }
	virtual PrimCol *rot_y(double angle) const
		{ return new PrimColInterp2(col->rot_y(angle)); }
	virtual PrimCol *rot_z(double angle) const
		{ return new PrimColInterp2(col->rot_z(angle)); }
	virtual PrimCol *scale  (Xyz factor) const
		{ return new PrimColInterp2(col->scale  (factor)); }
	virtual PrimCol *scale  (double factor) const
		{ return new PrimColInterp2(col->scale  (factor)); }
	virtual PrimCol *scale_x(double factor) const
		{ return new PrimColInterp2(col->scale_x(factor)); }
	virtual PrimCol *scale_y(double factor) const
		{ return new PrimColInterp2(col->scale_y(factor)); }
	virtual PrimCol *scale_z(double factor) const
		{ return new PrimColInterp2(col->scale_z(factor)); }
	virtual Rgb evaluate(double p0, double p1, double p2) const;
	};

Rgb PrimColInterp2::evaluate(double p0, double p1, double p2) const
	{
	double fp2 = floor(p2);
//	double fra = pos_fmod(p2, 1.0);
	double fra = p2-fp2;
	Rgb rgb0 = col->evaluate(p0, p1, fp2      );
	Rgb rgb1 = col->evaluate(p0, p1, fp2 + 1.0);
	return rgb0*(1.0-fra)+rgb1*fra;
	}
/*...e*/
/*...sPrimColField2d:0:*/
class PrimColField2d : public PrimCol
	{
protected:
	double bx, by;
	Bitmap bitmap;
public:
	PrimColField2d(double bx, double by, Bitmap bitmap) :
		bx(bx), by(by), bitmap(bitmap) {}
	~PrimColField2d() {}
	virtual PrimCol *copy() const
		{ return new PrimColField2d(bx, by, bitmap); }
	virtual PrimCol *trans  (Xyz t) const
		{ t=t; return copy(); }
	virtual PrimCol *trans_x(double t) const
		{ t=t; return copy(); }
	virtual PrimCol *trans_y(double t) const
		{ t=t; return copy(); }
	virtual PrimCol *trans_z(double t) const
		{ t=t; return copy(); }
	virtual PrimCol *rot_x(double angle) const
		{ angle=angle; return copy(); }
	virtual PrimCol *rot_y(double angle) const
		{ angle=angle; return copy(); }
	virtual PrimCol *rot_z(double angle) const
		{ angle=angle; return copy(); }
	virtual PrimCol *scale  (Xyz    factor) const
		{ factor=factor; return copy(); } 
	virtual PrimCol *scale  (double factor) const
		{ factor=factor; return copy(); } 
	virtual PrimCol *scale_x(double factor) const
		{ factor=factor; return copy(); } 
	virtual PrimCol *scale_y(double factor) const
		{ factor=factor; return copy(); } 
	virtual PrimCol *scale_z(double factor) const
		{ factor=factor; return copy(); }
	virtual Rgb evaluate(double p0, double p1, double p2) const;
	};

Rgb PrimColField2d::evaluate(double p0, double p1, double p2) const
	{
	p2=p2; // Suppress warning
	return bitmap.get_pixel(
		(int) pos_fmod(p0+bx, (double) bitmap.width ()),
		(int) pos_fmod(p1+by, (double) bitmap.height()));
	}
/*...e*/
/*...sPrimColField2dInterp:0:*/
// Just like PrimColField2d, only with interpolation
// This class defined as it is expected that interpolation on bitmap fields
// is going to be quite a common requirement, and that this implementation
// will be quite a bit faster than building the same thing using the
// PrimColInterp0, PrimColInterp1 and PrimColField2d classes.

class PrimColField2dInterp : public PrimColField2d
	{
public:
	PrimColField2dInterp(double bx, double by, Bitmap bitmap) :
		PrimColField2d(bx, by, bitmap) {}
	~PrimColField2dInterp() {}
	virtual PrimCol *copy() const
		{ return new PrimColField2dInterp(bx, by, bitmap); }
	virtual Rgb evaluate(double p0, double p1, double p2) const;
	};

// Interpolation fully expanded out for speed

Rgb PrimColField2dInterp::evaluate(double p0, double p1, double p2) const
	{
	p2=p2; // Suppress warning
	double x = p0 + bx, ix = floor(x), fx = x-ix;
	double y = p1 + by, iy = floor(y), fy = y-iy;
	int bx = bitmap.width();
	int ix0 = pos_mod((int) ix, bx);
	int ix1 = ix0+1; if ( ix1 == bx ) ix1 = 0;
	int by = bitmap.height();
	int iy0 = pos_mod((int) iy, by);
	int iy1 = iy0+1; if ( iy1 == by ) iy1 = 0;
	Rgb rgb00(bitmap.get_pixel(ix0, iy0));
	Rgb rgb10(bitmap.get_pixel(ix1, iy0));
	Rgb rgb01(bitmap.get_pixel(ix0, iy1));
	Rgb rgb11(bitmap.get_pixel(ix1, iy1));
	Rgb rgbx0(interp_rgb(rgb00, rgb10, fx));
	Rgb rgbx1(interp_rgb(rgb01, rgb11, fx));
	return interp_rgb(rgbx0, rgbx1, fy);
	}
/*...e*/
/*...sPrimColField3d:0:*/
class PrimColField3d : public PrimCol
	{
protected:
	double bx, by, bz;
	Texture texture;
public:
	PrimColField3d(double bx, double by, double bz, Texture texture)
		: bx(bx), by(by), bz(bz), texture(texture) {}
	~PrimColField3d() {}
	virtual PrimCol *copy() const
		{ return new PrimColField3d(bx, by, bz, texture); }
	virtual PrimCol *trans  (Xyz    t) const
		{ t=t; return copy(); }
	virtual PrimCol *trans_x(double t) const
		{ t=t; return copy(); }
	virtual PrimCol *trans_y(double t) const
		{ t=t; return copy(); }
	virtual PrimCol *trans_z(double t) const
		{ t=t; return copy(); }
	virtual PrimCol *rot_x(double angle) const
		{ angle=angle; return copy(); }
	virtual PrimCol *rot_y(double angle) const
		{ angle=angle; return copy(); }
	virtual PrimCol *rot_z(double angle) const
		{ angle=angle; return copy(); }
	virtual PrimCol *scale  (Xyz    factor) const
		{ factor=factor; return copy(); } 
	virtual PrimCol *scale  (double factor) const
		{ factor=factor; return copy(); } 
	virtual PrimCol *scale_x(double factor) const
		{ factor=factor; return copy(); } 
	virtual PrimCol *scale_y(double factor) const
		{ factor=factor; return copy(); } 
	virtual PrimCol *scale_z(double factor) const
		{ factor=factor; return copy(); }
	virtual Rgb evaluate(double p0, double p1, double p2) const;
	};

Rgb PrimColField3d::evaluate(double p0, double p1, double p2) const
	{
	return texture.get_voxel(
		(int) pos_fmod(p0+bx, (double) texture.width ()),
		(int) pos_fmod(p1+by, (double) texture.height()),
		(int) pos_fmod(p2+bz, (double) texture.depth ()));
	}
/*...e*/
/*...sPrimColField3dInterp:0:*/
// Just like PrimColField3d, only with interpolation
// This class defined as it is expected that interpolation on texture fields
// is going to be quite a common requirement, and that this implementation
// will be quite a bit faster than building the same thing using the
// PrimColInterp0, PrimColInterp1, PrimColInterp2 and PrimColField3d classes.

class PrimColField3dInterp : public PrimColField3d
	{
public:
	PrimColField3dInterp(double bx, double by, double bz, Texture texture) :
		PrimColField3d(bx, by, bz, texture) {}
	~PrimColField3dInterp() {}
	virtual PrimCol *copy() const
		{ return new PrimColField3dInterp(bx, by, bz, texture); }
	virtual Rgb evaluate(double p0, double p1, double p2) const;
	};

Rgb PrimColField3dInterp::evaluate(double p0, double p1, double p2) const
	{
	double x = p0 + bx, ix = floor(x), fx = x-ix;
	double y = p1 + by, iy = floor(y), fy = y-iy;
	double z = p2 + bz, iz = floor(z), fz = z-iz;
	int tx = texture.width();
	int ix0 = pos_mod((int) ix, tx);
	int ix1 = ix0+1; if ( ix1 == tx ) ix1 = 0;
	int ty = texture.height();
	int iy0 = pos_mod((int) iy, ty);
	int iy1 = iy0+1; if ( iy1 == ty ) iy1 = 0;
	int tz = texture.depth();
	int iz0 = pos_mod((int) iz, tz);
	int iz1 = iz0+1; if ( iz1 == tz ) iz1 = 0;
	Rgb rgb000(texture.get_voxel(ix0, iy0, iz0));
	Rgb rgb100(texture.get_voxel(ix1, iy0, iz0));
	Rgb rgb010(texture.get_voxel(ix0, iy1, iz0));
	Rgb rgb110(texture.get_voxel(ix1, iy1, iz0));
	Rgb rgb001(texture.get_voxel(ix0, iy0, iz1));
	Rgb rgb101(texture.get_voxel(ix1, iy0, iz1));
	Rgb rgb011(texture.get_voxel(ix0, iy1, iz1));
	Rgb rgb111(texture.get_voxel(ix1, iy1, iz1));
	Rgb rgbx00(interp_rgb(rgb000, rgb100, fx));
	Rgb rgbx10(interp_rgb(rgb010, rgb110, fx));
	Rgb rgbx01(interp_rgb(rgb001, rgb101, fx));
	Rgb rgbx11(interp_rgb(rgb011, rgb111, fx));
	Rgb rgbxy0(interp_rgb(rgbx00, rgbx10, fy));
	Rgb rgbxy1(interp_rgb(rgbx01, rgbx11, fy));
	return interp_rgb(rgbxy0, rgbxy1, fz);
	}
/*...e*/
/*...sPrimColRemap:0:*/
class PrimColRemap : public PrimCol
	{
	Xyz base, v0, v1, v2;
	Matrix3 inv_m;
	PrimCol *col;
public:
	PrimColRemap(Xyz base, Xyz v0, Xyz v1, Xyz v2, PrimCol *col)
		: base(base), v0(v0), v1(v1), v2(v2), col(col)
		{
		inv_m = Matrix3(v0, v1, v2).inverse();
		}
	~PrimColRemap() { delete col; }
	virtual PrimCol *copy() const
		{ return new PrimColRemap(base,v0,v1,v2,col->copy()); }
	virtual PrimCol *trans  (Xyz t) const
		{ return new PrimColRemap(base+t,v0,v1,v2,col->copy()); }
	virtual PrimCol *trans_x(double t) const
		{ return new PrimColRemap(base+Xyz(t,0.0,0.0),v0,v1,v2,col->copy()); }
	virtual PrimCol *trans_y(double t) const
		{ return new PrimColRemap(base+Xyz(0.0,t,0.0),v0,v1,v2,col->copy()); }
	virtual PrimCol *trans_z(double t) const
		{ return new PrimColRemap(base+Xyz(0.0,0.0,t),v0,v1,v2,col->copy()); }
	virtual PrimCol *rot_x(double angle) const
		{ return new PrimColRemap(
			base.rot_x(angle),
			v0  .rot_x(angle),
			v1  .rot_x(angle),
			v2  .rot_x(angle),
			col->copy()); }
	virtual PrimCol *rot_y(double angle) const
		{ return new PrimColRemap(
			base.rot_y(angle),
			v0  .rot_y(angle),
			v1  .rot_y(angle),
			v2  .rot_y(angle),
			col->copy()); }
	virtual PrimCol *rot_z(double angle) const
		{ return new PrimColRemap(
			base.rot_z(angle),
			v0  .rot_z(angle),
			v1  .rot_z(angle),
			v2  .rot_z(angle),
			col->copy()); }
	virtual PrimCol *scale  (Xyz factor) const
// This is ugly, its a shame that Xyz*Xyz could be either
// 'Xyz Product, 'Scalar Product' or 'Scale one by the other'.
		{ return new PrimColRemap(
			Xyz(base.x*factor.x,base.y*factor.y,base.z*factor.z),
			Xyz(v0  .x*factor.x,v0  .y*factor.y,v0  .z*factor.z),
			Xyz(v1  .x*factor.x,v1  .y*factor.y,v1  .z*factor.z),
			Xyz(v2  .x*factor.x,v2  .y*factor.y,v2  .z*factor.z),
			col->copy()); }
	virtual PrimCol *scale  (double factor) const
		{ return scale(Xyz(factor,factor,factor)); }
	virtual PrimCol *scale_x(double factor) const
		{ return new PrimColRemap(
			Xyz(base.x*factor,base.y,base.z),
			Xyz(v0  .x*factor,v0  .y,v0  .z),
			Xyz(v1  .x*factor,v1  .y,v1  .z),
			Xyz(v2  .x*factor,v2  .y,v2  .z),
			col->copy()); }
	virtual PrimCol *scale_y(double factor) const
		{ return new PrimColRemap(
			Xyz(base.x,base.y*factor,base.z),
			Xyz(v0  .x,v0  .y*factor,v0  .z),
			Xyz(v1  .x,v1  .y*factor,v1  .z),
			Xyz(v2  .x,v2  .y*factor,v2  .z),
			col->copy()); }
	virtual PrimCol *scale_z(double factor) const
		{ return new PrimColRemap(
			Xyz(base.x,base.y,base.z*factor),
			Xyz(v0  .x,v0  .y,v0  .z*factor),
			Xyz(v1  .x,v1  .y,v1  .z*factor),
			Xyz(v2  .x,v2  .y,v2  .z*factor),
			col->copy()); }
	virtual Rgb evaluate(double p0, double p1, double p2) const;
	};

/*...sevaluate:0:*/
// Point (p) = base (b) + multiples (c1,c2 and c3) of 3 vectors (v0, v1 and v2).
//
//         /                \ /    \    /    \       /                \
//         | v0.x v1.x v2.x | | c1 |    | c1 |       | v0.x v1.x v2.x |
// p = b + | v0.y v1.y v2.y | | c2 | => | c2 | = inv | v0.y v1.y v2.y | ( p-b )
// ~   ~   | v0.z v1.z v2.z | | c3 |    | c3 |       | v0.z v1.z v2.z |   ~ ~
//         \                / \    /    \    /       \                /
//
// The inverse matrix is kept ready computed.

Rgb PrimColRemap::evaluate(double p0, double p1, double p2) const
	{
	Xyz pb(p0-base.x,p1-base.y,p2-base.z);
	Xyz c(inv_m * pb);
	return col->evaluate(c.x, c.y, c.z);
	}
/*...e*/
/*...e*/
/*...sPrimColCylPolar:0:*/
class PrimColCylPolar : public PrimCol
	{
	double lond, rd, hd;
	PrimCol *col;
public:
	PrimColCylPolar(double lond, double rd, double hd, PrimCol *col) :
		lond(lond), rd(rd), hd(hd), col(col) {}
	~PrimColCylPolar() { delete col; }
	virtual PrimCol *copy() const
		{ return new PrimColCylPolar(lond,rd,hd, col->copy()); }
	virtual PrimCol *trans  (Xyz t) const
		{ return new PrimColCylPolar(lond,rd,hd, col->trans  (t)); }
	virtual PrimCol *trans_x(double t) const
		{ return new PrimColCylPolar(lond,rd,hd, col->trans_x(t)); }
	virtual PrimCol *trans_y(double t) const
		{ return new PrimColCylPolar(lond,rd,hd, col->trans_y(t)); }
	virtual PrimCol *trans_z(double t) const
		{ return new PrimColCylPolar(lond,rd,hd, col->trans_z(t)); }
	virtual PrimCol *rot_x(double angle) const
		{ return new PrimColCylPolar(lond,rd,hd, col->rot_x(angle)); }
	virtual PrimCol *rot_y(double angle) const
		{ return new PrimColCylPolar(lond,rd,hd, col->rot_y(angle)); }
	virtual PrimCol *rot_z(double angle) const
		{ return new PrimColCylPolar(lond,rd,hd, col->rot_z(angle)); }
	virtual PrimCol *scale  (Xyz factor) const
		{ return new PrimColCylPolar(lond,rd,hd, col->scale  (factor)); }
	virtual PrimCol *scale  (double factor) const
		{ return new PrimColCylPolar(lond,rd,hd, col->scale  (factor)); }
	virtual PrimCol *scale_x(double factor) const
		{ return new PrimColCylPolar(lond,rd,hd, col->scale_x(factor)); }
	virtual PrimCol *scale_y(double factor) const
		{ return new PrimColCylPolar(lond,rd,hd, col->scale_y(factor)); }
	virtual PrimCol *scale_z(double factor) const
		{ return new PrimColCylPolar(lond,rd,hd, col->scale_z(factor)); }
	virtual Rgb evaluate(double p0, double p1, double p2) const;
	};

Rgb PrimColCylPolar::evaluate(double p0, double p1, double p2) const
	{
	return col->evaluate(atan2(p1,p0)/lond, sqrt((p0*p0+p1*p1)/rd), p2/hd);
	}
/*...e*/
/*...sPrimColSphPolar:0:*/
class PrimColSphPolar : public PrimCol
	{
	double lond, latd, rd;
	PrimCol *col;
public:
	PrimColSphPolar(double lond, double latd, double rd, PrimCol *col) :
		lond(lond), latd(latd), rd(rd), col(col) {}
	~PrimColSphPolar() { delete col; }
	virtual PrimCol *copy() const
		{ return new PrimColSphPolar(lond,latd,rd, col->copy()); }
	virtual PrimCol *trans  (Xyz t) const
		{ return new PrimColSphPolar(lond,latd,rd, col->trans  (t)); }
	virtual PrimCol *trans_x(double t) const
		{ return new PrimColSphPolar(lond,latd,rd, col->trans_x(t)); }
	virtual PrimCol *trans_y(double t) const
		{ return new PrimColSphPolar(lond,latd,rd, col->trans_y(t)); }
	virtual PrimCol *trans_z(double t) const
		{ return new PrimColSphPolar(lond,latd,rd, col->trans_z(t)); }
	virtual PrimCol *rot_x(double angle) const
		{ return new PrimColSphPolar(lond,latd,rd, col->rot_x(angle)); }
	virtual PrimCol *rot_y(double angle) const
		{ return new PrimColSphPolar(lond,latd,rd, col->rot_y(angle)); }
	virtual PrimCol *rot_z(double angle) const
		{ return new PrimColSphPolar(lond,latd,rd, col->rot_z(angle)); }
	virtual PrimCol *scale  (Xyz factor) const
		{ return new PrimColSphPolar(lond,latd,rd, col->scale  (factor)); }
	virtual PrimCol *scale  (double factor) const
		{ return new PrimColSphPolar(lond,latd,rd, col->scale  (factor)); }
	virtual PrimCol *scale_x(double factor) const
		{ return new PrimColSphPolar(lond,latd,rd, col->scale_x(factor)); }
	virtual PrimCol *scale_y(double factor) const
		{ return new PrimColSphPolar(lond,latd,rd, col->scale_y(factor)); }
	virtual PrimCol *scale_z(double factor) const
		{ return new PrimColSphPolar(lond,latd,rd, col->scale_z(factor)); }
	virtual Rgb evaluate(double p0, double p1, double p2) const;
	};

Rgb PrimColSphPolar::evaluate(double p0, double p1, double p2) const
	{
	double lon = atan2(p1, p0);
	double p01 = p0*p0 + p1*p1;
	double lat = atan2(p2, sqrt(p01));
	double p012 = p01 + p2*p2;
	return col->evaluate(lon/lond,lat/latd,sqrt(p012)/rd);
	}
/*...e*/
/*...sPrimColMatrix2d:0:*/
class PrimColMatrix2d : public PrimCol
	{
	double m[2][2];
	PrimCol *col;
public:
	PrimColMatrix2d(const double m2[2][2], PrimCol *col)
		:
		col(col)
		{
		for ( int j = 0; j < 2; j++ )
			for ( int i = 0; i < 2; i++ )
				m[j][i] = m2[j][i];
		}
	~PrimColMatrix2d() { delete col; }
	virtual PrimCol *copy() const
		{ return new PrimColMatrix2d(m, col->copy()); }
	virtual PrimCol *trans  (Xyz t) const
		{ return new PrimColMatrix2d(m, col->trans  (t)); }
	virtual PrimCol *trans_x(double t) const
		{ return new PrimColMatrix2d(m, col->trans_x(t)); }
	virtual PrimCol *trans_y(double t) const
		{ return new PrimColMatrix2d(m, col->trans_y(t)); }
	virtual PrimCol *trans_z(double t) const
		{ return new PrimColMatrix2d(m, col->trans_z(t)); }
	virtual PrimCol *rot_x(double angle) const
		{ return new PrimColMatrix2d(m, col->rot_x(angle)); }
	virtual PrimCol *rot_y(double angle) const
		{ return new PrimColMatrix2d(m, col->rot_y(angle)); }
	virtual PrimCol *rot_z(double angle) const
		{ return new PrimColMatrix2d(m, col->rot_z(angle)); }
	virtual PrimCol *scale  (Xyz factor) const
		{ return new PrimColMatrix2d(m, col->scale  (factor)); }
	virtual PrimCol *scale  (double factor) const
		{ return new PrimColMatrix2d(m, col->scale  (factor)); }
	virtual PrimCol *scale_x(double factor) const
		{ return new PrimColMatrix2d(m, col->scale_x(factor)); }
	virtual PrimCol *scale_y(double factor) const
		{ return new PrimColMatrix2d(m, col->scale_y(factor)); }
	virtual PrimCol *scale_z(double factor) const
		{ return new PrimColMatrix2d(m, col->scale_z(factor)); }
	virtual Rgb evaluate(double p0, double p1, double p2) const;
	};

Rgb PrimColMatrix2d::evaluate(double p0, double p1, double p2) const
	{
	return col->evaluate(
		m[0][0] * p0 + m[0][1] * p1,
		m[1][0] * p0 + m[1][1] * p1,
		p2);
	}
/*...e*/
/*...sPrimColMatrix3d:0:*/
class PrimColMatrix3d : public PrimCol
	{
	double m[3][3];
	PrimCol *col;
public:
	PrimColMatrix3d(const double m3[3][3], PrimCol *col) : col(col)
		{
		for ( int j = 0; j < 3; j++ )
			for ( int i = 0; i < 3; i++ )
				m[j][i] = m3[j][i];
		}
	~PrimColMatrix3d() { delete col; }
	virtual PrimCol *copy() const
		{ return new PrimColMatrix3d(m, col->copy()); }
	virtual PrimCol *trans  (Xyz t) const
		{ return new PrimColMatrix3d(m, col->trans  (t)); }
	virtual PrimCol *trans_x(double t) const
		{ return new PrimColMatrix3d(m, col->trans_x(t)); }
	virtual PrimCol *trans_y(double t) const
		{ return new PrimColMatrix3d(m, col->trans_y(t)); }
	virtual PrimCol *trans_z(double t) const
		{ return new PrimColMatrix3d(m, col->trans_z(t)); }
	virtual PrimCol *rot_x(double angle) const
		{ return new PrimColMatrix3d(m, col->rot_x(angle)); }
	virtual PrimCol *rot_y(double angle) const
		{ return new PrimColMatrix3d(m, col->rot_y(angle)); }
	virtual PrimCol *rot_z(double angle) const
		{ return new PrimColMatrix3d(m, col->rot_z(angle)); }
	virtual PrimCol *scale  (Xyz factor) const
		{ return new PrimColMatrix3d(m, col->scale  (factor)); }
	virtual PrimCol *scale  (double factor) const
		{ return new PrimColMatrix3d(m, col->scale  (factor)); }
	virtual PrimCol *scale_x(double factor) const
		{ return new PrimColMatrix3d(m, col->scale_x(factor)); }
	virtual PrimCol *scale_y(double factor) const
		{ return new PrimColMatrix3d(m, col->scale_y(factor)); }
	virtual PrimCol *scale_z(double factor) const
		{ return new PrimColMatrix3d(m, col->scale_z(factor)); }
	virtual Rgb evaluate(double p0, double p1, double p2) const;
	};

Rgb PrimColMatrix3d::evaluate(double p0, double p1, double p2) const
	{
	return col->evaluate(
		m[0][0] * p0 + m[0][1] * p1 + m[0][2] * p2,
		m[1][0] * p0 + m[1][1] * p1 + m[1][2] * p2,
		m[2][0] * p0 + m[2][1] * p1 + m[2][2] * p2);
	}
/*...e*/

/*...sCol:0:*/
/*...sconstruction \47\ destruction etc\46\:0:*/
Col::Col() { p = new ColRefCnt(new PrimColRgb(Rgb(0.0,0.0,0.0))); }

Col::Col(PrimCol *prim) { p = new ColRefCnt(prim); }

Col::Col(const Col & col) { p = col.p; p->n++; }

Col & Col::operator=(const Col & col)
	{
	++col.p->n;
	if ( --(p->n) == 0 )
		delete p;
	p = col.p;
	return *this;
	}

Col::~Col()
	{
	if ( --(p->n) == 0 )
		delete p;
	}
/*...e*/
/*...smethods to construct colours:0:*/
Col Col::rgb(Rgb rgb)
	{ return Col(new PrimColRgb(rgb)); }

Col Col::rgb(Rgb rgb0, Rgb rgb1)
	{ return Col(new PrimColRgbBlend(rgb0, rgb1)); }

Col Col::nomove(Col col)
	{ return Col(new PrimColNoMove(((PrimCol *) col)->copy())); }

Col Col::interp0(Col col)
	{ return Col(new PrimColInterp0(((PrimCol *) col)->copy())); }

Col Col::interp1(Col col)
	{ return Col(new PrimColInterp1(((PrimCol *) col)->copy())); }

Col Col::interp2(Col col)
	{ return Col(new PrimColInterp2(((PrimCol *) col)->copy())); }

Col Col::field2d(double bx, double by, Bitmap bitmap)
	{ return Col(new PrimColField2d(bx, by, bitmap)); }

Col Col::field2dinterp(double bx, double by, Bitmap bitmap)
	{ return Col(new PrimColField2dInterp(bx, by, bitmap)); }

Col Col::field3d(double bx, double by, double bz, Texture texture)
	{ return Col(new PrimColField3d(bx, by, bz, texture)); }

Col Col::field3dinterp(double bx, double by, double bz, Texture texture)
	{ return Col(new PrimColField3dInterp(bx, by, bz, texture)); }

Col Col::remap(Xyz base, Xyz v0, Xyz v1, Xyz v2, Col col)
	{ return Col(new PrimColRemap(base, v0, v1, v2, ((PrimCol *) col)->copy())); }

Col Col::cyl_polar(double lond, double rd, double hd, Col col)
	{ return Col(new PrimColCylPolar(lond, rd, hd, ((PrimCol *) col)->copy())); }

Col Col::sph_polar(double lond, double latd, double rd, Col col)
	{ return Col(new PrimColSphPolar(lond, latd, rd, ((PrimCol *) col)->copy())); }

Col Col::matrix2d(double m[2][2], Col col)
	{ return Col(new PrimColMatrix2d(m, ((PrimCol *) col)->copy())); }

Col Col::matrix2d(double m00, double m01,
		  double m10, double m11,
		  Col col
	)
	{
	double m[2][2];
	m[0][0] = m00; m[0][1] = m01;
	m[1][0] = m10; m[1][1] = m11;
	return matrix2d(m, col);
	}

Col Col::matrix3d(double m[3][3], Col col)
	{ return Col(new PrimColMatrix3d(m, ((PrimCol *) col)->copy())); }

Col Col::matrix3d(double m00, double m01, double m02,
		  double m10, double m11, double m12,
		  double m20, double m21, double m22,
		  Col col
	)
	{
	double m[3][3];
	m[0][0] = m00; m[0][1] = m01; m[0][2] = m02;
	m[1][0] = m10; m[1][1] = m11; m[1][2] = m12;
	m[2][0] = m20; m[2][1] = m21; m[2][2] = m22;
	return matrix3d(m, col);
	}
/*...e*/
/*...sgeneral manipulation:0:*/
Col Col::trans  (Xyz    t     ) const { return Col(p->prim->trans  (t     )); }
Col Col::trans_x(double t     ) const { return Col(p->prim->trans_x(t     )); }
Col Col::trans_y(double t     ) const { return Col(p->prim->trans_y(t     )); }
Col Col::trans_z(double t     ) const { return Col(p->prim->trans_z(t     )); }
Col Col::rot_x  (double angle ) const { return Col(p->prim->rot_x  (angle )); }
Col Col::rot_y  (double angle ) const { return Col(p->prim->rot_y  (angle )); }
Col Col::rot_z  (double angle ) const { return Col(p->prim->rot_z  (angle )); }
Col Col::scale  (Xyz    factor) const { return Col(p->prim->scale  (factor)); }
Col Col::scale  (double factor) const { return Col(p->prim->scale  (factor)); }
Col Col::scale_x(double factor) const { return Col(p->prim->scale_x(factor)); }
Col Col::scale_y(double factor) const { return Col(p->prim->scale_y(factor)); }
Col Col::scale_z(double factor) const { return Col(p->prim->scale_z(factor)); }
/*...e*/
/*...e*/
