//
// col.h - Colours
//
// Colours can actually be defined in terms of other colours.
// So internally, a tree structure is used. To avoid excessive new-ing and
// delete-ing by the user I implement a reference counting wrapper.
// This does all the new-ing and delete-ing, and hopefully not as much too.
//

#ifndef COL_H
#define	COL_H

/*...sincludes:0:*/
#include "types.h"
#include "xyz.h"
#include "rgb.h"
#include "matrix.h"
#include "bitmap.h"
#include "texture.h"

/*...vtypes\46\h:0:*/
/*...vxyz\46\h:0:*/
/*...vrgb\46\h:0:*/
/*...vmatrix\46\h:0:*/
/*...vbitmap\46\h:0:*/
/*...vtexture\46\h:0:*/
/*...e*/

/*...sprivate PrimCol:0:*/
// Abstract base class. By implementing another derivation of this it is
// possible to extend the set of supported base colour types.

class PrimCol
	{
public:
	PrimCol() {}
	virtual ~PrimCol() {}
	virtual PrimCol *copy() const = 0;
	virtual PrimCol *trans  (Xyz    t) const = 0;
	virtual PrimCol *trans_x(double t) const = 0;
	virtual PrimCol *trans_y(double t) const = 0;
	virtual PrimCol *trans_z(double t) const = 0;
	virtual PrimCol *rot_x(double angle) const = 0;
	virtual PrimCol *rot_y(double angle) const = 0;
	virtual PrimCol *rot_z(double angle) const = 0;
	virtual PrimCol *scale  (Xyz    factor) const = 0;
	virtual PrimCol *scale  (double factor) const = 0;
	virtual PrimCol *scale_x(double factor) const = 0;
	virtual PrimCol *scale_y(double factor) const = 0;
	virtual PrimCol *scale_z(double factor) const = 0;
	virtual Rgb evaluate(double p0, double p1, double p2) const = 0;
	};
/*...e*/

class Col
	{
/*...sprivate:8:*/
class ColRefCnt
	{
public:
	PrimCol *prim;
	int n;
	ColRefCnt(PrimCol *prim) : n(1), prim(prim) {}
	~ColRefCnt() { delete prim; }
	};
ColRefCnt *p;	
/*...e*/
public:
	Col(); // Black
	Col(const Col & col);
	Col & operator=(const Col & col);
	~Col();

	// In the absence of a decent multiple constructor mechanism in C++
	// use these static member functions instead...

	static Col rgb(Rgb rgb);
	static Col rgb(Rgb rgb0, Rgb rgb1);
	static Col nomove(Col col);
	static Col interp0(Col col);
	static Col interp1(Col col);
	static Col interp2(Col col);
	static Col field2d      (double bx, double by, Bitmap bitmap);
	static Col field2dinterp(double bx, double by, Bitmap bitmap);
	static Col field3d      (double bx, double by, double bz, Texture texture);
	static Col field3dinterp(double bx, double by, double bz, Texture texture);
	static Col remap(Xyz base, Xyz v0, Xyz v1, Xyz v2, Col col);
	static Col cyl_polar(double lond, double rd, double hd, Col col);
	static Col sph_polar(double lond, double latd, double rd, Col col);
	static Col matrix2d(double m[2][2], Col col);
	static Col matrix2d(double m00, double m01,
			    double m10, double m11,
			    Col col);
	static Col matrix3d(double m[3][3], Col col);
	static Col matrix3d(double m00, double m01, double m02,
			    double m10, double m11, double m12,
			    double m20, double m21, double m22,
			    Col col);

	// The following allow a colour to move with the object it colours.

	Col trans  (Xyz    t     ) const;
	Col trans_x(double t     ) const;
	Col trans_y(double t     ) const;
	Col trans_z(double t     ) const;
	Col rot_x  (double angle ) const;
	Col rot_y  (double angle ) const;
	Col rot_z  (double angle ) const;
	Col scale  (Xyz    factor) const;
	Col scale  (double factor) const;
	Col scale_x(double factor) const;
	Col scale_y(double factor) const;
	Col scale_z(double factor) const;

	// Find out the colour given 3 parameters.
	// These methods must be fast, so are inline.
	Rgb evaluate(double p0, double p1, double p2) const
		{ return p->prim->evaluate(p0, p1, p2); }
	Rgb evaluate(const Xyz & v) const
		{ return p->prim->evaluate(v.x, v.y, v.z); }

	// Internal use only.
	// Its a pain to use the friend mechanism in so many places...

	Col(PrimCol *prim);
	operator PrimCol *() const { return p->prim; };
	};

#endif
