//
// matrix.C - Matrix tools
//

/*...sincludes:0:*/
#ifdef NO_CINCLUDES
  #include <stddef.h>
#else
  #include <cstddef>
#endif

#include "matrix.h"

/*...vmatrix\46\h:0:*/
/*...e*/

/*...sMatrix3:0:*/
/*...sMatrix3\39\s:0:*/
Matrix3::Matrix3(const double m[3][3])
	{
	for ( int row = 0; row < 3; row++ )
		for ( int col = 0; col < 3; col++ )
			Matrix3::m[row][col] = m[row][col];
	}

Matrix3::Matrix3(
	double m00, double m01, double m02,
	double m10, double m11, double m12,
	double m20, double m21, double m22
	)
	{
	m[0][0] = m00; m[0][1] = m01; m[0][2] = m02;
	m[1][0] = m10; m[1][1] = m11; m[1][2] = m12;
	m[2][0] = m20; m[2][1] = m21; m[2][2] = m22;
	}

Matrix3::Matrix3(const Xyz & xyz0, const Xyz & xyz1, const Xyz & xyz2)
	{
	m[0][0] = xyz0.x; m[0][1] = xyz1.x; m[0][2] = xyz2.x;
	m[1][0] = xyz0.y; m[1][1] = xyz1.y; m[1][2] = xyz2.y;
	m[2][0] = xyz0.z; m[2][1] = xyz1.z; m[2][2] = xyz2.z;
	}
/*...e*/

/*...sinverse:0:*/
//	 -1     1          T
//	M   = ------ cof(M)
//	      det(M)

/*...scofactor:0:*/
// See Stephenson p277 for defn of a 'minor'.
// See Stephenson p307 for defn of a 'cofactor'.

static double cofactor(int row, int col, const double m[3][3])
	{
	static int lower[] = { 1, 0, 0 };
	static int upper[] = { 2, 2, 1 };
	int lower_row = lower[row], upper_row = upper[row];
	int lower_col = lower[col], upper_col = upper[col];
	double minor = m[lower_row][lower_col] * m[upper_row][upper_col] -
		       m[lower_row][upper_col] * m[upper_row][lower_col];

	return ((row + col) & 1) ? -minor : minor;
	}
/*...e*/

Matrix3 Matrix3::inverse() const
	{
	int row, col;
	double det = 0.0;
	double cof_m[3][3];
	Matrix3 inv;

	for ( col = 0; col < 3; col++ )
		{
		for ( row = 0; row < 3; row++ )
			cof_m[row][col] = cofactor(row, col, m);
		det += m[0][col] * cof_m[0][col];
		}

	// determinant should not be 0, and if this then this implies an
	// error in some other part of the program.

	det = 1.0 / det;

	for ( col = 0; col < 3; col++ )
		for ( row = 0; row < 3; row++ )
			inv.m[col][row] = det * cof_m[row][col];

	return inv;
	}
/*...e*/

/*...soperator\42\ \40\m3\61\m3\42\m3\41\:0:*/
Matrix3 operator*(const Matrix3 & m1, const Matrix3 & m2)
	{
	Matrix3 product;
	for ( int row = 0; row < 3; row++ )
		for ( int col = 0; col < 3; col++ )
			{
			product.m[row][col] = 0.0;
			for ( int i = 0; i < 3; i++ )
				product.m[row][col] += m1.m[row][i] * m2.m[i][col];
			}
	return product;
	}
/*...e*/
/*...soperator\42\ \40\xyz\61\m3\42\xyz\41\:0:*/
Xyz operator*(const Matrix3 & m, const Xyz & xyz)
	{
	return Xyz(
		m.m[0][0] * xyz.x +
		m.m[0][1] * xyz.y +
		m.m[0][2] * xyz.z,
		m.m[1][0] * xyz.x +
		m.m[1][1] * xyz.y +
		m.m[1][2] * xyz.z,
		m.m[2][0] * xyz.x +
		m.m[2][1] * xyz.y +
		m.m[2][2] * xyz.z);
	}
/*...e*/
/*...e*/
/*...sMatrix4:0:*/
/*...sMatrix4\39\s:0:*/
Matrix4::Matrix4(const double m[4][4])
	{
	for ( int row = 0; row < 4; row++ )
		for ( int col = 0; col < 4; col++ )
			Matrix4::m[row][col] = m[row][col];
	}

Matrix4::Matrix4(
	double m00, double m01, double m02, double m03,
	double m10, double m11, double m12, double m13,
	double m20, double m21, double m22, double m23,
	double m30, double m31, double m32, double m33
	)
	{
	m[0][0] = m00; m[0][1] = m01; m[0][2] = m02; m[0][3] = m03;
	m[1][0] = m10; m[1][1] = m11; m[1][2] = m12; m[1][3] = m13;
	m[2][0] = m20; m[2][1] = m21; m[2][2] = m22; m[2][3] = m23;
	m[3][0] = m30; m[3][1] = m31; m[3][2] = m32; m[3][3] = m33;
	}

// Expand size of matrix

Matrix4::Matrix4(const Matrix3 & m3)
	{
	for ( int row = 0; row < 3; row++ )
		{
		for ( int col = 0; col < 3; col++ )
			m[row][col] = m3.m[row][col];
		m[row][3] = 0.0;
		}
	for ( int col = 0; col < 3; col++ )
		m[3][col] = 0.0;
	m[3][3] = 1.0;
	}
/*...e*/

/*...soperator\42\ \40\m4\61\m4\42\m4\41\:0:*/
Matrix4 operator*(const Matrix4 & m1, const Matrix4 & m2)
	{
	Matrix4 product;
	for ( int row = 0; row < 4; row++ )
		for ( int col = 0; col < 4; col++ )
			{
			product.m[row][col] = 0.0;
			for ( int i = 0; i < 4; i++ )
				product.m[row][col] += m1.m[row][i] * m2.m[i][col];
			}
	return product;
	}
/*...e*/
/*...soperator\42\ \40\xyz\61\m4\42\xyz\41\:0:*/
// Of course, this ought to read Matrix4 * Xyzw, but we shall assume an
// implied incoming W value of 1.
//
// The normal thing to do is to calculate W and homogenise before returning
// by dividing X,Y and Z by it.

Xyz operator*(const Matrix4 & m, const Xyz & xyz)
	{
	double x = m.m[0][0] * xyz.x +
		   m.m[0][1] * xyz.y +
		   m.m[0][2] * xyz.z +
		   m.m[0][3];
	double y = m.m[1][0] * xyz.x +
		   m.m[1][1] * xyz.y +
		   m.m[1][2] * xyz.z +
		   m.m[1][3];
	double z = m.m[2][0] * xyz.x +
		   m.m[2][1] * xyz.y +
		   m.m[2][2] * xyz.z +
		   m.m[2][3];
	double w = m.m[3][0] * xyz.x +
		   m.m[3][1] * xyz.y +
		   m.m[3][2] * xyz.z +
		   m.m[3][3];

	return Xyz(x/w, y/w, z/w);
	}
/*...e*/
/*...e*/

/*...sto_viewport:0:*/
// div_z transforms from view coordinates having a pyramidal view volume into
// an almost canonical view volume.
//
// pix scales our canonical view volume to screen pixels values.
//
// centre centers the pixel values onto the middle of display area.
//
// Clearly we desire the composite transformation matrix which combines all
// the above matrices into one.

static Matrix4 div_z(
	1.0, 0.0, 0.0, 0.0,
	0.0, 1.0, 0.0, 0.0,
	0.0, 0.0, 1.0, 0.0,
	0.0, 0.0, 1.0, 0.0);

Matrix4 to_viewport(int w, int h)
	{
	// Half of widths and heights much more interesting
	// bugfix: pix used to use w/2-1 and centre used to use w/2
	double hx = (double)(w-1)/2;
	double hy = (double)(h-1)/2;

	// Now scale from our canonical view volume to pixel coordinates
	Matrix4 pix(
		hx , 0.0, 0.0, 0.0,
		0.0, hy , 0.0, 0.0,
		0.0, 0.0, 1.0, 0.0,
		0.0, 0.0, 0.0, 1.0);

	// Now account for centralising in the screen
	Matrix4 centre(
		1.0, 0.0, 0.0, hx,
		0.0, 1.0, 0.0, hy,
		0.0, 0.0, 1.0, 0.0,
		0.0, 0.0, 0.0, 1.0);

	// Explicit multi-statement multiply to ensure right to left
	// evaluation of the multiplies.
	Matrix4 m(pix*div_z);
	return centre*m;
	}
/*...e*/
