//
// polygon.C - Support for Polygon rendering
//

/*...sincludes:0:*/
#include <stdio.h>
#include <stdlib.h>

#include "insane.h"
#include "polygon.h"

/*...vinsane\46\h:0:*/
/*...vpolygon\46\h:0:*/
/*...e*/

/*...sclass OpCol:0:*/
// This type of op specifies the drawing colour

class OpCol : public Op
	{
public:
	int index;
	OpCol(int index) : index(index) {}
	virtual ~OpCol() {}
	virtual Op *copy() const { return new OpCol(index); }
	virtual Op *trans  (Xyz t) const { t=t; return copy(); }
	virtual Op *trans_x(double t) const { t=t; return copy(); }
	virtual Op *trans_y(double t) const { t=t; return copy(); }
	virtual Op *trans_z(double t) const { t=t; return copy(); }
	virtual Op *rot_x(double angle) const { angle=angle; return copy(); }
	virtual Op *rot_y(double angle) const { angle=angle; return copy(); }
	virtual Op *rot_z(double angle) const { angle=angle; return copy(); }
	virtual Op *scale  (Xyz factor) const { factor=factor; return copy(); }
	virtual Op *scale  (double factor) const { factor=factor; return copy(); }
	virtual Op *scale_x(double factor) const { factor=factor; return copy(); }
	virtual Op *scale_y(double factor) const { factor=factor; return copy(); }
	virtual Op *scale_z(double factor) const { factor=factor; return copy(); }
	virtual Op *mult_matrix(const Matrix3 & m) const { return copy(); }
	virtual Op *mult_matrix(const Matrix4 & m) const { return copy(); }
	virtual int opcode() const { return 0; }
	};
/*...e*/
/*...sclass OpVertex:0:*/
class OpVertex : public Op
	{
public:
	Xyz xyz;
	OpVertex(const Xyz & xyz) : xyz(xyz) {}
	virtual ~OpVertex() {}
	virtual Op *copy() const { return new OpVertex(xyz); }
	virtual Op *trans  (Xyz t)    const { return new OpVertex(xyz+t); }
	virtual Op *trans_x(double t) const { return new OpVertex(Xyz(xyz.x+t,xyz.y,xyz.z)); }
	virtual Op *trans_y(double t) const { return new OpVertex(Xyz(xyz.x,xyz.y+t,xyz.z)); }
	virtual Op *trans_z(double t) const { return new OpVertex(Xyz(xyz.x,xyz.y,xyz.z+t)); }
	virtual Op *rot_x(double angle) const { return new OpVertex(xyz.rot_x(angle)); }
	virtual Op *rot_y(double angle) const { return new OpVertex(xyz.rot_y(angle)); }
	virtual Op *rot_z(double angle) const { return new OpVertex(xyz.rot_z(angle)); }
	virtual Op *scale  (Xyz factor)    const { return new OpVertex(scale_by(xyz,factor)); }
	virtual Op *scale  (double factor) const { return new OpVertex(xyz*factor); }
	virtual Op *scale_x(double t) const { return new OpVertex(Xyz(xyz.x*t,xyz.y,xyz.z)); }
	virtual Op *scale_y(double t) const { return new OpVertex(Xyz(xyz.x,xyz.y*t,xyz.z)); }
	virtual Op *scale_z(double t) const { return new OpVertex(Xyz(xyz.x,xyz.y,xyz.z*t)); }
	virtual Op *mult_matrix(const Matrix3 & m) const
		{ return new OpVertex(m * xyz); }
	virtual Op *mult_matrix(const Matrix4 & m) const
		{ return new OpVertex(m * xyz); }
	virtual int opcode() const { return 1; }
	};
/*...e*/
/*...sclass OpPolygon:0:*/
class OpPolygon : public Op
	{
public:
	int *vertexes;
	int n_vertexes;
	OpPolygon(int n_vertexes, const int vertexes[])
		:
		n_vertexes(n_vertexes)
		{
		OpPolygon::vertexes = new int [n_vertexes];
		for ( int i = 0; i < n_vertexes; i++ )
			OpPolygon::vertexes[i] = vertexes[i];
		}
	virtual ~OpPolygon() { delete[] vertexes; }
	virtual Op *copy() const { return new OpPolygon(n_vertexes, vertexes); }
	virtual Op *trans  (Xyz t) const { t=t; return copy(); }
	virtual Op *trans_x(double t) const { t=t; return copy(); }
	virtual Op *trans_y(double t) const { t=t; return copy(); }
	virtual Op *trans_z(double t) const { t=t; return copy(); }
	virtual Op *rot_x(double angle) const { angle=angle; return copy(); }
	virtual Op *rot_y(double angle) const { angle=angle; return copy(); }
	virtual Op *rot_z(double angle) const { angle=angle; return copy(); }
	virtual Op *scale  (Xyz factor) const { factor=factor; return copy(); }
	virtual Op *scale  (double factor) const { factor=factor; return copy(); }
	virtual Op *scale_x(double factor) const { factor=factor; return copy(); }
	virtual Op *scale_y(double factor) const { factor=factor; return copy(); }
	virtual Op *scale_z(double factor) const { factor=factor; return copy(); }
	virtual Op *mult_matrix(const Matrix3 & m) const { return copy(); }
	virtual Op *mult_matrix(const Matrix4 & m) const { return copy(); }
	virtual int opcode() const { return 2; }
	};
/*...e*/

/*...sPolygonOpList:0:*/
PolygonOpList::PolygonOpList() {}
/*...e*/
/*...sPolygonOpList \40\copy\41\:0:*/
PolygonOpList::PolygonOpList(const PolygonOpList & ol)
	{
	append(ol);
	}
/*...e*/
/*...soperator\61\:0:*/
PolygonOpList & PolygonOpList::operator=(const PolygonOpList & ol)
	{
	if ( &ol != this )
		{
		empty();
		append(ol);
		}
	return *this;
	}
/*...e*/
/*...soperator\43\\61\:0:*/
PolygonOpList & PolygonOpList::operator+=(const PolygonOpList & ol)
	{
	append(ol);
	return *this;
	}
/*...e*/
/*...s\126\PolygonOpList:0:*/
PolygonOpList::~PolygonOpList() { empty(); }
/*...e*/

/*...scol:0:*/
PolygonOpList & PolygonOpList::col(int index)
	{
	Op *op = new OpCol(index);
	if ( op != 0 )
		append(op);
	return *this;
	}
/*...e*/
/*...svertex:0:*/
PolygonOpList & PolygonOpList::vertex(const Xyz & xyz)
	{
	Op *op = new OpVertex(xyz);
	if ( op != 0 )
		append(op);
	return *this;
	}
/*...e*/
/*...spolygon:0:*/
// This gets a little monotonous, but hey, it makes it easy for the
// user of this class to draw things...

PolygonOpList & PolygonOpList::polygon(int v0, int v1, int v2)
	{
	int v[3];
	v[0] = v0; v[1] = v1; v[2] = v2;
	Op *op = new OpPolygon(3, v);
	if ( op != 0 )
		append(op);
	return *this;
	}

PolygonOpList & PolygonOpList::polygon(int v0, int v1, int v2, int v3)
	{
	int v[4];
	v[0] = v0; v[1] = v1; v[2] = v2; v[3] = v3;
	Op *op = new OpPolygon(4, v);
	if ( op != 0 )
		append(op);
	return *this;
	}

PolygonOpList & PolygonOpList::polygon(int v0, int v1, int v2, int v3, int v4)
	{
	int v[5];
	v[0] = v0; v[1] = v1; v[2] = v2; v[3] = v3; v[4] = v4;
	Op *op = new OpPolygon(5, v);
	if ( op != 0 )
		append(op);
	return *this;
	}

PolygonOpList & PolygonOpList::polygon(int v0, int v1, int v2, int v3, int v4, int v5)
	{
	int v[6];
	v[0] = v0; v[1] = v1; v[2] = v2; v[3] = v3; v[4] = v4; v[5] = v5;
	Op *op = new OpPolygon(6, v);
	if ( op != 0 )
		append(op);
	return *this;
	}

PolygonOpList & PolygonOpList::polygon(int v0, int v1, int v2, int v3, int v4, int v5, int v6)
	{
	int v[7];
	v[0] = v0; v[1] = v1; v[2] = v2; v[3] = v3; v[4] = v4; v[5] = v5; v[6] = v6;
	Op *op = new OpPolygon(7, v);
	if ( op != 0 )
		append(op);
	return *this;
	}

PolygonOpList & PolygonOpList::polygon(int v0, int v1, int v2, int v3, int v4, int v5, int v6, int v7)
	{
	int v[8];
	v[0] = v0; v[1] = v1; v[2] = v2; v[3] = v3; v[4] = v4; v[5] = v5; v[6] = v6; v[7] = v7;
	Op *op = new OpPolygon(8, v);
	if ( op != 0 )
		append(op);
	return *this;
	}

PolygonOpList & PolygonOpList::polygon(int n_vertexes, const int vertexes[])
	{
	Op *op = new OpPolygon(n_vertexes, vertexes);
	if ( op != 0 )
		append(op);
	return *this;
	}
/*...e*/

/*...strans:0:*/
PolygonOpList PolygonOpList::trans(Xyz t) const
	{
	PolygonOpList ol;
	for ( Op *op = first; op != 0; op = op->next )
		{
		Op *op2 = op->trans(t);
		if ( op2 == 0 )
			break;
		ol.append(op2);
		}
	return ol;
	}
/*...e*/
/*...strans_x:0:*/
PolygonOpList PolygonOpList::trans_x(double t) const
	{
	PolygonOpList ol;
	for ( Op *op = first; op != 0; op = op->next )
		{
		Op *op2 = op->trans_x(t);
		if ( op2 == 0 )
			break;
		ol.append(op2);
		}
	return ol;
	}
/*...e*/
/*...strans_y:0:*/
PolygonOpList PolygonOpList::trans_y(double t) const
	{
	PolygonOpList ol;
	for ( Op *op = first; op != 0; op = op->next )
		{
		Op *op2 = op->trans_y(t);
		if ( op2 == 0 )
			break;
		ol.append(op2);
		}
	return ol;
	}
/*...e*/
/*...strans_z:0:*/
PolygonOpList PolygonOpList::trans_z(double t) const
	{
	PolygonOpList ol;
	for ( Op *op = first; op != 0; op = op->next )
		{
		Op *op2 = op->trans_z(t);
		if ( op2 == 0 )
			break;
		ol.append(op2);
		}
	return ol;
	}
/*...e*/
/*...srot_x:0:*/
PolygonOpList PolygonOpList::rot_x(double angle) const
	{
	PolygonOpList ol;
	for ( Op *op = first; op != 0; op = op->next )
		{
		Op *op2 = op->rot_x(angle);
		if ( op2 == 0 )
			break;
		ol.append(op2);
		}
	return ol;
	}
/*...e*/
/*...srot_y:0:*/
PolygonOpList PolygonOpList::rot_y(double angle) const
	{
	PolygonOpList ol;
	for ( Op *op = first; op != 0; op = op->next )
		{
		Op *op2 = op->rot_y(angle);
		if ( op2 == 0 )
			break;
		ol.append(op2);
		}
	return ol;
	}
/*...e*/
/*...srot_z:0:*/
PolygonOpList PolygonOpList::rot_z(double angle) const
	{
	PolygonOpList ol;
	for ( Op *op = first; op != 0; op = op->next )
		{
		Op *op2 = op->rot_z(angle);
		if ( op2 == 0 )
			break;
		ol.append(op2);
		}
	return ol;
	}
/*...e*/
/*...sscale:0:*/
PolygonOpList PolygonOpList::scale(Xyz factor) const
	{
	PolygonOpList ol;
	for ( Op *op = first; op != 0; op = op->next )
		{
		Op *op2 = op->scale(factor);
		if ( op2 == 0 )
			break;
		ol.append(op2);
		}
	return ol;
	}
/*...e*/
/*...sscale:0:*/
PolygonOpList PolygonOpList::scale(double factor) const
	{
	PolygonOpList ol;
	for ( Op *op = first; op != 0; op = op->next )
		{
		Op *op2 = op->scale(factor);
		if ( op2 == 0 )
			break;
		ol.append(op2);
		}
	return ol;
	}
/*...e*/
/*...sscale_x:0:*/
PolygonOpList PolygonOpList::scale_x(double factor) const
	{
	PolygonOpList ol;
	for ( Op *op = first; op != 0; op = op->next )
		{
		Op *op2 = op->scale_x(factor);
		if ( op2 == 0 )
			break;
		ol.append(op2);
		}
	return ol;
	}
/*...e*/
/*...sscale_y:0:*/
PolygonOpList PolygonOpList::scale_y(double factor) const
	{
	PolygonOpList ol;
	for ( Op *op = first; op != 0; op = op->next )
		{
		Op *op2 = op->scale_y(factor);
		if ( op2 == 0 )
			break;
		ol.append(op2);
		}
	return ol;
	}
/*...e*/
/*...sscale_z:0:*/
PolygonOpList PolygonOpList::scale_z(double factor) const
	{
	PolygonOpList ol;
	for ( Op *op = first; op != 0; op = op->next )
		{
		Op *op2 = op->scale_z(factor);
		if ( op2 == 0 )
			break;
		ol.append(op2);
		}
	return ol;
	}
/*...e*/
/*...smult_matrix:0:*/
PolygonOpList PolygonOpList::mult_matrix(const Matrix3 & m) const
	{
	PolygonOpList ol;
	for ( Op *op = first; op != 0; op = op->next )
		{
		Op *op2 = op->mult_matrix(m);
		if ( op2 == 0 )
			break;
		ol.append(op2);
		}
	return ol;
	}
/*...e*/
/*...smult_matrix:0:*/
PolygonOpList PolygonOpList::mult_matrix(const Matrix4 & m) const
	{
	PolygonOpList ol;
	for ( Op *op = first; op != 0; op = op->next )
		{
		Op *op2 = op->mult_matrix(m);
		if ( op2 == 0 )
			break;
		ol.append(op2);
		}
	return ol;
	}
/*...e*/

/*...srender:0:*/
// Scan along the opcode list.
// Reject back-facing polygons.
// Obtain coordinates in canonical view volume.
// Do polygon clipped against canonical view volume.
// Compute a 'midpoint', as average of vertex coords.
// Midpoint will be somewhere close to centre of polygon.
// Use midpoint for lighting coefficient calculation.
// Add polygons to polygons list.
// Use BSP tree technique to draw/split polygons in order.
// Draw polygons, utilising services of Bitmap class.

#define	WRAP(n) ((n)&(N_POLYGON_VERTEXES-1))

/*...sclass Poly:0:*/
#define	STEP 10

class Poly
	{
	void enlarge();
	int n_vertexes_alloc;
public:
	Poly *next;
	Xyz normal;
	double d;
	int index;
	int n_vertexes;
	Xyz *vertexes;
	Poly()
		:
		n_vertexes(0),
		n_vertexes_alloc(STEP)
		{
		vertexes = new Xyz [n_vertexes_alloc];
		}
	Poly(int n_vertexes, const int vertexes[], const Xyz vs[], int v)
		:
		n_vertexes(n_vertexes),
		n_vertexes_alloc(n_vertexes)
		{
		Poly::vertexes = new Xyz [n_vertexes_alloc];
		for ( int i = 0; i < n_vertexes; i++ )
			Poly::vertexes[i] = vs[WRAP(v-vertexes[i])];
		}
	Poly(int n_vertexes, Xyz *vertexes)
		:
		n_vertexes(n_vertexes),
		n_vertexes_alloc(n_vertexes),
		vertexes(vertexes)
		{
		}
	~Poly()
		{
		delete[] vertexes;
		}
	void append(const Xyz & vertex)
		{
		if ( n_vertexes == n_vertexes_alloc )
			enlarge();
		vertexes[n_vertexes++] = vertex;
		}
	};

void Poly::enlarge()
	{
	Xyz *p = new Xyz [n_vertexes_alloc + STEP];
	for ( int i = 0; i < n_vertexes; i++ )
		p[i] = vertexes[i];
	delete[] vertexes;
	vertexes = p;
	n_vertexes_alloc += STEP;
	}

#undef STEP
/*...e*/
/*...scanonical_poly:0:*/
// Clip to the canonical perspective view volume.
//
// Visible points have :-
//   z >= zmin (near plane)
//   -1.0 <= x/z <= 1.0  or  -z <= x <= z
//   -1.0 <= y/z <= 1.0  or  -z <= y <= z
//
// Any point on the line from p0 to p1 may be described as
//   p0 + t * (p1-p0),  where 0.0 <= t <= 1.0
//
// Intersecting p0+t*dir with ax+by+cz+d==0 gives :-
//   t = -(a*p0.x+b*p0.y+c*p0.z+d)/(a*dir.x+b*dir.y+c*dir.z)
//
// Note that tiny inaccuracies in the arithmetic can result in t having
// values outside the 0.0 to 1.0 range. We handle these specially as this
// has been seen to be a problem otherwise.
//
// Clipping planes, a, b, c, d,     t=
//   zmin <= z      0  0  1  -zmin  -( p0.z-zmin)/dir.z
//   -z <= x        1  0  1  0      -( p0.x+p0.z)/( dir.x+dir.z)
//   x <= z        -1  0  1  0      -(-p0.x+p0.z)/(-dir.x+dir.z)
//   -z <= y        0  1  1  0      -( p0.y+p0.z)/( dir.y+dir.z)
//   y <= z         0 -1  1  0      -(-p0.y+p0.z)/(-dir.y+dir.z)
//
// Technique for this code stolen from the Laing Barsky line clipper.

/*...sz_ge_zmin:0:*/
#define	NEARPLANE (1e-10)

static Boolean test_z_ge_zmin(const Xyz & p) { return p.z >= NEARPLANE; }

static Xyz clip_z_ge_zmin(const Xyz & p0, const Xyz & p1)
	{
	double dz = p1.z-p0.z;
	double t = (NEARPLANE-p0.z)/dz;
	if ( t <= 0.0 ) return p0;
	if ( t >= 1.0 ) return p1;
	double dx = p1.x-p0.x, dy = p1.y-p0.y;
	return Xyz(p0.x+t*dx, p0.y+t*dy, NEARPLANE);
	}
/*...e*/
/*...sx_ge_mz:0:*/
static Boolean test_x_ge_mz(const Xyz & p) { return -p.z <= p.x; }

static Xyz clip_x_ge_mz(const Xyz & p0, const Xyz & p1)
	{
	double dx = p1.x-p0.x, dz = p1.z-p0.z;
	double t = -(p0.x+p0.z)/(dx+dz);
	if ( t <= 0.0 ) return p0;
	if ( t >= 1.0 ) return p1;
	double dy = p1.y-p0.y;
	return Xyz(p0.x+t*dx, p0.y+t*dy, p0.z+t*dz);
	}
/*...e*/
/*...sx_le_z:0:*/
static Boolean test_x_le_z(const Xyz & p) { return p.x <= p.z; }

static Xyz clip_x_le_z(const Xyz & p0, const Xyz & p1)
	{
	double dx = p1.x-p0.x, dz = p1.z-p0.z;
	double t = -(-p0.x+p0.z)/(-dx+dz);
	if ( t <= 0.0 ) return p0;
	if ( t >= 1.0 ) return p1;
	double dy = p1.y-p0.y;
	return Xyz(p0.x+t*dx, p0.y+t*dy, p0.z+t*dz);
	}
/*...e*/
/*...sy_ge_mz:0:*/
static Boolean test_y_ge_mz(const Xyz & p) { return -p.z <= p.y; }

static Xyz clip_y_ge_mz(const Xyz & p0, const Xyz & p1)
	{
	double dy = p1.y-p0.y, dz = p1.z-p0.z;
	double t = -(p0.y+p0.z)/(dy+dz);
	if ( t <= 0.0 ) return p0;
	if ( t >= 1.0 ) return p1;
	double dx = p1.x-p0.x;
	return Xyz(p0.x+t*dx, p0.y+t*dy, p0.z+t*dz);
	}
/*...e*/
/*...sy_le_z:0:*/
static Boolean test_y_le_z(const Xyz & p) { return p.y <= p.z; }

static Xyz clip_y_le_z(const Xyz & p0, const Xyz & p1)
	{
	double dy = p1.y-p0.y, dz = p1.z-p0.z;
	double t = -(-p0.y+p0.z)/(-dy+dz);
	if ( t <= 0.0 ) return p0;
	if ( t >= 1.0 ) return p1;
	double dx = p1.x-p0.x;
	return Xyz(p0.x+t*dx, p0.y+t*dy, p0.z+t*dz);
	}
/*...e*/

/*...splane_clip:0:*/
static void plane_clip(
	const Xyz p [], int   n ,
	      Xyz p2[], int & n2,
	Boolean (*test)(const Xyz & p),
	Xyz     (*clip)(const Xyz & p0, const Xyz & p1)
	)
	{
	n2 = 0;
	if ( n == 0 )
		return;
	int i, prev_i = n-1;
	Boolean in, prev_in = test(p[prev_i]);
	for ( i	= 0; i < n; prev_in = in, prev_i = i++ )
		// Consider the line segment from vertex prev_i to i
		{
		in = test(p[i]);
		if ( prev_in )
			{
			p2[n2++] = p[prev_i];
			if ( !in )
				p2[n2++] = clip(p[prev_i], p[i]);
			}
		else
			{
			if ( in )
				p2[n2++] = clip(p[prev_i], p[i]);
			}	
		}
	}
/*...e*/

static Poly *canonical_poly(const Poly & poly)
	{
	Xyz *v2 = new Xyz[poly.n_vertexes+5];
	Xyz *v3 = new Xyz[poly.n_vertexes+5];
	int n2, n3;
	plane_clip(poly.vertexes, poly.n_vertexes, v2, n2, test_z_ge_zmin, clip_z_ge_zmin);
	plane_clip(v2, n2, v3, n3, test_x_ge_mz, clip_x_ge_mz);
	plane_clip(v3, n3, v2, n2, test_x_le_z , clip_x_le_z );
	plane_clip(v2, n2, v3, n3, test_y_ge_mz, clip_y_ge_mz);
	plane_clip(v3, n3, v2, n2, test_y_le_z , clip_y_le_z );
	delete[] v3;
	if ( n2 > 2 )
		return new Poly(n2, v2);
	else
		// Polygons which are single points or
		// edge on lines are 'razor' thin and don't count.
		// We'll need 3 vertexes for normal calculation later.
		{
		delete[] v2;
		return 0;
		}
	}

/*...ssafety checks:0:*/
#ifdef SAFETY

// After clipping to the canonical view volume, all points should pass the
// following test. Also, all points on all lines between pairs of such points
// should also pass. Notice the use of EPSILON to allow a small amount of
// inaccuracy in the floating point arithmetic. Anyhow, new points, formed
// by dividing lines between canonical points, should also pass the test.

#define	EPSILON (NEARPLANE/2)	// Must be smaller than NEARPLANE,
				// and NEARPLANE is pretty small to start with

inline void check_cvv(const Xyz & p)
	{
	if (  p.z >= NEARPLANE - EPSILON &&
	     -p.z <= p.x + EPSILON &&
	      p.x <= p.z + EPSILON &&
	     -p.z <= p.y + EPSILON &&
	      p.y <= p.z + EPSILON )
		;
	else
		insane("point outside canonical view volume");
	}

static void check_cvv(const Xyz p[], int len)
	{
	for ( int i = 0; i < len; i++ )
		check_cvv(p[i]);
	}

inline void check_cvv(const Poly *poly)
	{ check_cvv(poly->vertexes, poly->n_vertexes); }	

#endif
/*...e*/
/*...e*/
/*...srender_polys:0:*/
// Render a polygon list, freeing it up as we go
//
// This is essentially the classic recursive BSP algorithm.
// However a tree is not built and then displayed, its all displayed
// as the recursive subdivision proceeds. Everything is computed in
// viewpoint relative coordinates, and then perspective transformed
// and mapped to the viewport at the very end.
//
// Stack requirement for one level of recursion is approximately 100 bytes,
// on a machine which doesn't require have a minimum stack frame requirement
// (such as RS/6000, where things could be a lot worse).

/*...stest_plane and clip_plane:0:*/
// Determine whether point is behind the plane of the polygon or not.

inline Boolean test_plane(const Poly *poly, const Xyz & p)
	{
	return poly->normal.x * p.x +
	       poly->normal.y * p.y +
	       poly->normal.z * p.z +
	       poly->d <= 0;
	}

// Any point on the line from p0 to p1 may be described as
//   p0 + t * (p1-p0),  where 0.0 <= t <= 1.0
//
// Intersecting p0+t*dir with ax+by+cz+d==0 gives :-
//   t = -(a*p0.x+b*p0.y+c*p0.z+d)/(a*dir.x+b*dir.y+c*dir.z)
//
// I have seen the denominator be 0.0, which is indicative of a line
// being parallel to the plane, which means that the test for coplanarity
// isn't catching this case, perhaps due to arithmetic inaccuracies.
//
// Note that tiny inaccuracies in the arithmetic can result in t having
// values outside the 0.0 to 1.0 range. We shall handle these specially,
// as this has seen to be a problem.

inline Xyz clip_plane(const Poly *poly, const Xyz & p0, const Xyz & p1)
	{
	double dx = p1.x-p0.x, dy = p1.y-p0.y, dz=p1.z-p0.z;
	double den = ( poly->normal.x*dx +
		       poly->normal.y*dy +
		       poly->normal.z*dz );
	if ( den == 0.0 )
		return p1; // Return anything I suppose
	double num = ( poly->normal.x*p0.x +
		       poly->normal.y*p0.y +
		       poly->normal.z*p0.z +
		       poly->d               );
	double t = -num/den;
	if ( t <= 0.0 ) return p0;
	if ( t >= 1.0 ) return p1;
	return Xyz(p0.x+t*dx, p0.y+t*dy, p0.z+t*dz);
	}
/*...e*/
/*...ssplit_poly:0:*/
// In this code, 'in', means one side of the reference polygon,
// and 'out' means the other.

static void split_poly(
	const Xyz p [], int   n ,
	      Xyz p2[], int & n2,
	      Xyz p3[], int & n3,
	const Poly *poly
	)
	{
	n2 = n3 = 0;
	if ( n == 0 )
		return;
	int i, prev_i = n-1;
	Boolean in, prev_in = test_plane(poly, p[prev_i]);
	for ( i	= 0; i < n; prev_in = in, prev_i = i++ )
		// Consider the line segment from vertex prev_i to i
		{
		in = test_plane(poly, p[i]);
		if ( prev_in )
			{
			p2[n2++] = p[prev_i];
			if ( !in )
				{
				p2[n2++] = p3[n3++] = clip_plane(poly, p[prev_i], p[i]);
/*...ssafety:32:*/
#ifdef SAFETY
check_cvv(p2[n2-1]);
#endif
/*...e*/
				}
			}
		else
			{
			p3[n3++] = p[prev_i];
			if ( in )
				{
				p2[n2++] = p3[n3++] = clip_plane(poly, p[prev_i], p[i]);
/*...ssafety:32:*/
#ifdef SAFETY
check_cvv(p2[n2-1]);
#endif
/*...e*/
				}
			}
		}
	}
/*...e*/
/*...ssanitise:0:*/
// Try to simplify polygons by eliminating vertexes that are virtually
// on top of one another. This shouldn't cause any visible differences,
// as CLOSE is defined tiny enough not to be visible at the pixel level,
// but large enough to cover arithmetic inaccuracies which cause coincident
// vertexes in the first place.
//
// Also, try to shrink the allocation size of the vertex array from the
// massive initial allocation, to something closer to that which is needed.

#define	CLOSE (1e-10)

inline Boolean distinct(const Xyz & p0, const Xyz & p1)
	{
	return fabs(p0.x-p1.x) > CLOSE ||
	       fabs(p0.y-p1.y) > CLOSE ||
	       fabs(p0.z-p1.z) > CLOSE;
	}

inline void sanitise(Xyz * (&src), int & nv)
	{
	if ( nv > 2 )
		{
		Xyz *dst = new Xyz [nv];
		int i, prev_i = nv - 1, n = 0;
		for ( i = 0; i < nv; prev_i = i++ )
			if ( distinct(src[prev_i], src[i]) )
				dst[n++] = src[i];
		nv = n;
		delete[] src;
		src = dst;
		}
	}
/*...e*/
/*...scoplanar:0:*/
// For some reason the numerical accuraccy on the iPhone isn't as good
// as most other places, so we need to be more tolerant of error.

#ifdef IPHONE
  #define VERY_CLOSE (1e-14)
#else
  #define VERY_CLOSE (1e-15)
#endif

inline Boolean coplanar(const Poly & p0, const Poly & p1)
	{
	return fabs(p0.normal.x-p1.normal.x) < VERY_CLOSE &&
	       fabs(p0.normal.y-p1.normal.y) < VERY_CLOSE &&
	       fabs(p0.normal.z-p1.normal.z) < VERY_CLOSE &&
	       fabs(p0.d       -p1.d       ) < VERY_CLOSE;
	}
/*...e*/

static void render_polys(
	Poly *polys,
	const Matrix4 & to_port,
	Bitmap & bitmap
	)
	{
	Poly *b = 0, *a = 0; // Before and after polygon lists
	Poly *c = polys; // Coplanar polygon list
	Poly *m = polys; // Middle poly (for BSP splitting against)
	polys = polys->next; // Consider other polygons
	c->next = 0; // Initially coplanar list has only m on it

	while ( polys != 0 )
		{
		Poly *p = polys; polys = polys->next;
		if ( coplanar(*p, *m) )
			{ p->next = c; c = p; }
		else
/*...ssplit into 2 perhaps:24:*/
{
Xyz *bv = new Xyz [p->n_vertexes*2]; int nbv;
Xyz *av = new Xyz [p->n_vertexes*2]; int nav;
split_poly(p->vertexes, p->n_vertexes,
	  bv, nbv, av, nav, m);
/*...ssafety:24:*/
#ifdef SAFETY
if ( nbv > p->n_vertexes*2 || nav > p->n_vertexes*2 )
	insane("too many vertexes generated by split");
#endif
/*...e*/
sanitise(bv, nbv);
sanitise(av, nav);
if ( nbv > 2 )
	{
	Poly *bp = new Poly(nbv, bv);
	bp->normal = p->normal;
	bp->d      = p->d;
	bp->index  = p->index;
	bp->next = b; b = bp;
	}
else
	delete[] bv;
if ( nav > 2 )
	{
	Poly *ap = new Poly(nav, av);
	ap->normal = p->normal;
	ap->d      = p->d;
	ap->index  = p->index;
	ap->next = a; a = ap;
	}
else
	delete[] av;
delete p;
}
/*...e*/
		}

	if ( b != 0 )
		render_polys(b, to_port, bitmap);

	do
		{
		Poly *d = c; c = c->next; // Poly to draw
		int *xs = new int [d->n_vertexes];
		int *ys = new int [d->n_vertexes];
		for ( int i = 0; i < d->n_vertexes; i++ )
			{
			Xyz v(to_port * d->vertexes[i]);
			xs[i] = (int) ( v.x + 0.5 );
			ys[i] = (int) ( v.y + 0.5 );
			}
		bitmap.polygon(xs, ys, d->n_vertexes, d->index);
		delete[] ys;
		delete[] xs;
		delete d;
		}
	while ( c != 0 );

	if ( a != 0 )
		render_polys(a, to_port, bitmap);
	}
/*...e*/

Boolean PolygonOpList::render(
	const View & view, Bitmap & bitmap,
	const Xyz & light,
	double max_shade
	) const
	{
	Xyz ulight(light.unit());
	Xyz ws[N_POLYGON_VERTEXES];
	Xyz vs[N_POLYGON_VERTEXES];
	Boolean mapped[N_POLYGON_VERTEXES];
	int v = -1;
	int w = bitmap.width();
	int h = bitmap.height();
	if ( w < 1 || h < 1 )
		return FALSE;
	Matrix4 to_view(view.to_view_volume());
	int index = 0; // Start with colour 0
	Poly *polys = 0;
	for ( Op *op = first; op != 0; op = op->next )
		switch ( op->opcode() )		
			{
/*...s0 \45\ set colour:24:*/
case 0:
	{
	OpCol *op2 = (OpCol *) op;
	index = op2->index;
	}
	break;
/*...e*/
/*...s1 \45\ define vertex:24:*/
case 1:
	{
	OpVertex *op2 = (OpVertex *) op;
	++v;
	ws[WRAP(v)] = op2->xyz;
	mapped[WRAP(v)] = FALSE;
	}
	break;
/*...e*/
/*...s2 \45\ define polygon:24:*/
case 2:
	{
	int i;
	OpPolygon *op2 = (OpPolygon *) op;
	Xyz p0(ws[WRAP(v-op2->vertexes[0])]);
	Xyz p1(ws[WRAP(v-op2->vertexes[1])]);
	Xyz p2(ws[WRAP(v-op2->vertexes[2])]);
	Xyz normal((p1-p0)*(p2-p1));
	if ( scalar_product(normal, view.eye-p0) > 0.0 )
		// Polygons normal points towards eye
		{

		// Ensure we have viewpoint relative coordinates,
		// in the canonical view volume
		for ( i = 0; i < op2->n_vertexes; i++ )
			{
			int inx = WRAP(v-op2->vertexes[i]);
			if ( !mapped[inx] )
				{
				vs[inx] = to_view * ws[inx];
				mapped[inx] = TRUE;
				}
			}

		// Generate a clipped polygon
		Poly poly(op2->n_vertexes, op2->vertexes, vs, v);
		Poly *poly2;
		if ( (poly2 = canonical_poly(poly)) != 0 )
			// Polygon is visible after clipping
			// to canonical view volume
			{
			// Use the unit light direction vector to
			// calculate the shade of the colour
			double kd = scalar_product(normal.unit(), ulight);
			if ( kd > 0.0 )
				poly2->index = index + (int) (max_shade * kd);
			else
				poly2->index = index;

#ifdef WIN32
			// MSVC 5.0/6.0 complain if I use references
			// I can't think of justification for this
			const Xyz   q0(poly2->vertexes[0]);
			const Xyz   q1(poly2->vertexes[1]);
			const Xyz   q2(poly2->vertexes[2]);
#else
			const Xyz & q0(poly2->vertexes[0]);
			const Xyz & q1(poly2->vertexes[1]);
			const Xyz & q2(poly2->vertexes[2]);
#endif
			poly2->normal = ( (q1-q2)*(q0-q1) ).unit();
			poly2->d = - ( poly2->normal.x * q0.x +
				       poly2->normal.y * q0.y +
				       poly2->normal.z * q0.z );
			poly2->next = polys; polys = poly2;
			}
		}
	}
	break;
/*...e*/
			}

	if ( polys != 0 )
		// Everything is in canonical view volume
		// All polygons are front facing
		// Polygons are known in ax+by+cz+d==0 form
		render_polys(polys, to_viewport(w,h), bitmap);

	return TRUE;
	}
/*...e*/
