//
// polygon.h - Support for Polygon rendering
//
// Wherever the word 'polygon' is used assume 'convex polygon' unless
// otherwise specified.
//
// The intention is to allow the renderer to be able to generate a much simpler
// polygon rendered version of their model, which hopefully can be rendered up
// in smaller times. Thus the geometry may be checked for consistency and
// asthetic value before paying the price of a full raytrace.
//
// We shall implement colours, and diffuse shading, based on a single light
// source. Thus we shall make assumptions about the layout of the palette.
// eg: Palette indexes 0..15 can be increasing shades of one colour, 16..31
// can be increasing shades of another colour, if max_shade=15.0 etc...
//
// When defining a polygon, the vertexes should be listed in anti-clockwise
// order, so as to make the normal vectors point out of the object of which
// the polygon is a part. Also, all vertexes should be co-planar. As it is
// acknowledged that virtually all polygons will have 8 sides or much fewer,
// convenient inline overloaded polygon methods are provided.
//
// No polygon may be created with more than N_POLYGON_VERTEXES vertexes.
//

#ifndef POLYGON_H
#define	POLYGON_H

/*...sincludes:0:*/
#include "types.h"
#include "xyz.h"
#include "rgb.h"
#include "matrix.h"
#include "bitmap.h"
#include "view.h"
#include "oplist.h"

/*...vtypes\46\h:0:*/
/*...vxyz\46\h:0:*/
/*...vrgb\46\h:0:*/
/*...vmatrix\46\h:0:*/
/*...vbitmap\46\h:0:*/
/*...vview\46\h:0:*/
/*...voplist\46\h:0:*/
/*...e*/

#define	N_POLYGON_VERTEXES 32 // must be a power of 2

class PolygonOpList : public OpList
	{
public:
	PolygonOpList();
	PolygonOpList(const PolygonOpList & ol);
	PolygonOpList & operator=(const PolygonOpList & ol);
	PolygonOpList & operator+=(const PolygonOpList & ol);
	~PolygonOpList();

	// Actual drawing commands
	PolygonOpList & col(int index);
	PolygonOpList & vertex(const Xyz & xyz);
	PolygonOpList & polygon(int v0, int v1, int v2);
	PolygonOpList & polygon(int v0, int v1, int v2, int v3);
	PolygonOpList & polygon(int v0, int v1, int v2, int v3, int v4);
	PolygonOpList & polygon(int v0, int v1, int v2, int v3, int v4, int v5);
	PolygonOpList & polygon(int v0, int v1, int v2, int v3, int v4, int v5, int v6);
	PolygonOpList & polygon(int v0, int v1, int v2, int v3, int v4, int v5, int v6, int v7);
	PolygonOpList & polygon(int n_vertexes, const int vertexes[]);

	// Manipulating coordinates within lists
	PolygonOpList trans  (Xyz t) const;
	PolygonOpList trans_x(double t) const;
	PolygonOpList trans_y(double t) const;
	PolygonOpList trans_z(double t) const;
	PolygonOpList rot_x(double angle) const;
	PolygonOpList rot_y(double angle) const;
	PolygonOpList rot_z(double angle) const;
	PolygonOpList scale  (Xyz factor) const;
	PolygonOpList scale  (double factor) const;
	PolygonOpList scale_x(double factor) const;
	PolygonOpList scale_y(double factor) const;
	PolygonOpList scale_z(double factor) const;
	PolygonOpList mult_matrix(const Matrix3 & m) const;
	PolygonOpList mult_matrix(const Matrix4 & m) const;

	// Render to a bitmap
	Boolean render(
		const View & view,
		Bitmap & bitmap,
		const Xyz & light,
		double max_shade = 15.0
		) const;
	};

#endif
