//
// progwin.C - Implement ProgressWin using a Windows window
//

/*...sincludes:0:*/
#include <cstdio>
#include <cstdlib>
#include <cstring>
#include <ctime>

#ifndef NO_STDNS
  using namespace std;
#endif

#include <process.h>

#define	BOOLEAN BOOLEANx
#define	WIN32_LEAN_AND_MEAN
#include <windows.h>
#undef BOOLEAN

extern "C" {
#include "gbm.h"
#include "gbmerr.h"
}

#include "bitmap.h"
#include "progwin.h"

/*...vbitmap\46\h:0:*/
/*...vprogwin\46\h:0:*/
/*...e*/

static HINSTANCE hinst;
static int cxScreen, cyScreen, cyCaption, cxFixedFrame, cyFixedFrame;
static int iBppErrDiff;
static GBMRGB gbmrgbErrDiff[0x100];

/*...sstr_of_time:0:*/
static char *str_of_time(time_t t, char *buf)
	{
	strcpy(buf, ctime(&t));
	buf[19] = '\0';
	return buf + 11;
	}
/*...e*/
/*...sclass ProgressWinInternal:0:*/
#define	WC_PROGWIN "ProgressWindow"

#define	N_MESSAGES 8

class ProgressWinInternal
	{
public:
	HANDLE hmutex;        // Control access to an instance of this class
	HANDLE heventCreated; // Signalled when window initialised etc.
	HANDLE heventUpdated; // Signalled when update window done
	HANDLE heventDeleted; // Signalled when window destroyed
	ProgressWin::WinFlags wf;

	HWND hwnd;
	BITMAPINFOHEADER bmih;
	RGBQUAD rgbqPalette[0x100]; // Must immediately follow
	HBITMAP hbitmap;
	HPALETTE hpalette;
	char wtitle[200+1];
	Boolean window_closed;

	int inx_message;
	char *messages[N_MESSAGES];
	int percent;
	GBM *gbm; Byte *data24, *data;
	time_t t_started, t_fin;
	int iTextMode;
	Boolean show_messages;
	Boolean show_progress;

	ProgressWinInternal(Bitmap & b, const char *title, ProgressWin::WinFlags wf);
	~ProgressWinInternal();

	void redraw();
	void begin();
	void done(int sofar, int total);
	void end();
	void message(const char *message);
	};

/*...sProgressWinWindowProc:0:*/
#define	WM_UPDATEWINDOW  WM_USER
#define	WM_DELETEWINDOW (WM_USER+1)

/*...sMakePalette:0:*/
static HPALETTE MakePalette(GBMRGB *gbmrgb, unsigned nCols)
	{
	LOGPALETTE *pal = (LOGPALETTE *) malloc(sizeof(LOGPALETTE)+nCols*sizeof(PALETTEENTRY));
	pal->palNumEntries = nCols;
	pal->palVersion = 0x300;
	for ( unsigned i = 0; i < nCols; i++ )
		{
		pal->palPalEntry[i].peRed   = (BYTE) gbmrgb[i].r;
		pal->palPalEntry[i].peGreen = (BYTE) gbmrgb[i].g;
		pal->palPalEntry[i].peBlue  = (BYTE) gbmrgb[i].b;
		pal->palPalEntry[i].peFlags = (BYTE) 0;
		}
	HPALETTE hpal = CreatePalette(pal);
	free(pal);
	return hpal;
	}
/*...e*/
/*...sRealizeOurPalette:0:*/
static void RealizeOurPalette(HWND hwnd, HPALETTE hpalette, BOOL bForceBackground)
	{
	HDC hdc = GetWindowDC(hwnd);
	HPALETTE hpaletteOld = SelectPalette(hdc, hpalette, FALSE);
	UINT nChanged = RealizePalette(hdc);
	SelectPalette(hdc, hpaletteOld, TRUE);
	ReleaseDC(hwnd, hdc);
	if ( nChanged != 0 )
		// Palette changes occured, we should repaint all
		// so as to get the best rendition to the new palette
		InvalidateRect(hwnd, NULL, TRUE);
	}
/*...e*/

// was FAR PASCAL
LRESULT CALLBACK ProgressWinWindowProc(
	HWND hwnd,
	unsigned message,
	WPARAM wParam,
	LPARAM lParam
	)
	{
	switch ( message )
		{
/*...sWM_CREATE          \45\ create:16:*/
case WM_CREATE:
	{
	CREATESTRUCT *cs = (CREATESTRUCT *) lParam;
	ProgressWinInternal *pwi = (ProgressWinInternal *) cs->lpCreateParams;
	SetWindowLongPtr(hwnd, 0, (LONG_PTR) pwi);
	memset(&(pwi->bmih), 0, sizeof(pwi->bmih));
	pwi->bmih.biSize        = sizeof(pwi->bmih);
	pwi->bmih.biWidth       = pwi->gbm->w;
	pwi->bmih.biHeight      = pwi->gbm->h;
	pwi->bmih.biPlanes      = 1;
	pwi->bmih.biBitCount    = iBppErrDiff;
	pwi->bmih.biCompression = BI_RGB;
	for ( unsigned i = 0; i < 0x100; i++ )
		{
		pwi->rgbqPalette[i].rgbRed      = (BYTE) gbmrgbErrDiff[i].r;
		pwi->rgbqPalette[i].rgbGreen    = (BYTE) gbmrgbErrDiff[i].g;
		pwi->rgbqPalette[i].rgbBlue     = (BYTE) gbmrgbErrDiff[i].b;
		pwi->rgbqPalette[i].rgbReserved = 0;
		}
	switch ( iBppErrDiff )
		{
		case 8:
			pwi->hpalette = MakePalette(gbmrgbErrDiff, 7*8*4);
			break;
		default:
			pwi->hpalette = (HPALETTE) NULL;
			break;
		}
	HDC hdcScreen = GetWindowDC((HWND) NULL);
	pwi->hbitmap = CreateDIBSection(
		hdcScreen,
		(BITMAPINFO *) &(pwi->bmih),
		DIB_RGB_COLORS,
		NULL,
		(HANDLE) NULL,
		0);
	ReleaseDC((HWND) NULL, hdcScreen);
	}
	return 0;
/*...e*/
/*...sWM_PAINT           \45\ paint client:16:*/
case WM_PAINT:
	{
	PAINTSTRUCT ps;
	HDC hdc = BeginPaint(hwnd, &ps);

	RECT rcClient;
	GetClientRect(hwnd, &rcClient);
	int cxClient = rcClient.right  - rcClient.left;
	int cyClient = rcClient.bottom - rcClient.top ;

	TEXTMETRIC tm;
	GetTextMetrics(hdc, &tm);
	int xChar = tm.tmAveCharWidth;
	int yChar = tm.tmHeight + tm.tmExternalLeading;

	ProgressWinInternal *pwi = (ProgressWinInternal *) GetWindowLongPtr(hwnd, 0);

	WaitForSingleObject(pwi->hmutex, INFINITE);

	HDC hdcCompatible = CreateCompatibleDC(hdc);
	HBITMAP hbitmapOld = (HBITMAP) SelectObject(hdcCompatible, pwi->hbitmap);

	HPALETTE hpaletteOld;
	if ( pwi->hpalette != (HPALETTE) NULL )
		hpaletteOld = SelectPalette(hdc, pwi->hpalette, FALSE);

	pwi->bmih.biHeight = pwi->gbm->h;
	SetDIBits(hdcCompatible, pwi->hbitmap,
		0, pwi->gbm->h, pwi->data,
		(CONST BITMAPINFO *) &(pwi->bmih), DIB_RGB_COLORS);

	BitBlt(hdc, 0, 0, pwi->gbm->w, pwi->gbm->h,
		hdcCompatible, 0, 0, SRCCOPY);

	if ( pwi->hpalette != (HPALETTE) NULL )
		SelectPalette(hdc, hpaletteOld, TRUE);

	SelectObject(hdcCompatible, hbitmapOld);
	DeleteDC(hdcCompatible);

#define	N_TEXTMODES 4
	switch ( pwi->iTextMode )
		{
		case 0:
			SetTextColor(hdc, RGB(255,255,255));
			SetBkMode(hdc, TRANSPARENT);
			break;
		case 1:
			SetTextColor(hdc, RGB(0,0,0));
			SetBkMode(hdc, TRANSPARENT);
			break;
		case 2:
			SetTextColor(hdc, RGB(255,255,255));
			SetBkColor(hdc, RGB(0,0,0));
			SetBkMode(hdc, OPAQUE);
			break;
		case 3:
			SetTextColor(hdc, RGB(0,0,0));
			SetBkColor(hdc, RGB(255,255,255));
			SetBkMode(hdc, OPAQUE);
			break;
		}

	char s[100+1], buf[40+1];
	SIZE sz;

	if ( pwi->show_progress )
		{
		if ( pwi->t_started != 0 )
			{
			sprintf(s, "Start %s", str_of_time(pwi->t_started, buf));
			GetTextExtentPoint(hdc, s, strlen(s), &sz);
			TextOut(hdc, cxClient-sz.cx, 0, s, strlen(s));
			}
		if ( pwi->t_fin != 0 )
			{
			sprintf(s, "Ready %s", str_of_time(pwi->t_fin, buf));
			GetTextExtentPoint(hdc, s, strlen(s), &sz);
			TextOut(hdc, cxClient-sz.cx, yChar, s, strlen(s));
			}
		sprintf(s, "%d%% done", pwi->percent);
		GetTextExtentPoint(hdc, s, strlen(s), &sz);
		TextOut(hdc, cxClient-sz.cx, 2*yChar, s, strlen(s));
		}

	if ( pwi->show_messages )
		{
		int inx = pwi->inx_message;
		for ( int i = 0; i < N_MESSAGES; i++ )
			{
			if ( --inx < 0 )
				inx = N_MESSAGES-1;
			if ( pwi->messages[inx] == 0 )
				break;
			TextOut(hdc, 0, cyClient-(i+1)*yChar,
				pwi->messages[inx], strlen(pwi->messages[inx]));
			}
		}

	ReleaseMutex(pwi->hmutex);

	EndPaint(hwnd, &ps);
	}
	return 0;
/*...e*/
/*...sWM_QUERYNEWPALETTE \45\ got focus\44\ set our palette:16:*/
case WM_QUERYNEWPALETTE:
	// We've got input focus, set our choice of palette
	{
	ProgressWinInternal *pwi = (ProgressWinInternal *) GetWindowLongPtr(hwnd, 0);
	if ( pwi->hpalette != (HPALETTE) NULL )
		RealizeOurPalette(hwnd, pwi->hpalette, FALSE);
	}
	return 0;
/*...e*/
/*...sWM_PALETTECHANGED  \45\ other changed palette:16:*/
case WM_PALETTECHANGED:
	if ( (HWND) wParam != hwnd )
		// Someone else got palette
		// Aquire what entries we can
		{
		ProgressWinInternal *pwi = (ProgressWinInternal *) GetWindowLongPtr(hwnd, 0);
		if ( pwi->hpalette != (HPALETTE) NULL )
			RealizeOurPalette(hwnd, pwi->hpalette, FALSE);
		}
	return 0;
/*...e*/
/*...sWM_CLOSE           \45\ close:16:*/
case WM_CLOSE:
	{
	ProgressWinInternal *pwi = (ProgressWinInternal *) GetWindowLongPtr(hwnd, 0);
	WaitForSingleObject(pwi->hmutex, INFINITE);
	pwi->window_closed = TRUE;
	DestroyWindow(hwnd);
	ReleaseMutex(pwi->hmutex);
	}
	return 0;
/*...e*/
/*...sWM_DESTROY         \45\ destroy:16:*/
case WM_DESTROY:
	{
	ProgressWinInternal *pwi = (ProgressWinInternal *) GetWindowLongPtr(hwnd, 0);
	WaitForSingleObject(pwi->hmutex, INFINITE);
	pwi->hwnd = (HWND) NULL;
	if ( pwi->hpalette != (HPALETTE) NULL )
		DeleteObject(pwi->hpalette);
	DeleteObject(pwi->hbitmap);
	ReleaseMutex(pwi->hmutex);
	SetEvent(pwi->heventUpdated);
	SetEvent(pwi->heventDeleted);
	PostQuitMessage(0);
	}
	return 0;
/*...e*/
/*...sWM_CHAR            \45\ characters pressed:16:*/
case WM_CHAR:
	switch ( wParam )
		{
/*...sT\44\t \45\ change text display mode:32:*/
case 'T': case 't':
	{
	ProgressWinInternal *pwi = (ProgressWinInternal *) GetWindowLongPtr(hwnd, 0);
	WaitForSingleObject(pwi->hmutex, INFINITE);
	pwi->iTextMode = (pwi->iTextMode+1)%N_TEXTMODES;
	ReleaseMutex(pwi->hmutex);
	}
	return 0L;
/*...e*/
/*...sM\44\m \45\ show\47\hide messages:32:*/
case 'M': case 'm':
	{
	ProgressWinInternal *pwi = (ProgressWinInternal *) GetWindowLongPtr(hwnd, 0);
	WaitForSingleObject(pwi->hmutex, INFINITE);
	pwi->show_messages = !pwi->show_messages;
	ReleaseMutex(pwi->hmutex);
	}
	return 0L;
/*...e*/
/*...sP\44\p \45\ show\47\hide progress:32:*/
case 'P': case 'p':
	{
	ProgressWinInternal *pwi = (ProgressWinInternal *) GetWindowLongPtr(hwnd, 0);
	WaitForSingleObject(pwi->hmutex, INFINITE);
	pwi->show_progress = !pwi->show_progress;
	ReleaseMutex(pwi->hmutex);
	}
	return 0L;
/*...e*/
		}
	break;
/*...e*/
/*...sWM_UPDATEWINDOW    \45\ update window:16:*/
case WM_UPDATEWINDOW:
	{
	ProgressWinInternal *pwi = (ProgressWinInternal *) GetWindowLongPtr(hwnd, 0);
	UpdateWindow(hwnd);
	SetEvent(pwi->heventUpdated);
	}
	return 0;
/*...e*/
/*...sWM_DELETEWINDOW    \45\ delete window:16:*/
case WM_DELETEWINDOW:
	DestroyWindow(hwnd);
	return 0;
/*...e*/
		}
	return DefWindowProc(hwnd, message, wParam, lParam);
	}
/*...e*/
/*...sUiThread:0:*/
// The main thread of the program isn't necessarily a windows thread.
// So we create a second thread to create and watch the progress window.

static void _cdecl UiThread(void *p)
	{
	ProgressWinInternal *pwi = (ProgressWinInternal *) p;
	int x, y;
	if ( pwi->wf & ProgressWin::centralized )
		{
		x = (cxScreen - pwi->gbm->w)/2 - cxFixedFrame;
		y = (cyScreen - pwi->gbm->h)/2 - cyFixedFrame - cyCaption;
		// Now ensure user can easily get to system menu
		if ( x < 0 || y < 0 ) x = y = 10;
		}
	else
		x = y = CW_USEDEFAULT;
	pwi->hwnd = CreateWindow(
		WC_PROGWIN,		// Window class name
		pwi->wtitle,		// Window title
		WS_BORDER|WS_SYSMENU|WS_MINIMIZEBOX|WS_OVERLAPPED,
		x,
		y,
		cxFixedFrame + pwi->gbm->w + cxFixedFrame,
		cyFixedFrame + cyCaption + pwi->gbm->h + cyFixedFrame,
		NULL,			// No parent for this window
		NULL,			// Use the class menu (not our own)
		hinst,			// Who created the window
		pwi);			// Parameters to pass on
	ShowWindow(pwi->hwnd,
		( pwi->wf & ProgressWin::minimized ) != 0
			? SW_SHOWMINIMIZED
			: SW_SHOWNORMAL);
	SetWindowPos(pwi->hwnd, HWND_TOP, 0, 0, 0, 0,
		SWP_NOMOVE|SWP_NOSIZE|SWP_SHOWWINDOW);
	SetEvent(pwi->heventCreated);
	MSG msg;
	while ( GetMessage(&msg, NULL, 0, 0) )
		{
		TranslateMessage(&msg); // So that WM_CHAR works!
		DispatchMessage(&msg);
		}
	}
/*...e*/

/*...sProgressWinInternal:0:*/
ProgressWinInternal::ProgressWinInternal(
	Bitmap & b, const char *title, ProgressWin::WinFlags wf
	)
	:
	wf(wf)
	{
	// Mutex used to ensure either main tracing thread or windows
	// thread can only access pwi at any one time.
	hmutex = CreateMutex(NULL, FALSE, NULL);

	// Event which gets signalled when initialisation completes
	heventCreated = CreateEvent(NULL, TRUE, FALSE, NULL);

	// Event which gets signalled when update window occurs
	heventUpdated = CreateEvent(NULL, TRUE, FALSE, NULL);

	// Event which gets signalled when window destroyed
	heventDeleted = CreateEvent(NULL, TRUE, FALSE, NULL);

	for ( int i = 0; i < N_MESSAGES; i++ )
		messages[i] = 0;
	inx_message = 0;
	percent = 0;

	GBMRGB *gbmrgb_dummy; // 24bpp bitmap has no palette
	b.get_gbm_internals(gbm, gbmrgb_dummy, data24);

	switch ( iBppErrDiff )
		{
		case 8:
			data = new Byte[((gbm->w+3)&~3)*gbm->h];
			break;
		case 4:
			data = new Byte[((gbm->w*4+31)/32*4)*gbm->h];
			break;
		case 24:
			data = data24;
			break;
		}

	t_started = 0;
	t_fin     = 0;

	iTextMode = 0;
	show_messages = TRUE;
	show_progress = TRUE;

	if ( title != 0 )
		strcpy(wtitle, title);
	else
		strcpy(wtitle, "Raytrace progress");
	window_closed = FALSE;
	_beginthread(UiThread, 0x10000, this);
	WaitForSingleObject(heventCreated, INFINITE);
	}
/*...e*/
/*...s\126\ProgressWinInternal:0:*/
ProgressWinInternal::~ProgressWinInternal()
	{
	WaitForSingleObject(hmutex, INFINITE);
	if ( hwnd != (HWND) NULL )
		PostMessage(hwnd, WM_DELETEWINDOW, 0, 0);
	ReleaseMutex(hmutex);
	WaitForSingleObject(heventDeleted, INFINITE);
	for ( int i = 0; i < N_MESSAGES; i++ )
		if ( messages[i] )
			delete[] messages[i];
	if ( data != data24 )
		delete[] data;
	}
/*...e*/
/*...sredraw:0:*/
void ProgressWinInternal::redraw()
	{
	WaitForSingleObject(hmutex, INFINITE);
	if ( hwnd != (HWND) NULL )
		{
		ResetEvent(heventUpdated);
		InvalidateRect(hwnd, NULL, TRUE);
		PostMessage(hwnd, WM_UPDATEWINDOW, 0, 0); // delayed redraw
		ReleaseMutex(hmutex);
		WaitForSingleObject(heventUpdated, INFINITE);
		}
	else
		ReleaseMutex(hmutex);
	}
/*...e*/
/*...sbegin:0:*/
void ProgressWinInternal::begin()
	{
	WaitForSingleObject(hmutex, INFINITE);
	t_started = time(NULL);
	ReleaseMutex(hmutex);
	}
/*...e*/
/*...send:0:*/
void ProgressWinInternal::end() {}
/*...e*/
/*...sdone:0:*/
void ProgressWinInternal::done(int sofar, int total)
	{
	WaitForSingleObject(hmutex, INFINITE);
	t_fin = t_started + (((time(NULL) - t_started) * total) / sofar);
	int old_percent = percent;
	percent = (sofar*100)/total;
	if ( percent > old_percent )
		// Only redraw if a reasonable amount of progress made
		{
		switch ( iBppErrDiff )
			{
			case 8:
				gbm_errdiff_7R8G4B(gbm, data24, data);
				break;
			case 4:
				gbm_errdiff_VGA(gbm, data24, data);
				break;
			default:
				break;
			}
		ReleaseMutex(hmutex);
		redraw();
		}
	else
		ReleaseMutex(hmutex);
	}
/*...e*/
/*...smessage:0:*/
void ProgressWinInternal::message(const char *message)
	{
	int len = strlen(message);
	WaitForSingleObject(hmutex, INFINITE);
	if ( messages[inx_message] )
		delete[] messages[inx_message];
	messages[inx_message] = new char[len+1];
	memcpy(messages[inx_message], message, len);
	messages[inx_message][len] = '\0';
	if ( ++inx_message == N_MESSAGES )
		inx_message = 0;
	ReleaseMutex(hmutex);
	redraw();
	}
/*...e*/
/*...e*/
/*...sclass ProgressWin:0:*/
// Conceal internal details (needed by window class) from external users.

ProgressWin::ProgressWin(Bitmap & b, const char *title, WinFlags wf)
	{ pwi = new ProgressWinInternal(b, title, wf); }
ProgressWin::~ProgressWin()
	{ delete pwi; }
void ProgressWin::begin()
	{ pwi->begin(); }
void ProgressWin::end()
	{ pwi->end(); }
void ProgressWin::done(int sofar, int total)
	{
	pwi->done(sofar, total);
	if ( pwi->window_closed )
		abort();
	}
void ProgressWin::message(const char *message)
	{ pwi->message(message); }
/*...e*/
/*...sclass ProgwinWindows:0:*/
// A mechanism to ensure global Windows initialisation is done prior to
// any progress indicators being created.

class ProgwinWindows
	{
public:
	ProgwinWindows();
	};

ProgwinWindows::ProgwinWindows()
	{
	hinst = (HINSTANCE) GetModuleHandle((LPCSTR) NULL);

	// Register the window class
	WNDCLASS wc;
	wc.lpszClassName = WC_PROGWIN;
	wc.hInstance     = hinst;
	wc.lpfnWndProc   = ProgressWinWindowProc;
	wc.hCursor       = LoadCursor(NULL, IDC_ARROW);
	wc.hIcon         = (HICON) NULL;
	wc.lpszMenuName  = NULL;
	wc.hbrBackground = (HBRUSH) GetStockObject(NULL_BRUSH);
	wc.style         = CS_HREDRAW|CS_VREDRAW;
	wc.cbClsExtra    = 0;
	wc.cbWndExtra    = sizeof(void *);
	RegisterClass(&wc);

	// Learn some system metrics
	cxScreen     = GetSystemMetrics(SM_CXSCREEN);
	cyScreen     = GetSystemMetrics(SM_CYSCREEN);
	cyCaption    = GetSystemMetrics(SM_CYCAPTION);
	cxFixedFrame = GetSystemMetrics(SM_CXFIXEDFRAME);
	cyFixedFrame = GetSystemMetrics(SM_CYFIXEDFRAME);

	// Decide what we are going to error-diffuse down to (or not)
	HDC hdcScreen = GetWindowDC((HWND) NULL);
	LONG lRasterCaps = GetDeviceCaps( hdcScreen, RASTERCAPS );
	LONG lPlanes     = GetDeviceCaps( hdcScreen, PLANES     );
	LONG lBitCount   = GetDeviceCaps( hdcScreen, BITSPIXEL  );
	ReleaseDC((HWND) NULL, hdcScreen);
	if ( (lRasterCaps & RC_PALETTE) != 0 && lPlanes == 1 && lBitCount == 8 )
		{
		gbm_errdiff_pal_7R8G4B(gbmrgbErrDiff);
		iBppErrDiff = 8;
		}
	else if ( lPlanes == 1 && lBitCount == 4 )
		{
		gbm_errdiff_pal_VGA(gbmrgbErrDiff);
		iBppErrDiff = 4;
		}
	else
		iBppErrDiff = 24;
	}

static ProgwinWindows windows;
/*...e*/
