//
// raytrace.C - Ray Trace a scene
//

/*...sincludes:0:*/
#include <string.h>

#ifdef NO_CINCLUDES
  #include <iostream.h>
  #if defined(STRSTREA)
    #include <strstrea.h>
  #else
    #include <strstream.h>
  #endif
#else
  #include <iostream>
  #if defined(LINUX)
    #include <backward/strstream>
  #else
    #include <strstream>
  #endif
#endif

#ifndef NO_STDNS
  using namespace std;
#endif

#include "raytrace.h"

/*...vraytrace\46\h:0:*/
/*...e*/

double least_vis_diff = 1.0 / 255.0;

/*...sscale_by_distance:0:*/
// The further light travels (from light source to surface, or from surface to
// eye etc.) the fainter it gets. Therefore as t increases, the amount the
// intensity is reduced by should increase.

inline Rgb scale_by_distance(const Rgb & i, double t, const Lights & lights)
	{
	double scalar = lights.af1 * pow(lights.af2, t * lights.af3);
	if ( scalar > 1.0 )
		return i;
	return i * scalar;
	}
/*...e*/
/*...strace:0:*/
#define	EPSILON (1.0e-5) // A tiny number

/*...sshadow_calc:0:*/
// In a CSG system without refraction, no rays can pass through any solid.
// Hence this code would return 1.0 or 0.0 depending on whether il is empty.
// We're only really interested in the range of t from 0 to dist_l being empty.
// This is because the light is dist_l away, things further don't cause shadows.
//
// With refraction, things can be lit through glass-like solids.
// Let me admit, from the start, that refraction in a CSG system is a kludge!
// Glass-like solids transmit kt of the light coming from their other side.
// So we look along the intersections for transmissive surfaces and work out
// the combined transmissiveness.
//
// We assume that pathalogical shapes (transmissive on one side,
// non-transmissive on the the other) do not exist. We also assume the kt
// coefficients match for a given shape on its entry and exit. We only use kt
// once (on entry) per shape.
//
// Actually, to correctly handle illumination through refractive shapes is a
// nightmare, and requires that we calculate a full multi-part path back to the
// light. This is horrendous, and we will assume that refraction does not alter
// the path (much) back to the light.
//
// In doing so, we will prevent glass causing shadows, which is the main goal.

static double shadow_calc(IsectList *il, double dist_l)
	{
	int j = 0;
	double kt_accum = 1.0;

	while ( j < il->n_isects && il->isects[j].t < dist_l )
		{
		double kt = il->isects[j].shape->surf.kt;

		if ( kt == 0.0 )
			return 0.0;
		kt_accum *= kt;
		j += 2;
		}

	return kt_accum;
	}
/*...e*/

/*...sreflect:0:*/
inline Xyz reflect(const Xyz & unit_v, const Xyz & unit_n, double cos_vn)
	{
	return unit_n * (cos_vn+cos_vn) - unit_v;
	}
/*...e*/
/*...srefract:0:*/
/*
          ^ N             Given an incident unit vector I, approaching a
          | -             surface with unit normal vector N, compute the
  \       |               transmitted ray T. This ray will not necessarily
   \      |               be a unit vector.
    \     |               
     \  0 |               If the term under the square root is negative
      \  i|               then this indicates total internal reflection.
       \  |               
     I  \||               n_it is the relative refractive index of the
     -  -\|               incident medium relative the the transmissive
----------+----------> S  medium. Thus for air (1.0) entering crown glass
          |\           -  (1.5) this number would be 1.0/1.5 = 0.66 approx.
          | \             
          |  \            We use the equation given in Byte Magazine Dec 90.
          | 0 \           
          |  t \          
          |     \         
          |   T  \|       
          |   -  -\       
          |               
*/

static Boolean refract(
	const Xyz & unit_i,
	const Xyz & unit_n,
	double n_it,
	Xyz & t
	)
	{
	double cos_ni = -scalar_product(unit_n, unit_i);
	double under_root = 1.0 + n_it*n_it * (cos_ni*cos_ni - 1.0);

	if ( under_root < 0.0 )
		return FALSE; // Total internal reflection

	double n_comp = n_it * cos_ni - sqrt(under_root);

	t = ( unit_i * n_it + unit_n * n_comp ).unit();

	return TRUE;
	}
/*...e*/

static Rgb trace(
	PrimShape *root_shape,		// To work with at this level
	PrimShape *nest_shape,		// To work with at recursed levels
	const Lights & lights,
	const Xyz & start, const Xyz & direction,
	int depth,
	IsectList *ils[]
	)
	{
	double mag_direction = direction.magnitude();
	if ( mag_direction == 0.0 )
		return lights.i_untraced;
	Xyz unit_direction(direction / mag_direction);

	root_shape->intersect(start, unit_direction, ils);
	ils[0]->t_after(EPSILON);
	if ( ils[0]->is_empty() )
		return lights.i_background;

	// Hit something

	double t                   = ils[0]->isects[0].t;
	const PrimShapeSurf *shape = ils[0]->isects[0].shape;
	Xyz isect_posn(start + unit_direction*t);
	Xyz unit_n(shape->normal(isect_posn));
	if ( ils[0]->isects[0].negate_normal )
		unit_n = -unit_n;

	// Calculate colours at intersection position

	Rgb od(shape->surf.od.evaluate(isect_posn));
	Rgb os(shape->surf.os.evaluate(isect_posn));

	// Ambient light

	Rgb i(lights.i_ambient * shape->surf.ka * od);

	Xyz unit_v(-unit_direction);

	// For each light source

	for ( int j = 0; j < lights.n_lights; j++ )
/*...shandle contribution of this light source:16:*/
// l is the vector from the intersection point to the light.
// dist_l is the distance to l.
// unit_l is a unit vector pointing to the light.
// We can reuse the intersection list used to hit the object for the shadow calc.

{
Xyz l(lights.lights[j].posn - isect_posn);
double dist_l = l.magnitude();
Xyz unit_l(l / dist_l);

// Can we see the light from the point of intersection?

nest_shape->intersect(isect_posn, unit_l, ils);
ils[0]->t_after(EPSILON);

double kt_accum;
if ( (kt_accum = shadow_calc(ils[0], dist_l)) > 0.0 )
	{
	Rgb i_light(scale_by_distance(lights.lights[j].i, dist_l, lights) * kt_accum);

	// Diffuse lighting, using Lambert's law

	double cos_ln;
	if ( (cos_ln = scalar_product(unit_l, unit_n)) > 0.0 )
		i += ( i_light * od * ( shape->surf.kd * cos_ln ) );

	// Specular lighting by light source, using Phong model

	Xyz unit_r(reflect(unit_l, unit_n, cos_ln));

	double cos_rv;
	if ( (cos_rv = scalar_product(unit_r, unit_v)) > 0.0 )
		i += ( i_light * os * ( shape->surf.ks * pow(cos_rv, shape->surf.phong) ) );
	}
}
/*...e*/

	if ( depth > 0 )
		{
		if ( shape->surf.ks > 0.0 )
/*...sreflection:24:*/
{
double cos_nv;
if ( (cos_nv = scalar_product(unit_n, unit_v)) > 0.0 )
	{
	Xyz unit_r(reflect(unit_v, unit_n, cos_nv));
	Rgb i_r(trace(nest_shape, nest_shape, lights, isect_posn, unit_r, depth - 1, ils));
	i += ( os * i_r * shape->surf.ks );
	}
}
/*...e*/
		if ( shape->surf.kt > 0.0 )
/*...srefraction:24:*/
// Refractive index of outside medium is assumed to be 1.0.
// Bend the ray into the medium, work out where it comes out, and bend it again.
// Then trace the emerging ray and accumulate its effect.
// If total internal reflection occurs, entering the shape, treat as reflection.
// If it occurs leaving, bounce the ray inside the shape and try to exit again.

#define	MAX_BOUNCE 10

{
Xyz unit_r;				// Unit refracted ray at entry
if ( refract(unit_direction, unit_n, 1.0 / shape->surf.rinx, unit_r) )
	// Refraction has occurred
	{
	double kt_comp = shape->surf.kt;// Composite scale down factor
	double t_comp = 0.0;		// Composite/total distance inside
	int n_bounce = 0;		// # of internal bounces
	Xyz isect_posn_out(isect_posn);
	Xyz unit_r_out(unit_r);
	for ( ;; )
		{
		nest_shape->intersect(isect_posn_out, unit_r_out, ils);
		ils[0]->t_after(EPSILON);

		double t_out                   = ils[0]->isects[1].t;
		const PrimShapeSurf *shape_out = ils[0]->isects[1].shape;
		isect_posn_out                += unit_r_out * t_out;
		Xyz unit_n_out(shape_out->normal(isect_posn_out));

		if ( !ils[0]->isects[1].negate_normal )
			unit_n_out = -unit_n_out;

		t_comp += t_out;

		if ( refract(unit_r_out, unit_n_out, shape_out->surf.rinx, unit_r_out) )
			break; // Refracted out of solid

		// Total internal reflection trying to leave solid
		// This implies the solid has an index > 1.0

		if ( ++n_bounce == MAX_BOUNCE )
			break; // Reached our bounce limit, give up!

		double cos_rn = scalar_product(unit_r_out, unit_n_out);
		unit_r_out = reflect(-unit_r_out, unit_n_out, -cos_rn);
		kt_comp *= shape_out->surf.kt;
			// Accumulate this as we effectively re-enter solid
		}

	if ( n_bounce < MAX_BOUNCE )
		{
		Rgb i_r(trace(nest_shape, nest_shape, lights, isect_posn_out, unit_r_out, depth - 1, ils));
		i += ( scale_by_distance(i_r, t_comp, lights) * kt_comp );
		}
	}
else
	// Total internal reflection trying to enter solid
	// This implies the solid has an index < 1.0
	// This is not actually very likely (glass is 1.5, water 1.3 etc)
	{
	double cos_nv;
	if ( (cos_nv = scalar_product(unit_n, unit_v)) > 0.0 )
		{
		Xyz u_r(reflect(unit_v, unit_n, cos_nv));
		Rgb i_r(trace(nest_shape, nest_shape, lights, isect_posn, u_r, depth - 1, ils));
		i += ( i_r * shape->surf.kt );
		}
	}
}
/*...e*/
		}

	return scale_by_distance(i, t, lights);
	}
/*...e*/

/*...srayfuncs:0:*/
// Given various parameters, a rayfunc will give the vector to initially fire.
// The user can choose to use any of the below, or even write their own.
// Profiling reveals that quite a lot of time is spent in rayfuncs.
// The sample ray functions evaluate the same things on every ray.
// So the raytrace function detects the use of the predefined ones and it
// substitutes modified simpler 'cheat' versions that rely on being passed
// arguments that mean different things.

/*...srayfunc_normal:0:*/
// Calculates a ray given usual perspective rules.
// Need to know which pixel of how many we are tracing.
// Need to know right and up vectors.

Xyz rayfunc_normal(
	double hfactor, double vfactor,
	double hangle, double vangle,
	const Xyz & unit_forward, const Xyz & unit_right, const Xyz & unit_up
	)
	{
	return unit_forward +
	       unit_right * ( tan(hangle) * hfactor ) +
	       unit_up    * ( tan(vangle) * vfactor ) ;
	}

// If the rayfunc_normal_cheat is subsituted for rayfunc_normal, then
// unit_right and unit_up are also scaled by tan(hangle) and tan(vangle)
// prior to this function being called, thus eliminating 2 *s, and 2 tan()'s.

static Xyz rayfunc_normal_cheat(
	double hfactor, double vfactor,
	double hangle, double vangle,
	const Xyz & unit_forward, const Xyz & unit_right, const Xyz & unit_up
	)
	{
	hangle=hangle; vangle=vangle; // Suppress warnings
#ifndef NO_HAND_OPTIMISE
	return Xyz(
		unit_forward.x + unit_right.x * hfactor + unit_up.x * vfactor,
		unit_forward.y + unit_right.y * hfactor + unit_up.y * vfactor,
		unit_forward.z + unit_right.z * hfactor + unit_up.z * vfactor);
#else
	return unit_forward +
	       unit_right * hfactor +
	       unit_up    * vfactor ;
#endif
	}
/*...e*/
/*...srayfunc_wideangle:0:*/
// Wideangle projection.
// @@@ Untested

Xyz rayfunc_wideangle(
	double hfactor, double vfactor,
	double hangle, double vangle,
	const Xyz & unit_forward, const Xyz & unit_right, const Xyz & unit_up
	)
	{
	double ha = hangle * hfactor;
	return unit_right   * sin(ha) +
	       unit_forward * cos(ha) +
	       unit_up      * ( tan(vangle) * vfactor );
	}

// unit_up is pre-scaled by tan(vangle).

static Xyz rayfunc_wideangle_cheat(
	double hfactor, double vfactor,
	double hangle, double vangle,
	const Xyz & unit_forward, const Xyz & unit_right, const Xyz & unit_up
	)
	{
	vangle=vangle; // Suppress warnings
	double ha = hangle * hfactor;
	return unit_right   * sin(ha) +
	       unit_forward * cos(ha) +
	       unit_up      * vfactor;
	}
/*...e*/
/*...srayfunc_tallangle:0:*/
// Calculates a ray given using the twisted perspective Escher used in his
// "Above and Below" and "House of Stairs" works.

Xyz rayfunc_tallangle(
	double hfactor, double vfactor,
	double hangle, double vangle,
	const Xyz & unit_forward, const Xyz & unit_right, const Xyz & unit_up
	)
	{
	double va = vangle * vfactor;
	return unit_right   * ( tan(hangle) * hfactor ) +
	       unit_forward * cos(va) +
	       unit_up      * sin(va);
	}

// unit_right is pre-scaled by tan(hangle).

static Xyz rayfunc_tallangle_cheat(
	double hfactor, double vfactor,
	double hangle, double vangle,
	const Xyz & unit_forward, const Xyz & unit_right, const Xyz & unit_up
	)
	{
	hangle=hangle; // Suppress warnings
	double va = vangle * vfactor;
	return unit_right   * hfactor +
	       unit_forward * cos(va) +
	       unit_up      * sin(va);
	}
/*...e*/
/*...srayfunc_fisheye:0:*/
// 'Fisheye' projection
// @@@ Untested

Xyz rayfunc_fisheye(
	double hfactor, double vfactor,
	double hangle, double vangle,
	const Xyz & unit_forward, const Xyz & unit_right, const Xyz & unit_up
	)
	{
	double ha = hangle * hfactor;
	double va = vangle * vfactor;

	// Compute vector on the equator, at the right longitude
	Xyz equator = unit_right   * sin(ha) +
		      unit_forward * cos(ha) ;

	// Now rotate this to give a point at a given latitude
	return unit_up * sin(va) +
	       equator * cos(va) ;
	}
/*...e*/
/*...e*/
/*...sraytrace:0:*/
// We try to cut down the time spent inside rayfuncs, by detecting
// the common case(s) and substituting simpler rayfuncs.
//
// Also, if we know that all first rays are fired forwards, we can
// produce a simpler initial model for the first ray.

/*...sbasic_render:0:*/
// Traditional basic render operation.
// Send one ray for each pixel (excluding extras for reflection etc.).
// ie: Sample once for each pixel, do NOT supersample.
// Also, do not perform any 'jittering' of rays.

static Boolean basic_render(
	PrimShape *pshape1, PrimShape *pshape,
	const Lights & lights,
	IsectList *ils[],
	Xyz eye, Xyz unit_forward, Xyz unit_right, Xyz unit_up,
	double hangle, double vangle,
	RAYFUNC rayfunc,
	Bitmap & bitmap,
	int depth,
	Progress & progress
	)
	{
	int hpixels = bitmap.width(),  hpixels2 = (hpixels >> 1);
	int vpixels = bitmap.height(), vpixels2 = (vpixels >> 1);
	progress.begin();
	for ( int v = 0; v < vpixels; v++ )
		{
		double vfactor = (double) (v - vpixels2) / vpixels2;
		for ( int h = 0; h < hpixels; h++ )
			{
			double hfactor = (double) (h - hpixels2) / hpixels2;
			Xyz ray = (*rayfunc)(hfactor, vfactor, hangle, vangle, unit_forward, unit_right, unit_up);
			Rgb rgb = trace(pshape1, pshape, lights, eye, ray, depth, ils);
			bitmap.set_pixel(h, v, rgb.clamped());
			}
		progress.done(v+1, vpixels);
		if ( progress.aborted() )
			{
			progress.end();
			return FALSE;
			}
		}
	progress.end();

	return TRUE;
	}
/*...e*/
/*...swhitted_render:0:*/
// Render using Whitted adaptive supersampling.
// Send rays at the 4 corners of a pixel.
// Ask function whitted for a combined pixel value.

class Sample
	{
public:
	Xyz ray;
	Rgb rgb;
	Boolean traced;
	Sample() : traced(FALSE) {}
	~Sample() {}
	};

#define	MAX_SS 4 // Supersample by upto 4 each way

/*...swhitted:0:*/
// Return a combined weighting of the 4 corner samples.
// If they differ 'notably', break the area into 4 sub-cells.
// Generate the 5 extra samples needed for 4 sub-cells.
// Then call ourself to get colours for each cell, and return average.

/*...sis_close:0:*/
// 2 colours are not close when their red green and blue components differ
// more than the least visible difference.

inline Boolean is_close(const Rgb & a, const Rgb & b)
	{
	return fabs(a.r - b.r) < least_vis_diff &&
	       fabs(a.g - b.g) < least_vis_diff &&
	       fabs(a.b - b.b) < least_vis_diff ;
	}
/*...e*/

static Rgb whitted(
	Sample *samples[MAX_SS+1],
	int h,				// Scaled by ss
	int ss,				// Sample spacing
	PrimShape *pshape1, PrimShape *pshape,
	const Lights & lights,
	Xyz eye,
	int depth,
	IsectList *ils[],
	long & extra
	)
	{
	Sample *sa = &(samples[ 0][h     ]);
	Sample *sb = &(samples[ 0][h + ss]);
	Sample *sc = &(samples[ss][h     ]);
	Sample *sd = &(samples[ss][h + ss]);
	Rgb m_rgb = (sa->rgb+sb->rgb+sc->rgb+sd->rgb)*0.25;

	if ( ss == 1 )
		return m_rgb;

	// Are all 4 corners close to the middle

	if ( is_close(m_rgb, sa->rgb) &&
	     is_close(m_rgb, sb->rgb) &&
	     is_close(m_rgb, sc->rgb) &&
	     is_close(m_rgb, sd->rgb) )
		return m_rgb;

	Xyz a = sa->ray, b = sb->ray, c = sc->ray, d = sd->ray;
	int ss2 = (ss >> 1);
	Sample *sab   = &(samples[0  ][h + ss2]);
	Sample *scd   = &(samples[ss ][h + ss2]);
	Sample *sac   = &(samples[ss2][h      ]);
	Sample *sbd   = &(samples[ss2][h + ss ]);
	Sample *sabcd = &(samples[ss2][h + ss2]);

	// Trace any that are not already traced

	if ( ! sab->traced )
		{
		sab->ray = (a+b)*0.5;
		sab->rgb = trace(pshape1, pshape, lights, eye, sab->ray, depth, ils);
		sab->traced = TRUE;
		extra++;
		}

	if ( ! scd->traced )
		{
		scd->ray = (c+d)*0.5;
		scd->rgb = trace(pshape1, pshape, lights, eye, scd->ray, depth, ils);
		scd->traced = TRUE;
		extra++;
		}

	if ( ! sac->traced )
		{
		sac->ray = (a+c)*0.5;
		sac->rgb = trace(pshape1, pshape, lights, eye, sac->ray, depth, ils);
		sac->traced = TRUE;
		extra++;
		}

	if ( ! sbd->traced )
		{
		sbd->ray = (b+d)*0.5;
		sbd->rgb = trace(pshape1, pshape, lights, eye, sbd->ray, depth, ils);
		sbd->traced = TRUE;
		extra++;
		}

	if ( ! sabcd->traced )
		{
		sabcd->ray = (a+b+c+d)*0.25;
		sabcd->rgb = trace(pshape1, pshape, lights, eye, sabcd->ray, depth, ils);
		sabcd->traced = TRUE;
		extra++;
		}

	// Now obtain average of 4 nested whitted values

	return ( whitted(samples    , h    , ss2, pshape1, pshape, lights, eye, depth, ils, extra) +
		 whitted(samples+ss2, h    , ss2, pshape1, pshape, lights, eye, depth, ils, extra) +
		 whitted(samples    , h+ss2, ss2, pshape1, pshape, lights, eye, depth, ils, extra) +
		 whitted(samples+ss2, h+ss2, ss2, pshape1, pshape, lights, eye, depth, ils, extra) ) * 0.25;
	}
/*...e*/

static Boolean whitted_render(
	PrimShape *pshape1, PrimShape *pshape,
	const Lights & lights,
	IsectList *ils[],
	Xyz eye, Xyz unit_forward, Xyz unit_right, Xyz unit_up,
	double hangle, double vangle,
	RAYFUNC rayfunc,
	Bitmap & bitmap,
	int depth,
	Progress & progress
	)
	{
	int hpixels = bitmap.width(),  hpixels2 = (hpixels >> 1);
	int vpixels = bitmap.height(), vpixels2 = (vpixels >> 1);
	int h, v, ssh, ssv;
	int hsamples = (MAX_SS * hpixels) + 1;
	Sample *samples[MAX_SS+1], *t;
	long normal = 0, extra = 0;

	progress.begin();

	for ( ssv = 0; ssv <= MAX_SS; ssv++ )
		samples[ssv] = new Sample [hsamples];

	// Work out the top row
	for ( h = 0; h <= hpixels; h++ )
		{
		double hfactor = (double) (h - hpixels2) / hpixels2;
		Xyz ray = (*rayfunc)(hfactor, -1.0, hangle, vangle, unit_forward, unit_right, unit_up);
		Sample *s = samples[0];
		s[h * MAX_SS].ray = ray;
		s[h * MAX_SS].rgb = trace(pshape1, pshape, lights, eye, ray, depth, ils);
		s[h * MAX_SS].traced = TRUE;
		normal++;
		}

	progress.done(1, vpixels+1);
	if ( progress.aborted() )
		{
		progress.end();
		return FALSE;
		}

	for ( v = 0; v < vpixels; v++ )
		{
		double vfactor = (double) ((v+1) - vpixels2) / vpixels2;
		// Work out bottom row (for this scan line of pixels)
		for ( h = 0; h <= hpixels; h++ )
			{
			double hfactor = (double) (h - hpixels2) / hpixels2;
			Xyz ray = (*rayfunc)(hfactor, vfactor, hangle, vangle, unit_forward, unit_right, unit_up);
			Sample *s = samples[MAX_SS];
			s[h * MAX_SS].ray = ray;
			s[h * MAX_SS].rgb = trace(pshape1, pshape, lights, eye, ray, depth, ils);
			s[h * MAX_SS].traced = TRUE;
			normal++;
			}
		for ( h = 0; h < hpixels; h++ )
			{
			Rgb rgb = whitted(samples, h * MAX_SS, MAX_SS, pshape1, pshape, lights, eye, depth, ils, extra);
			bitmap.set_pixel(h, v, rgb.clamped());
			}
		t = samples[0]; samples[0] = samples[MAX_SS]; samples[MAX_SS] = t;
		for ( ssv = 1; ssv <= MAX_SS; ssv++ )
			for ( ssh = 0; ssh < hsamples; ssh++ )
				samples[ssv][ssh].traced = FALSE;
		progress.done(v+1, vpixels+1);
		if ( progress.aborted() )
			{
			progress.end();
			return FALSE;
			}
		}

	for ( ssv = 0; ssv <= MAX_SS; ssv++ )
		delete[] samples[ssv];

	progress.end();

/*...sdisplay number of \39\normal\39\ and \39\extra\39\ rays:8:*/
{
ostrstream ss;
ss << normal << " normal rays, " << extra << " extra rays" << ends;
char *s = ss.str();
progress.message(s);
delete[] s;
}
/*...e*/

	return TRUE;
	}
/*...e*/
/*...ssuper_render:0:*/
// Always supersample by 4 in each direction

#define	SS 4

static Boolean super_render(
	PrimShape *pshape1, PrimShape *pshape,
	const Lights & lights,
	IsectList *ils[],
	Xyz eye, Xyz unit_forward, Xyz unit_right, Xyz unit_up,
	double hangle, double vangle,
	RAYFUNC rayfunc,
	Bitmap & bitmap,
	int depth,
	Progress & progress
	)
	{
	int hpixels = bitmap.width(),  hpixels2 = (hpixels >> 1) * SS;
	int vpixels = bitmap.height(), vpixels2 = (vpixels >> 1) * SS;
	progress.begin();
	for ( int v = 0; v < vpixels; v++ )
		{
		for ( int h = 0; h < hpixels; h++ )
			{
			Rgb rgb(0.0,0.0,0.0);
			for ( int v2 = 0; v2 < SS; v2++ )
				{
				double vfactor = (double) (v*SS+v2 - vpixels2) / vpixels2;
				for ( int h2 = 0; h2 < SS; h2++ )
					{
					double hfactor = (double) (h*SS+h2 - hpixels2) / hpixels2;
					Xyz ray = (*rayfunc)(hfactor, vfactor, hangle, vangle, unit_forward, unit_right, unit_up);
					rgb += trace(pshape1, pshape, lights, eye, ray, depth, ils);
					}
				}
			bitmap.set_pixel(h, v, (rgb/(SS*SS)).clamped());
			}
		progress.done(v+1, vpixels);
		if ( progress.aborted() )
			{
			progress.end();
			return FALSE;
			}
		}
	progress.end();

	return TRUE;
	}

#undef SS
/*...e*/
/*...sunpix_render:0:*/
// Unpixelatation renderer
// Starts by rendering one pixel in 16*16 and then homes in with finer detail

static Boolean unpix_render(
	PrimShape *pshape1, PrimShape *pshape,
	const Lights & lights,
	IsectList *ils[],
	Xyz eye, Xyz unit_forward, Xyz unit_right, Xyz unit_up,
	double hangle, double vangle,
	RAYFUNC rayfunc,
	Bitmap & bitmap,
	int depth,
	Progress & progress
	)
	{
	int hpixels = bitmap.width(),  hpixels2 = (hpixels >> 1);
	int vpixels = bitmap.height(), vpixels2 = (vpixels >> 1);
	int sofar = 0+1, total = hpixels * vpixels + 1, step, mask;
	progress.begin();
	for ( step = 16, mask = ~0; step > 0; step >>= 1, mask = step*2-1 )
		for ( int v = 0; v < vpixels; v += step )
			{
			int vamount = Min(vpixels-v, step);
			double vfactor = (double) (v - vpixels2) / vpixels2;
			for ( int h = 0; h < hpixels; h += step )
				if ( (h & mask) != 0 || (v & mask) != 0 )
					{
					int hamount = Min(hpixels-h, step);
/*...strace and store to block of pixels:40:*/
double hfactor = (double) (h - hpixels2) / hpixels2;
Xyz ray = (*rayfunc)(hfactor, vfactor, hangle, vangle, unit_forward, unit_right, unit_up);
Rgb rgb = trace(pshape1, pshape, lights, eye, ray, depth, ils);
bitmap.set_pixel(h, v, hamount, vamount, rgb.clamped());
/*...e*/
					sofar++;
					}
			progress.done(sofar+1, total);
			if ( progress.aborted() )
				{
				progress.end();
				return FALSE;
				}
			}
	progress.end();

	return TRUE;
	}
/*...e*/

Boolean raytrace(
	Shape shape,
	const Lights & lights,
	const View & view,
	Bitmap & bitmap,
	Progress & progress,
	int depth,
	int aa,
	RAYFUNC rayfunc
	)
	{
	Xyz unit_forward  = view.forward.unit();
	Xyz unit_up       = view.up.unit();
	Xyz unit_right    = unit_forward * unit_up;	// Vector product

	PrimShape *pshape = (PrimShape *) shape;	// Get access to guts!
	pshape->preprocess();				// Prepare for tracing
	pshape->extent();				// Labels nodes
	int isects_reqd   = pshape->isects_reqd();	// Largest IsectList
	int isect_depth   = pshape->isect_depth();	// # IsectList needed

/*...sdisplay requirements for \39\normal\39\ rays:8:*/
{
ostrstream ss;
ss << "Normal rays: "
   << isect_depth << " lists, "
   << isects_reqd << " isects" << ends;
char *s = ss.str();
progress.message(s);
delete[] s;
}
/*...e*/

	IsectList **ils = new IsectList* [isect_depth];
	for ( int i = 0; i < isect_depth; i++ )
		ils[i] = new IsectList(isects_reqd);

	if ( rayfunc == 0 )
		rayfunc = rayfunc_normal;

	Boolean cull_behind;
/*...ssubstitute for faster rayfuncs if we can:8:*/
if ( rayfunc == rayfunc_normal )
	{
	rayfunc = rayfunc_normal_cheat;
	unit_right *= tan(view.hangle);
	unit_up    *= tan(view.vangle);
	cull_behind = TRUE;
	}
else if ( rayfunc == rayfunc_wideangle )
	{
	rayfunc = rayfunc_wideangle_cheat;
	unit_up *= tan(view.vangle);
	cull_behind = ( view.hangle <= PI/2 );
	}
else if ( rayfunc == rayfunc_tallangle )
	{
	rayfunc = rayfunc_tallangle_cheat;
	unit_right *= tan(view.hangle);
	cull_behind = ( view.vangle <= PI/2 );
	}
else
	// Could be any old rayfunc, so can't make this optimisation
	cull_behind = FALSE;
/*...e*/

	PrimShape *pshape1 = 0;
	if ( cull_behind )
		// Cull part of model behind eyepoint, if by looking
		// at the rayfuncs earlier, we saw this made sense
		{
		Xyz n(-unit_forward);
		pshape1 = pshape->simplify(n.x, n.y, n.z,
			-(n.x*view.eye.x+n.y*view.eye.y+n.z*view.eye.z));
			// pshape1 can be 0 entire shape culled away
		}
	if ( pshape1 != 0 )
		{
		pshape1->preprocess();
		pshape1->extent();
/*...sdisplay requirements for \39\first\39\ rays:16:*/
{
ostrstream ss;
ss << "First rays: "
   << pshape1->isect_depth() << " lists, "
   << pshape1->isects_reqd() << " isects" << ends;
char *s = ss.str();
progress.message(s);
delete[] s;
}
/*...e*/
		}
	else
		pshape1 = pshape;

	Boolean ok;
	switch ( aa )
		{
		case AA_NONE:
			ok = basic_render(pshape1, pshape, lights, ils,
				view.eye, unit_forward, unit_right, unit_up,
				view.hangle, view.vangle,
				rayfunc, bitmap, depth, progress);
			break;
		case AA_WHITTED:
			ok = whitted_render(pshape1, pshape, lights, ils,
				view.eye, unit_forward, unit_right, unit_up,
				view.hangle, view.vangle,
				rayfunc, bitmap, depth, progress);
			break;
		case AA_SUPERSAMPLE:
			ok = super_render(pshape1, pshape, lights, ils,
				view.eye, unit_forward, unit_right, unit_up,
				view.hangle, view.vangle,
				rayfunc, bitmap, depth, progress);
			break;
		case AA_UNPIXELATE:
			ok = unpix_render(pshape1, pshape, lights, ils,
				view.eye, unit_forward, unit_right, unit_up,
				view.hangle, view.vangle,
				rayfunc, bitmap, depth, progress);
			break;
		default:
			ok = FALSE;
			break;
		}

	for ( int j = 0; j < isect_depth; j++ )
		delete ils[j];
	delete[] ils;

	if ( pshape1 != pshape )
		delete pshape1;

	return ok;
	}
/*...e*/
