//
// roots.C - Find roots of equations
//
// Note that conventional wisdom is that the analytical approach below can
// have sufficiently large errors so as to cause rendering artefacts.
// POV-Ray, for example, can use the Sturm method for root solving.
// In tests thus far, my results have been presentable enough.
// Note the use of fudge factors in various places in MR, including a
// "close to zero" value in solve_quartic_1.
//

/*...sincludes:0:*/
#include "roots.h"

/*...vroots\46\h:0:*/
/*...e*/

/*...ssolve_linear:0:*/
// Solve ax+b=0

int solve_linear(double a, double b, Complex roots[])
	{
	if ( a == 0.0 )
		return 0;
	roots[0] = -b / a;
	return 1;
	}
/*...e*/
/*...ssolve_quadratic:0:*/
//         2
// Solve ax +bx+c=0

int solve_quadratic(double a, double b, double c, Complex roots[])
	{
	if ( a == 0.0 )
		return solve_linear(b, c, roots);
	Complex t1(sqrt(Complex(b*b-4.0*a*c)));
	double t2 = 2.0 * a;
	roots[0] = ( -b + t1 ) / t2;
	roots[1] = ( -b - t1 ) / t2;
	return 2;
	}
/*...e*/
/*...ssolve_resolvent_cubic:0:*/
//         3
// Solve: x +ax+b=0

static const double one_third = 1.0 / 3.0;

inline Complex cube_root(const Complex & z)
	{
	if ( real(z) >= 0 )
		return pow(z, one_third);
	else
		return -pow(-z, one_third);
	}

static Complex root_neg_3;

class ComplexConstants
	{
public:
	ComplexConstants()
		{
		root_neg_3 = sqrt(Complex(-3.0));
		}
	};

static ComplexConstants cc;

inline void solve_resolvent_cubic(double a, double b, Complex roots[])
	{
	double  t1 = 0.5 * b;
	Complex t2 = sqrt(Complex(b*b/4.0+a*a*a/27.0));
	Complex t3 = cube_root( - t1 + t2 );
	Complex t4 = cube_root( - t1 - t2 );
	Complex t5 = t3+t4;
	Complex t6 = root_neg_3 * (t3-t4);
	roots[0] = t5;
	roots[1] = 0.5 * ( - t5 - t6 );
	roots[2] = 0.5 * ( - t5 + t6 );
	}

/*

This text was taken from a news posting on sci.math by
Rob Johnson of Apple Computer, Inc. I have simplified it for
just the resolvent cubic case, and handled the mapping myself.

     3
    y  + a y + b = 0

    y = d+e

        -(d+e) + (d-e) sqrt(-3)
    y = -----------------------
                   2

        -(d+e) - (d-e) sqrt(-3)
    y = -----------------------
                   2

        +-      +-   2    3  -+1/2 -+1/3
        |   b   |   b    a    |     |
    d = | - - + |   -- + --   |     |
        |   2   |   4    27   |     |
        +-      +-           -+    -+

        +-      +-   2    3  -+1/2 -+1/3
        |   b   |   b    a    |     |
    e = | - - - |   -- + --   |     |
        |   2   |   4    27   |     |
        +-      +-           -+    -+

*/
/*...e*/
/*...ssolve_cubic_1:0:*/
//        3   2
// Solve x +bx +cx+d=0

static int solve_cubic_1(double b, double c, double d, Complex roots[])
	{
	// Solve making the substition x -> y-(b/3.0)
	// the y squared term vanishes, giving the 'resolvent' cubic

	double b2 = b*b;
	double p = c-one_third*b2;
	double q = b * ( (2.0/27.0)*b2 - one_third*c ) + d;

	//        3
	// Solve y +py+q=0

	solve_resolvent_cubic(p, q, roots);

	// Back substitute, to give roots of original cubic

	Complex t1(one_third*b);
	roots[0] -= t1;
	roots[1] -= t1;
	roots[2] -= t1;
	return 3;
	}
/*...e*/
/*...ssolve_cubic:0:*/
//         3   2
// Solve ax +bx +cx+d=0

int solve_cubic(double a, double b, double c, double d, Complex roots[])
	{
	if ( a == 0.0 )
		return solve_quadratic(b, c, d, roots);
	else
		return solve_cubic_1(b/a, c/a, d/a, roots);
	}
/*...e*/
/*...ssolve_quartic_1:0:*/
//         4   3   2
// Solve: x +mx +nx + px+q=0

#define	VERY_SMALL 1e-10

static int solve_quartic_1(double m, double n, double p, double q, Complex roots[])
	{
	double m2 = m*m;
	double a = n - 0.375*m2;
	double b = (0.125*m2-0.5*n)*m + p;
	double c = ( 16.0*m2*n-3.0*m2*m2-64.0*p*m ) / 256.0 + q;

	solve_cubic_1(2.0*a, a*a-4.0*c, -b*b, roots);
	// roots[0] will be real and will be e*e

	m *= -0.25;
	if ( fabs(real(roots[0])) > VERY_SMALL )
		{
		Complex e(sqrt(roots[0]));
		Complex t1(e*e+2.0*a);
		Complex t2(2.0*b/e);
		Complex t3(sqrt(Complex(-(t1+t2))));
		roots[0] = m + 0.5 * (  e + t3 );
		roots[1] = m + 0.5 * (  e - t3 );
		Complex t4(sqrt(Complex(-(t1-t2))));
		roots[2] = m + 0.5 * ( -e + t4 );
		roots[3] = m + 0.5 * ( -e - t4 );
		}
	else
		{
		Complex t1(sqrt(Complex(a*a-4.0*c)));
		Complex t2(sqrt(Complex(-2.0*(a+t1))));
		roots[0] = m + 0.5 * t2;
		roots[1] = m - 0.5 * t2;
		Complex t3(sqrt(Complex(-2.0*(a-t1))));
		roots[2] = m + 0.5 * t3;
		roots[3] = m - 0.5 * t3;
		}
	return 4;
	}

/*

This text was taken from a news posting on sci.math by Rob Johnson of
Apple Computer, Inc..

                            2
          m    e +/- sqrt(-(e  + 2 a + 2 b/e))
    x = - - + --------------------------------
          4                  2

                             2
          m   -e +/- sqrt(-(e  + 2 a - 2 b/e))
    x = - - + --------------------------------
          4                  2

or if e = 0 (only if m^3 - 4mn + 8p = 0),

                                    2
          m     sqrt(-2(a +/- sqrt(a  - 4 c)))
    x = - - +/- ------------------------------
          4                   2

where

                 2                3
        8 n - 3 m                m  - 4 m n + 8 p
    a = ----------           b = ----------------
            8                           8

            2        4
        16 m  n - 3 m  - 64 p m + 256 q
    c = -------------------------------
                      256

and e is any solution of 

     6        4     2         2    2
    e  + 2 a e  + (a  - 4 c) e  - b  = 0

*/
/*...e*/
/*...ssolve_quartic:0:*/
//          4   3   2
// Solve: ax +bx +cx + dx+e=0

int solve_quartic(double a, double b, double c, double d, double e, Complex roots[])
	{
	if ( a == 0.0 )
		return solve_cubic(b, c, d, e, roots);
	else
		return solve_quartic_1(b/a, c/a, d/a, e/a, roots);
	}
/*...e*/

#ifdef TEST_HARNESS
/*...stest harness:0:*/
#define	TOLERANCE 1e-8
static int loop;
/*...ssolve_quadratic_check:0:*/
int solve_quadratic_check(double a, double b, double c, Complex roots[])
	{
	int n_roots = solve_quadratic(a, b, c, roots);
	int bad = 0;
	for ( int i = 0; i < n_roots; i++ )
		{
		Complex r(roots[i]);
		Complex check = (a*r+b)*r+c;
		if ( abs(check) > a*TOLERANCE )
			{
			if ( ++bad == 1 )
				cout << "quadratic   " << loop << " : " << a << "," << b << "," << c << endl;
			cout << "\troot[" << i << "]=" << r << " f=" << check << endl;
			}
		}
	return n_roots;
	}
/*...e*/
/*...ssolve_cubic_check:0:*/
int solve_cubic_check(double a, double b, double c, double d, Complex roots[])
	{
	int n_roots = solve_cubic(a, b, c, d, roots);
	int bad = 0;
	for ( int i = 0; i < n_roots; i++ )
		{
		Complex r(roots[i]);
		Complex check = ((a*r+b)*r+c)*r+d;
		if ( abs(check) > a*TOLERANCE )
			{
			if ( ++bad == 1 )
				cout << "cubic   " << loop << " : " << a << "," << b << "," << c << "," << d << endl;
			cout << "\troot[" << i << "]=" << r << " f=" << check << endl;
			}
		}
	return n_roots;
	}
/*...e*/
/*...ssolve_quartic_check:0:*/
int solve_quartic_check(double a, double b, double c, double d, double e, Complex roots[])
	{
	int n_roots = solve_quartic(a, b, c, d, e, roots);
	int bad = 0;
	for ( int i = 0; i < n_roots; i++ )
		{
		Complex r(roots[i]);
		Complex check = (((a*r+b)*r+c)*r+d)*r+e;
		if ( abs(check) > a*TOLERANCE )
			{
			if ( ++bad == 1 )
				cout << "quartic " << loop << " : " << a << "," << b << "," << c << "," << d << "," << e << endl;
			cout << "\troot[" << i << "]=" << r << " f=" << check << endl;
			}
		}
	return n_roots;
	}
/*...e*/
/*...smain:0:*/
#include <stdio.h>
#include <stdlib.h>

static double random_double() { return ( (double) rand() / 100.0 ) - 100.0; }

int main(int argc, char *argv[])
	{
	int n = 20;
	if ( argc > 1 )
		sscanf(argv[1], "%i", &n);
	cout.precision(20);
	for ( int i = 0; i < n; i++ )
		{
		double a = random_double();
		double b = random_double();
		double c = random_double();
		double d = random_double();
		double e = random_double();
		Complex roots[4];
		loop = i;
		int n_roots = solve_quartic_check(a, b, c, d, e, roots);
		}
	return 0;
	}
/*...e*/
/*...e*/
#endif
