//
// shape.C - Geometric Shapes
//
// This code should be faster than the RT code its derived from because it
// makes more efforts to not copy the intersection list. It swaps pointers.
//
// Also C++'s virtual methods are faster than the equivelent C switch's.
//
// And, the intersection logic for a sphere uses a new custom intersector.
//
// And, all normals returned are returned ready unitised, and using faster
// logic to do it too.
//
// Shame it doesn't actually go any faster, eh?
// This is probably VisualAge C++ letting me down...
//

/*...sincludes:0:*/
#include "shape.h"

/*...vshape\46\h:0:*/
/*...e*/

#define	BETTER_DIFF

/*...sIsectList helpers:0:*/
// Some useful functions

/*...sswap_isectls:0:*/
inline void swap_isectls(IsectList * & il_a, IsectList * & il_b)
	{
	IsectList *il = il_a; il_a = il_b; il_b = il;
	}
/*...e*/
/*...scopy_isectls:0:*/
inline void copy_isectls(IsectList *il_dst, const IsectList *il_src)
	{
	il_dst->n_isects = il_src->n_isects;
	for ( int i = 0; i < il_dst->n_isects; i++ )
		il_dst->isects[i] = il_src->isects[i];
	}
/*...e*/
/*...scombine_isectls:0:*/
// Given 2 intersection lists, produce a new one that is a combination of them.
//
// The in_old flag is used to determine if the point of intersection changes
// meaning from in->out to out->in, or vice-versa. If this happens, the
// sense of the normal vector must change :-
//
//          ..... .....                .....
//         .     .     .              .     .
//        .  A  . .  B  .            . A-B .
//       .     .   .     .          .     .
//     <-.   <-.   .->   .->      <-.     .-> This vector changes direction!
//       .     .   .     .          .     .
//        .     . .     .            .     .
//         .     .     .              .     .
//          ..... .....                .....

static void combine_isectls(
	Boolean (*combine)(Boolean in_a, Boolean in_b),
	const IsectList *il_a, const IsectList *il_b, IsectList *il
	)
	{
	Boolean in_a = FALSE;	// When t = -LARGEST, in_a = FALSE
	Boolean in_b = FALSE;	// When t = -LARGEST, in_b = FALSE
	Boolean in   = FALSE;	// Therefore combination starts as FALSE
	int ptr_a = 0;
	int ptr_b = 0;
	Boolean	in_new, in_old;

	il->n_isects = 0;

	// Work through both a and b, looking at nearest ones first

	while ( ptr_a < il_a->n_isects && ptr_b < il_b->n_isects )
		{
		Isect *isect;
		double t_a = il_a->isects[ptr_a].t;
		double t_b = il_b->isects[ptr_b].t;

		if ( t_a < t_b )
			{
			in_a ^= TRUE;
			in_old = in_a;
			isect = &(il_a->isects[ptr_a++]);
			}
		else if ( t_a > t_b )
			{
			in_b ^= TRUE;
			in_old = in_b;
			isect = &(il_b->isects[ptr_b++]);
			}
		else
			// Two surfaces at exactly the same place
			// Not a very frequent event, but problematical
			// Just label intersection arbitrarily as with B
			{
			in_a ^= TRUE; ptr_a++;
			in_b ^= TRUE;
			in_old = in_b;
			isect = &(il_b->isects[ptr_b++]);
			}

		if ( (in_new = (*combine)(in_a, in_b)) != in )
			// Need to keep a record of this transition
			{
			il->isects[il->n_isects] = *isect;
			il->isects[il->n_isects].entering = in = in_new;
			if ( in_new != in_old )
				il->isects[il->n_isects].negate_normal ^= TRUE;
			il->n_isects++;
			}
		}

	// Either a or b is exhausted, so one of a or b may be left

	while ( ptr_a < il_a->n_isects )
		{
		in_a ^= TRUE;
		in_old = in_a;
		if ( (in_new = (*combine)(in_a, in_b)) != in )
			// Need to keep a record of this transition
			{
			il->isects[il->n_isects] = il_a->isects[ptr_a];
			il->isects[il->n_isects].entering = in = in_new;
			if ( in_new != in_old )
				il->isects[il->n_isects].negate_normal ^= TRUE;
			il->n_isects++;
			}
		ptr_a++;
		}

	while ( ptr_b < il_b->n_isects )
		{
		in_b ^= TRUE;
		in_old = in_b;
		if ( (in_new = (*combine)(in_a, in_b)) != in )
			// Need to keep a record of this transition
			{
			il->isects[il->n_isects] = il_b->isects[ptr_b];
			il->isects[il->n_isects].entering = in = in_new;
			if ( in_new != in_old )
				il->isects[il->n_isects].negate_normal ^= TRUE;
			il->n_isects++;
			}
		ptr_b++;
		}
	}
/*...e*/
/*...sconcat_isectls:0:*/
// This is a quick case of unioning two intersection lists for use when it is
// known that they do not overlap.

static void concat_isectls(
	const IsectList *il_a, const IsectList *il_b,
	IsectList *il
	)
	{
	const IsectList *il_rest;
	int i, j;

	if ( il_a->n_isects == 0 )
		{
		copy_isectls(il, il_b);
		return;
		}

	if ( il_b->n_isects == 0 )
		{
		copy_isectls(il, il_a);
		return;
		}

	if ( il_a->isects[0].t < il_b->isects[0].t )
		{
		copy_isectls(il, il_a);
		il_rest = il_b;
		}
	else
		{
		copy_isectls(il, il_b);
		il_rest = il_a;
		}

	for ( i = 0, j = il->n_isects; i < il_rest->n_isects; i++, j++ )
		il->isects[j] = il_rest->isects[i];

	il->n_isects = j;
	}
/*...e*/

// Intersection lists

/*...sIsectList:0:*/
IsectList::IsectList(int n_isects_max)
	{
	isects = new Isect[n_isects_max];
	}
/*...e*/
/*...s\126\IsectList:0:*/
IsectList::~IsectList()
	{
	delete[] isects;
	}
/*...e*/
/*...st_after \45\ eliminate all intersections before a t value:0:*/
void IsectList::t_after(double t)
	{
	int i;

	for ( i = 0; i < n_isects; i++ )
		if ( isects[i].t >= t )
			break;

	if ( i == n_isects )
		// All behind t
		n_isects = 0;
	else if ( i != 0 )
		// Some behind and some after t
		{
		int j = 0;

		if ( isects[i].entering == FALSE )
			// Had better make an entering isect case first
			{
			isects[j  ]   = isects[i - 1];
			isects[j++].t = t;
			}

		while ( i < n_isects )
			isects[j++] = isects[i++];
		n_isects = j;
		}
	}
/*...e*/
/*...e*/
/*...sintersection logic:0:*/
// Logic to fill in IsectLists

// Intersection lists start outside the shape and transition into and out of
// it again. If the shape is solid at t=-infinity, then we start the list
// with an transition into the shape, positioned at t=-LARGEST.
// If we #define NO_SKIMP, we also ensure the intersection list ends outside
// the shape, which we do by appending a transition out of the shape.
// However, there doesn't appear to be any value in doing this.

#define	ZERO(x) ( ((x)>=-1.0e-10) && ((x)<=1.0e-10) )
#define	SIDESTEP (1.0e6)
#define	REAL(c) ZERO(imag(c))

/*...sintersect_linear_t:0:*/
static void intersect_linear_t(
	double coeff_of_t, double constant_term,
	const Xyz & p, const Xyz & q,
	const PrimShapeSurf *shape,
	IsectList *il
	)
	{
	if ( ZERO(coeff_of_t) )
		// No intersection with solid surface
		{
		if ( shape->evaluate(p) <= 0.0 )
			// Completely in solid
			{
#ifdef NO_SKIMP	
			il->n_isects = 2;
#else
			il->n_isects = 1;
#endif
			il->isects[0].t             = -LARGEST;
			il->isects[0].entering      = TRUE;
			il->isects[0].shape         = shape;
			il->isects[0].negate_normal = FALSE;
#ifdef NO_SKIMP
			il->isects[1].t             = LARGEST;
			il->isects[1].entering      = FALSE;
			il->isects[1].shape         = shape;
			il->isects[1].negate_normal = FALSE;
#endif
			}
		else
			// Completely out of solid
			il->n_isects = 0;
		}
	else
		// Intersects solid exactly once
		{
		double t = - constant_term / coeff_of_t;

		if ( shape->evaluate(p+q*(t+SIDESTEP)) <= 0.0 )
			// Entering solid
			{
#ifdef NO_SKIMP
			il->n_isects = 2;
#else
			il->n_isects = 1;
#endif
			il->isects[0].t             = t;
			il->isects[0].entering      = TRUE;
			il->isects[0].shape         = shape;
			il->isects[0].negate_normal = FALSE;
#ifdef NO_SKIMP
			il->isects[1].t             = LARGEST;
			il->isects[1].entering      = FALSE;
			il->isects[1].shape         = shape;
			il->isects[1].negate_normal = FALSE;
#endif
			}
		else
			// Leaving solid
			{
			il->n_isects = 2;
			il->isects[0].t             = -LARGEST;
			il->isects[0].entering      = TRUE;
			il->isects[0].shape         = shape;
			il->isects[0].negate_normal = FALSE;
			il->isects[1].t             = t;
			il->isects[1].entering      = FALSE;
			il->isects[1].shape         = shape;
			il->isects[1].negate_normal = FALSE;
			}
		}
	}
/*...e*/
/*...sintersect_quadratic_t:0:*/
static void intersect_quadratic_t(
	double qa, double qb, double qc,
	const Xyz & p, const Xyz & q,
	const PrimShapeSurf *shape,
	IsectList *il
	)
	{
	double qs;
	if ( ZERO(qa) )
/*...sdo linear case:16:*/
intersect_linear_t(qb, qc, p, q, shape, il);
/*...e*/
	else if ( (qs = qb*qb - 4.0*qa*qc) < 0.0 )
/*...sno real roots:16:*/
// No real roots => no intersections, all solid or all empty
{
if ( shape->evaluate(p) <= 0.0 )
	{
#ifdef NO_SKIMP
	il->n_isects = 2;
#else
	il->n_isects = 1;
#endif
	il->isects[0].t             = -LARGEST;
	il->isects[0].entering      = TRUE;
	il->isects[0].shape         = shape;
	il->isects[0].negate_normal = FALSE;
#ifdef NO_SKIMP
	il->isects[1].t             = LARGEST;
	il->isects[1].entering      = FALSE;
	il->isects[1].shape         = shape;
	il->isects[1].negate_normal = FALSE;
#endif
	}
else
	il->n_isects = 0;
}
/*...e*/
	else if ( ZERO(qs) )
/*...sone double root:16:*/
{
double	t = -qb / (qa + qa);
Boolean from_inside = ( shape->evaluate(p+q*(t-SIDESTEP)) <= 0.0 );
Boolean to_inside   = ( shape->evaluate(p+q*(t+SIDESTEP)) <= 0.0 );

#define	DUAL_SWITCH(f,t) (((f)<<1)+(t))

switch ( DUAL_SWITCH(from_inside, to_inside) )
	{
	case DUAL_SWITCH(FALSE, FALSE): // Grazing surface
		il->n_isects = 0;
		break;
	case DUAL_SWITCH(FALSE,  TRUE):	// Entering solid area
#ifdef NO_SKIMP
		il->n_isects = 2;
#else
		il->n_isects = 1;
#endif
		il->isects[0].t             = t;
		il->isects[0].entering      = TRUE;
		il->isects[0].shape         = shape;	
		il->isects[0].negate_normal = FALSE;
#ifdef NO_SKIMP
		il->isects[1].t             = LARGEST;
		il->isects[1].entering      = FALSE;
		il->isects[1].shape         = shape;	
		il->isects[1].negate_normal = FALSE;
#endif
		break;
	case DUAL_SWITCH( TRUE, FALSE):	// Leaving solid area
		il->n_isects = 2;
		il->isects[0].t             = -LARGEST;
		il->isects[0].entering      = TRUE;
		il->isects[0].shape         = shape;	
		il->isects[0].negate_normal = FALSE;
		il->isects[1].t             = t;
		il->isects[1].entering      = FALSE;
		il->isects[1].shape         = shape;	
		il->isects[1].negate_normal = FALSE;
		break;
	case DUAL_SWITCH( TRUE,  TRUE):	// Never left solid area
#ifdef NO_SKIMP
		il->n_isects = 2;
#else
		il->n_isects = 1;
#endif
		il->isects[0].t             = -LARGEST;
		il->isects[0].entering      = TRUE;
		il->isects[0].shape         = shape;	
		il->isects[0].negate_normal = FALSE;
#ifdef NO_SKIMP
		il->isects[0].t             = -LARGEST;
		il->isects[0].entering      = TRUE;
		il->isects[0].shape         = shape;	
		il->isects[0].negate_normal = FALSE;
		il->isects[1].t             = LARGEST;
		il->isects[1].entering      = FALSE;
		il->isects[1].shape         = shape;	
		il->isects[1].negate_normal = FALSE;
#endif
		break;
	}
}
/*...e*/
	else
/*...stwo roots:16:*/
// Where I say t1-SIDESTEP, I used to say the more obvious (t1+t2)*0.5.
// I changed it because if t1 and t2 are very close, inaccuracies in the
// arithmetic can cause the value of shape evaluated at (t1+t2)*0.5 to be just
// the wrong side of 0.0. The newer test is much more likely to give the right
// result as we are now evaluating the shape further away from any roots.

{
double rooted = sqrt(qs);
double t1 = (-qb - rooted) / (qa + qa);
double t2 = (-qb + rooted) / (qa + qa);

if ( t1 > t2 )
	// Ensure t1 is lower than t2
	{ double t = t1; t1 = t2; t2 = t; }

il->isects[0].entering      = TRUE;
il->isects[0].shape         = shape;
il->isects[0].negate_normal = FALSE;
il->isects[1].entering      = FALSE;
il->isects[1].shape         = shape;
il->isects[1].negate_normal = FALSE;

if ( shape->evaluate(p+q*(t1-SIDESTEP)) > 0.0 )
	// Middle part is the solid part
	{
	il->n_isects = 2;
	il->isects[0].t = t1;
	il->isects[1].t = t2;
	}
else
	// Regions before and after middle part are the solid parts
	{
#ifdef NO_SKIMP
	il->n_isects = 4;
#else
	il->n_isects = 3;
#endif
	il->isects[0].t             = -LARGEST;
	il->isects[1].t             = t1;
	il->isects[2].t             = t2;
	il->isects[2].entering      = TRUE;
	il->isects[2].shape         = shape;
	il->isects[2].negate_normal = FALSE;
#ifdef NO_SKIMP
	il->isects[3].t             = LARGEST;
	il->isects[3].entering      = FALSE;
	il->isects[3].shape         = shape;
	il->isects[3].negate_normal = FALSE;
#endif
	}
}
/*...e*/
	}
/*...e*/
/*...sintersect_quadratic_t_sphere:0:*/
// Like intersect_quadratic_t_sphere, only its known we are calling to find
// intersection points going through a sphere. So we know :-
//   a) qa can not be zero.
//      this implies there cannot be one root only.
//   b) when 1 double root, we must be skimming the outside
//   c) when there are 2 roots, the lower must be entering, the higher leaving.
// Accordingly we can make this version simpler, and thus faster.

static void intersect_quadratic_t_sphere(
	double qa, double qb, double qc,
	const Xyz & p, const Xyz & q,
	const PrimShapeSurf *shape,
	IsectList *il
	)
	{
	double qs;
	if ( (qs = qb*qb - 4.0*qa*qc) <= 0.0 )
		il->n_isects = 0;
	else
/*...stwo roots\44\ so gone through sphere:16:*/
{
double rooted = sqrt(qs);
double t1 = (-qb - rooted) / (qa + qa);
double t2 = (-qb + rooted) / (qa + qa);

if ( t1 > t2 )
	// Ensure t1 is lower than t2
	{ double t = t1; t1 = t2; t2 = t; }

il->n_isects = 2;
il->isects[0].t             = t1;
il->isects[0].entering      = TRUE;
il->isects[0].shape         = shape;
il->isects[0].negate_normal = FALSE;
il->isects[1].t             = t2;
il->isects[1].entering      = FALSE;
il->isects[1].shape         = shape;
il->isects[1].negate_normal = FALSE;
}
/*...e*/
	}
/*...e*/
/*...sintersect_cubic_t:0:*/
static void intersect_cubic_t(
	double qa, double qb, double qc, double qd,
	const Xyz & p, const Xyz & q,
	const PrimShapeSurf *shape,
	IsectList *il
	)
	{
	if ( ZERO(qa) )
/*...sdo quadratic case:16:*/
intersect_quadratic_t(qb, qc, qd, p, q, shape, il);
/*...e*/
	else
		{
		Complex r[3];
		solve_cubic(qa, qb, qc, qd, r);
		if ( REAL(r[1]) )
/*...s3 real roots:24:*/
{
double t0 = real(r[0]);
double t1 = real(r[1]);
double t2 = real(r[2]);
// Sort them in order
if ( t0 > t1 ) { double t = t0; t0 = t1; t1 = t; }
if ( t0 > t2 ) { double t = t0; t0 = t2; t2 = t; }
if ( t1 > t2 ) { double t = t1; t1 = t2; t2 = t; }
if ( shape->evaluate(p+q*(t0-SIDESTEP)) <= 0 )
	// Exiting, entering, exiting solid area
	{
	il->n_isects = 4;
	il->isects[0].t             = -LARGEST;
	il->isects[0].entering      = TRUE;
	il->isects[0].shape         = shape;	
	il->isects[0].negate_normal = FALSE;
	il->isects[1].t             = t0;
	il->isects[1].entering      = FALSE;
	il->isects[1].shape         = shape;	
	il->isects[1].negate_normal = FALSE;
	il->isects[2].t             = t1;
	il->isects[2].entering      = TRUE;
	il->isects[2].shape         = shape;	
	il->isects[2].negate_normal = FALSE;
	il->isects[3].t             = t2;
	il->isects[3].entering      = FALSE;
	il->isects[3].shape         = shape;	
	il->isects[3].negate_normal = FALSE;
	}
else
	// Entering, exiting, entering solid area
	{
#ifdef NO_SKIMP
	il->n_isects = 4;
#else
	il->n_isects = 3;
#endif
	il->isects[0].t             = t0;
	il->isects[0].entering      = TRUE;
	il->isects[0].shape         = shape;	
	il->isects[0].negate_normal = FALSE;
	il->isects[1].t             = t1;
	il->isects[1].entering      = FALSE;
	il->isects[1].shape         = shape;	
	il->isects[1].negate_normal = FALSE;
	il->isects[2].t             = t2;
	il->isects[2].entering      = TRUE;
	il->isects[2].shape         = shape;	
	il->isects[2].negate_normal = FALSE;
#ifdef NO_SKIMP
	il->isects[3].t             = LARGEST;
	il->isects[3].entering      = FALSE;
	il->isects[3].shape         = shape;	
	il->isects[3].negate_normal = FALSE;
#endif
	}
}
/*...e*/
		else
/*...s1 real root:24:*/
{
double t = real(r[0]); // first root will be the real one
if ( shape->evaluate(p+q*(t-SIDESTEP)) <= 0 )
	// Exiting solid area
	{
	il->n_isects = 2;
	il->isects[0].t             = -LARGEST;
	il->isects[0].entering      = TRUE;
	il->isects[0].shape         = shape;	
	il->isects[0].negate_normal = FALSE;
	il->isects[1].t             = t;
	il->isects[1].entering      = FALSE;
	il->isects[1].shape         = shape;	
	il->isects[1].negate_normal = FALSE;
	}
else
	// Entering solid area
	{
#ifdef NO_SKIMP
	il->n_isects = 2;
#else
	il->n_isects = 1;
#endif
	il->isects[0].t             = t;
	il->isects[0].entering      = TRUE;
	il->isects[0].shape         = shape;	
	il->isects[0].negate_normal = FALSE;
#ifdef NO_SKIMP
	il->isects[1].t             = LARGEST;
	il->isects[1].entering      = FALSE;
	il->isects[1].shape         = shape;	
	il->isects[1].negate_normal = FALSE;
#endif
	}
}
/*...e*/
		}
	}
/*...e*/
/*...sintersect_quartic_t:0:*/
static void intersect_quartic_t(
	double qa, double qb, double qc, double qd, double qe,
	const Xyz & p, const Xyz & q,
	const PrimShapeSurf *shape,
	IsectList *il
	)
	{
	if ( ZERO(qa) )
/*...sdo cubic case:16:*/
intersect_cubic_t(qb, qc, qd, qe, p, q, shape, il);
/*...e*/
	else
		{
		Complex r[4];
		int n_r = solve_quartic(qa, qb, qc, qd, qe, r);

		// Capture real roots
		double t[4];
		int n_t = 0;
		for ( int i = 0; i < n_r; i++ )
			if ( REAL(r[i]) )
				t[n_t++] = real(r[i]);

		if ( n_t == 4 )
/*...s4 real roots:24:*/
{
// Sort them in order
if ( t[0] > t[1] ) { double tt = t[0]; t[0] = t[1]; t[1] = tt; }
if ( t[0] > t[2] ) { double tt = t[0]; t[0] = t[2]; t[2] = tt; }
if ( t[0] > t[3] ) { double tt = t[0]; t[0] = t[3]; t[3] = tt; }
if ( t[1] > t[2] ) { double tt = t[1]; t[1] = t[2]; t[2] = tt; }
if ( t[1] > t[3] ) { double tt = t[1]; t[1] = t[3]; t[3] = tt; }
if ( t[2] > t[3] ) { double tt = t[2]; t[2] = t[3]; t[3] = tt; }
if ( shape->evaluate(p+q*(t[0]-SIDESTEP)) <= 0 )
	// Exiting, entering, exiting, entering solid area
	{
#ifdef NO_SKIMP
	il->n_isects = 6;
#else
	il->n_isects = 5;
#endif
	il->isects[0].t             = -LARGEST;
	il->isects[0].entering      = TRUE;
	il->isects[0].shape         = shape;	
	il->isects[0].negate_normal = FALSE;
	il->isects[1].t             = t[0];
	il->isects[1].entering      = FALSE;
	il->isects[1].shape         = shape;	
	il->isects[1].negate_normal = FALSE;
	il->isects[2].t             = t[1];
	il->isects[2].entering      = TRUE;
	il->isects[2].shape         = shape;	
	il->isects[2].negate_normal = FALSE;
	il->isects[3].t             = t[2];
	il->isects[3].entering      = FALSE;
	il->isects[3].shape         = shape;	
	il->isects[3].negate_normal = FALSE;
	il->isects[4].t             = t[3];
	il->isects[4].entering      = TRUE;
	il->isects[4].shape         = shape;	
	il->isects[4].negate_normal = FALSE;
#ifdef NO_SKIMP
	il->isects[5].t             = LARGEST;
	il->isects[5].entering      = FALSE;
	il->isects[5].shape         = shape;	
	il->isects[5].negate_normal = FALSE;
#endif
	}
else
	// Entering, exiting, entering, exiting solid area
	{
	il->n_isects = 4;
	il->isects[0].t             = t[0];
	il->isects[0].entering      = TRUE;
	il->isects[0].shape         = shape;	
	il->isects[0].negate_normal = FALSE;
	il->isects[1].t             = t[1];
	il->isects[1].entering      = FALSE;
	il->isects[1].shape         = shape;	
	il->isects[1].negate_normal = FALSE;
	il->isects[2].t             = t[2];
	il->isects[2].entering      = TRUE;
	il->isects[2].shape         = shape;	
	il->isects[2].negate_normal = FALSE;
	il->isects[3].t             = t[3];
	il->isects[3].entering      = FALSE;
	il->isects[3].shape         = shape;	
	il->isects[3].negate_normal = FALSE;
	}
}
/*...e*/
		else if ( n_t >= 2 )
/*...s2 real roots:24:*/
{
// Sort them in order
if ( t[0] > t[1] ) { double tt = t[0]; t[0] = t[1]; t[1] = tt; }
if ( shape->evaluate(p+q*(t[0]-SIDESTEP)) <= 0 )
	// Exiting, entering, solid area
	{
#ifdef NO_SKIMP
	il->n_isects = 4;
#else
	il->n_isects = 3;
#endif
	il->isects[0].t             = -LARGEST;
	il->isects[0].entering      = TRUE;
	il->isects[0].shape         = shape;	
	il->isects[0].negate_normal = FALSE;
	il->isects[1].t             = t[0];
	il->isects[1].entering      = FALSE;
	il->isects[1].shape         = shape;	
	il->isects[1].negate_normal = FALSE;
	il->isects[2].t             = t[1];
	il->isects[2].entering      = TRUE;
	il->isects[2].shape         = shape;	
	il->isects[2].negate_normal = FALSE;
#ifdef NO_SKIMP
	il->isects[3].t             = LARGEST;
	il->isects[3].entering      = FALSE;
	il->isects[3].shape         = shape;	
	il->isects[3].negate_normal = FALSE;
#endif
	}
else
	// Entering, exiting solid area
	{
	il->n_isects = 2;
	il->isects[0].t             = t[0];
	il->isects[0].entering      = TRUE;
	il->isects[0].shape         = shape;	
	il->isects[0].negate_normal = FALSE;
	il->isects[1].t             = t[1];
	il->isects[1].entering      = FALSE;
	il->isects[1].shape         = shape;	
	il->isects[1].negate_normal = FALSE;
	}
}
/*...e*/
		else
/*...sno real roots:24:*/
{
if ( shape->evaluate(p) <= 0 )
	// In solid area
	{
#ifdef NO_SKIMP
	il->n_isects = 2;
#else
	il->n_isects = 1;
#endif
	il->isects[0].t             = -LARGEST;
	il->isects[0].entering      = TRUE;
	il->isects[0].shape         = shape;	
	il->isects[0].negate_normal = FALSE;
#ifdef NO_SKIMP
	il->isects[1].t             = LARGEST;
	il->isects[1].entering      = FALSE;
	il->isects[1].shape         = shape;	
	il->isects[1].negate_normal = FALSE;
#endif
	}
else
	// Not in solid area
	il->n_isects = 0;
}
/*...e*/
		}
	}
/*...e*/
/*...e*/

/*...ssimpex \45\ try to be more aggressive in rejecting sub\45\shapes\44\ using extents:0:*/
// eg: A suitably aligned cubiod will return a finite extent from extent().
// We can use this information to reject a cubiod entirely the wrong side of
// the plane, despite none of the cuboids faces being aligned with the plane.

inline Boolean testpoint(
	double x, double y, double z,
	double pa, double pb, double pc, double pd
	)
	{
	return pa*x + pb*y + pc*z + pd <= 0;
	}

inline Boolean testextent(
	const Extent & ext,
	double pa, double pb, double pc, double pd
	)
	{
	return testpoint(ext.xmin.x, ext.xmin.y, ext.xmin.z, pa,pb,pc,pd) ||
	       testpoint(ext.xmin.x, ext.xmin.y, ext.xmax.z, pa,pb,pc,pd) ||
	       testpoint(ext.xmin.x, ext.xmax.y, ext.xmin.z, pa,pb,pc,pd) ||
	       testpoint(ext.xmin.x, ext.xmax.y, ext.xmax.z, pa,pb,pc,pd) ||
	       testpoint(ext.xmax.x, ext.xmin.y, ext.xmin.z, pa,pb,pc,pd) ||
	       testpoint(ext.xmax.x, ext.xmin.y, ext.xmax.z, pa,pb,pc,pd) ||
	       testpoint(ext.xmax.x, ext.xmax.y, ext.xmin.z, pa,pb,pc,pd) ||
	       testpoint(ext.xmax.x, ext.xmax.y, ext.xmax.z, pa,pb,pc,pd) ;
	}

static PrimShape *simpex(
	PrimShape *s, double pa, double pb, double pc, double pd
	)
	{
	if ( s == 0 )
		return 0;
	if ( testextent(s->extent(), pa,pb,pc,pd) )
		return s;
	delete s;
	return 0;
	}
/*...e*/

/*...sPrimShapeEmpty:0:*/
class PrimShapeEmpty : public PrimShape
	{
public:
	PrimShapeEmpty();
	virtual ~PrimShapeEmpty();
	virtual PrimShape *copy() const;
	virtual PrimShape *trans  (Xyz    t     ) const;
	virtual PrimShape *trans_x(double t     ) const;
	virtual PrimShape *trans_y(double t     ) const;
	virtual PrimShape *trans_z(double t     ) const;
	virtual PrimShape *rot_x  (double angle ) const;
	virtual PrimShape *rot_y  (double angle ) const;
	virtual PrimShape *rot_z  (double angle ) const;
	virtual PrimShape *scale  (Xyz    factor) const;
	virtual PrimShape *scale  (double factor) const;
	virtual PrimShape *scale_x(double factor) const;
	virtual PrimShape *scale_y(double factor) const;
	virtual PrimShape *scale_z(double factor) const;
	virtual PrimShape *resurf(Surf s) const;
	virtual Xyz normal(const Xyz & p) const;
	virtual Extent extent();
	virtual int isects_reqd() const;
	virtual int isect_depth() const;
	virtual void intersect(const Xyz & p, const Xyz & q, IsectList *ils[]) const;
	};

/*...sPrimShapeEmpty:0:*/
PrimShapeEmpty::PrimShapeEmpty() {}
/*...e*/
/*...s\126\PrimShapeEmpty:0:*/
PrimShapeEmpty::~PrimShapeEmpty() {}
/*...e*/
/*...scopy:0:*/
PrimShape *PrimShapeEmpty::copy() const { return new PrimShapeEmpty(); }
/*...e*/
/*...strans\39\s:0:*/
PrimShape *PrimShapeEmpty::trans  (Xyz    t) const { t=t; return copy(); }
PrimShape *PrimShapeEmpty::trans_x(double t) const { t=t; return copy(); }
PrimShape *PrimShapeEmpty::trans_y(double t) const { t=t; return copy(); }
PrimShape *PrimShapeEmpty::trans_z(double t) const { t=t; return copy(); }
/*...e*/
/*...srot\39\s:0:*/
PrimShape *PrimShapeEmpty::rot_x(double angle ) const
	{ angle=angle; return copy(); }
PrimShape *PrimShapeEmpty::rot_y(double angle ) const
	{ angle=angle; return copy(); }
PrimShape *PrimShapeEmpty::rot_z(double angle ) const
	{ angle=angle; return copy(); }
/*...e*/
/*...sscale\39\s:0:*/
PrimShape *PrimShapeEmpty::scale  (Xyz    factor) const
	{ factor=factor; return copy(); }
PrimShape *PrimShapeEmpty::scale  (double factor) const
	{ factor=factor; return copy(); }
PrimShape *PrimShapeEmpty::scale_x(double factor) const
	{ factor=factor; return copy(); }
PrimShape *PrimShapeEmpty::scale_y(double factor) const
	{ factor=factor; return copy(); }
PrimShape *PrimShapeEmpty::scale_z(double factor) const
	{ factor=factor; return copy(); }
/*...e*/
/*...sresurf:0:*/
PrimShape *PrimShapeEmpty::resurf(Surf s) const { return copy(); }
/*...e*/
/*...snormal:0:*/
Xyz PrimShapeEmpty::normal(const Xyz & p) const { return Xyz(0.0,0.0,0.0); } // Return Null vector
/*...e*/
/*...sextent:0:*/
Extent PrimShapeEmpty::extent() { return Extent(); } // Return empty extent
/*...e*/
/*...sisects_reqd:0:*/
int PrimShapeEmpty::isects_reqd() const { return 0; } // No intersections possible
/*...e*/
/*...sisect_depth:0:*/
int PrimShapeEmpty::isect_depth() const { return 1; }
/*...e*/
/*...sintersect:0:*/
void PrimShapeEmpty::intersect(const Xyz & p, const Xyz & q, IsectList *ils[]) const
	{ ils[0]->n_isects = 0; }
/*...e*/
/*...e*/
/*...sPrimShapePlane:2:*/
// ax+by+cz+d<= 0

class PrimShapePlane : public PrimShapeSurf
	{
	double a, b, c, d;
public:
	PrimShapePlane(double a, double b, double c, double d, Surf surf);
	virtual ~PrimShapePlane();
	virtual PrimShape *copy() const;
	virtual PrimShape *trans  (Xyz    t     ) const;
	virtual PrimShape *trans_x(double t     ) const;
	virtual PrimShape *trans_y(double t     ) const;
	virtual PrimShape *trans_z(double t     ) const;
	virtual PrimShape *rot_x  (double angle ) const;
	virtual PrimShape *rot_y  (double angle ) const;
	virtual PrimShape *rot_z  (double angle ) const;
	virtual PrimShape *scale  (Xyz    factor) const;
	virtual PrimShape *scale  (double factor) const;
	virtual PrimShape *scale_x(double factor) const;
	virtual PrimShape *scale_y(double factor) const;
	virtual PrimShape *scale_z(double factor) const;
	virtual PrimShape *resurf(Surf s) const;
	virtual Xyz normal(const Xyz & p) const;
	virtual Extent extent();
	virtual void preprocess();
	virtual PrimShape *simplify(
		double pa, double pb, double pc, double pd) const;
	virtual int isects_reqd() const;
	virtual double evaluate(const Xyz & v) const;
	virtual void intersect(const Xyz & p, const Xyz & q, IsectList *ils[]) const;
	};

/*...sPrimShapePlane:2:*/
PrimShapePlane::PrimShapePlane(double a, double b, double c, double d, Surf surf)
	:
	PrimShapeSurf(surf), a(a), b(b), c(c), d(d)
	{
	}
/*...e*/
/*...s\126\PrimShapePlane:2:*/
PrimShapePlane::~PrimShapePlane() {}
/*...e*/
/*...scopy:2:*/
PrimShape *PrimShapePlane::copy() const { return new PrimShapePlane(a, b, c, d, surf); }
/*...e*/
/*...strans\39\s:2:*/
PrimShape *PrimShapePlane::trans  (Xyz    t) const
	{ return new PrimShapePlane(a,b,c,d-a*t.x-b*t.y-c*t.z, surf.trans(t)); }

PrimShape *PrimShapePlane::trans_x(double t) const
	{ return new PrimShapePlane(a,b,c,d-a*t, surf.trans_x(t)); }

PrimShape *PrimShapePlane::trans_y(double t) const
	{ return new PrimShapePlane(a,b,c,d-b*t, surf.trans_y(t)); }

PrimShape *PrimShapePlane::trans_z(double t) const
	{ return new PrimShapePlane(a,b,c,d-c*t, surf.trans_z(t)); }
/*...e*/
/*...srot\39\s:2:*/
PrimShape *PrimShapePlane::rot_x(double angle) const
	{
	double ca = cos(angle), sa = sin(angle);
	return new PrimShapePlane(a, b*ca-c*sa, b*sa+c*ca, d, surf.rot_x(angle));
	}

PrimShape *PrimShapePlane::rot_y(double angle) const
	{
	double ca = cos(angle), sa = sin(angle);
	return new PrimShapePlane(c*sa+a*ca, b, c*ca-a*sa, d, surf.rot_y(angle));
	}

PrimShape *PrimShapePlane::rot_z(double angle) const
	{
	double ca = cos(angle), sa = sin(angle);
	return new PrimShapePlane(a*ca-b*sa, a*sa+b*ca, c, d, surf.rot_z(angle));
	}
/*...e*/
/*...sscale\39\s:2:*/
PrimShape *PrimShapePlane::scale  (Xyz    factor) const
	{ return new PrimShapePlane(a/factor.x, b/factor.y, c/factor.z, d, surf.scale(factor)); }

PrimShape *PrimShapePlane::scale  (double factor) const
	{ return new PrimShapePlane(a/factor, b/factor, c/factor, d, surf.scale(factor)); }

PrimShape *PrimShapePlane::scale_x(double factor) const
	{ return new PrimShapePlane(a/factor, b, c, d, surf.scale_x(factor)); }

PrimShape *PrimShapePlane::scale_y(double factor) const
	{ return new PrimShapePlane(a, b/factor, c, d, surf.scale_y(factor)); }

PrimShape *PrimShapePlane::scale_z(double factor) const
	{ return new PrimShapePlane(a, b, c/factor, d, surf.scale_z(factor)); }
/*...e*/
/*...sresurf:2:*/
PrimShape *PrimShapePlane::resurf(Surf surf) const { return new PrimShapePlane(a, b, c, d, surf); }
/*...e*/
/*...snormal:2:*/
// Use partial derivatives to find normal at a given point.
//
// d/dx:	a
// d/dy:	b
// d/dz:	c

Xyz PrimShapePlane::normal(const Xyz & p) const { return Xyz(a,b,c); }
/*...e*/
/*...sextent:2:*/
// Quite a lot of models have their plane faces aligned along x,y y,z or x,z
// planes. Hence this code should discover that fact quite often.

Extent PrimShapePlane::extent()
	{
	Extent ext(Xyz(-LARGEST,-LARGEST,-LARGEST),
	           Xyz( LARGEST, LARGEST, LARGEST));

	if ( b == 0.0 && c == 0.0 )
		/* ax<=-d */
		{
		if ( a >= 0.0 )
			ext.xmax.x = -d/a;
		else
			ext.xmin.x = -d/a;
		}
	else if ( a == 0.0 && c == 0.0 )
		/* by<=-d */
		{
		if ( b >= 0.0 )
			ext.xmax.y = -d/b;
		else
			ext.xmin.y = -d/b;
		}
	else if ( a == 0.0 && b == 0.0 )
		/* cz<=-d */
		{
		if ( c >= 0.0 )
			ext.xmax.z = -d/c;
		else
			ext.xmin.z = -d/c;
		}

	return ext;
	}
/*...e*/
/*...spreprocess:2:*/
// If we ensure that the a*a+b*b+c*c == 1, then we can be sure we will
// return unit normal vectors when asked, without any need to unitise.

void PrimShapePlane::preprocess()
	{
	double len = sqrt(a*a+b*b+c*c);
	a /= len; b /= len; c /= len; d /= len;
	}
/*...e*/
/*...ssimplify:2:*/
// To be able to eliminate the plane, the normals must be opposite
// Normal of plane is unitised too
//  ax+by+cz+d<=0
// -ax-by-cz-d>=0
// -ax-by-cz>=d, compare with pax+pby+pcz<=-pd

PrimShape *PrimShapePlane::simplify(double pa, double pb, double pc, double pd) const
	{
	return ( pa == -a && pb == -b && pc == -c && d > -pd ) ? 0 : copy();
	}
/*...e*/
/*...sisects_reqd:2:*/
// Remember, intersections might be -LARGEST, x or x, LARGEST

int PrimShapePlane::isects_reqd() const { return 2; }
/*...e*/
/*...sevaluate:2:*/
double PrimShapePlane::evaluate(const Xyz & v) const
	{
	return a*v.x + b*v.y + c*v.z + d;
	}
/*...e*/
/*...sintersect:2:*/
// Any point along the line we are interested in is of the form p + tq.
// 								~    ~
// Given:	ax+by+cz+d=0
// 
// Subs:	x = x +tx		y = y +ty		z = z +tz
// 		     p   q		     p   q		     p   q
// 
// Gives:	(ax +by +cz )t + ax +by +cz +d = 0
// 		   q   q   q       p   p   p

void PrimShapePlane::intersect(const Xyz & p, const Xyz & q, IsectList *ils[]) const
	{
	double coeff_of_t    = a * q.x + b * q.y + c * q.z;
	double constant_term = a * p.x + b * p.y + c * p.z + d;
	intersect_linear_t(coeff_of_t, constant_term, p, q, this, ils[0]);
	}
/*...e*/
/*...e*/
/*...sPrimShapeBiPlane:2:*/
// ax+by+cz+d1 >  0 and
// ax+by+cz+d2 <= 0

class PrimShapeBiPlane : public PrimShapeSurf
	{
	double a, b, c, d1, d2;
public:
	PrimShapeBiPlane(
		double a, double b, double c, double d1, double d2,
		Surf surf
		);
	virtual ~PrimShapeBiPlane();
	virtual PrimShape *copy() const;
	virtual PrimShape *trans  (Xyz    t     ) const;
	virtual PrimShape *trans_x(double t     ) const;
	virtual PrimShape *trans_y(double t     ) const;
	virtual PrimShape *trans_z(double t     ) const;
	virtual PrimShape *rot_x  (double angle ) const;
	virtual PrimShape *rot_y  (double angle ) const;
	virtual PrimShape *rot_z  (double angle ) const;
	virtual PrimShape *scale  (Xyz    factor) const;
	virtual PrimShape *scale  (double factor) const;
	virtual PrimShape *scale_x(double factor) const;
	virtual PrimShape *scale_y(double factor) const;
	virtual PrimShape *scale_z(double factor) const;
	virtual PrimShape *resurf(Surf s) const;
	virtual Xyz normal(const Xyz & p) const;
	virtual Extent extent();
	virtual void preprocess();
	virtual PrimShape *simplify(
		double pa, double pb, double pc, double pd) const;
	virtual int isects_reqd() const;
	virtual void intersect(const Xyz & p, const Xyz & q, IsectList *ils[]) const;
	};

/*...sPrimShapeBiPlane:2:*/
PrimShapeBiPlane::PrimShapeBiPlane(
	double a, double b, double c, double d1, double d2,
	Surf surf
	)
	:
	PrimShapeSurf(surf), a(a), b(b), c(c), d1(d1), d2(d2)
	{
	}
/*...e*/
/*...s\126\PrimShapeBiPlane:2:*/
PrimShapeBiPlane::~PrimShapeBiPlane() {}
/*...e*/
/*...scopy:2:*/
PrimShape *PrimShapeBiPlane::copy() const
	{
	return new PrimShapeBiPlane(a, b, c, d1, d2, surf);
	}
/*...e*/
/*...strans\39\s:2:*/
PrimShape *PrimShapeBiPlane::trans  (Xyz t) const
	{
	double d = a*t.x+b*t.y+c*t.z;
	return new PrimShapeBiPlane(a,b,c,d1-d,d2-d, surf.trans(t));
	}

PrimShape *PrimShapeBiPlane::trans_x(double t) const
	{
	double d = a*t;
	return new PrimShapeBiPlane(a,b,c,d1-d,d2-d, surf.trans_x(t));
	}

PrimShape *PrimShapeBiPlane::trans_y(double t) const
	{
	double d = b*t;
	return new PrimShapeBiPlane(a,b,c,d1-d,d2-d, surf.trans_y(t));
	}

PrimShape *PrimShapeBiPlane::trans_z(double t) const
	{
	double d = c*t;
	return new PrimShapeBiPlane(a,b,c,d1-d,d2-d, surf.trans_z(t));
	}
/*...e*/
/*...srot\39\s:2:*/
PrimShape *PrimShapeBiPlane::rot_x(double angle) const
	{
	double ca = cos(angle), sa = sin(angle);
	return new PrimShapeBiPlane(a, b*ca-c*sa, b*sa+c*ca, d1, d2, surf.rot_x(angle));
	}

PrimShape *PrimShapeBiPlane::rot_y(double angle) const
	{
	double ca = cos(angle), sa = sin(angle);
	return new PrimShapeBiPlane(c*sa+a*ca, b, c*ca-a*sa, d1, d2, surf.rot_y(angle));
	}

PrimShape *PrimShapeBiPlane::rot_z(double angle) const
	{
	double ca = cos(angle), sa = sin(angle);
	return new PrimShapeBiPlane(a*ca-b*sa, a*sa+b*ca, c, d1, d2, surf.rot_z(angle));
	}
/*...e*/
/*...sscale\39\s:2:*/
PrimShape *PrimShapeBiPlane::scale  (Xyz    factor) const
	{ return new PrimShapeBiPlane(a/factor.x, b/factor.y, c/factor.z, d1, d2, surf.scale(factor)); }

PrimShape *PrimShapeBiPlane::scale  (double factor) const
	{ return new PrimShapeBiPlane(a/factor, b/factor, c/factor, d1, d2, surf.scale(factor)); }

PrimShape *PrimShapeBiPlane::scale_x(double factor) const
	{ return new PrimShapeBiPlane(a/factor, b, c, d1, d2, surf.scale_x(factor)); }

PrimShape *PrimShapeBiPlane::scale_y(double factor) const
	{ return new PrimShapeBiPlane(a, b/factor, c, d1, d2, surf.scale_y(factor)); }

PrimShape *PrimShapeBiPlane::scale_z(double factor) const
	{ return new PrimShapeBiPlane(a, b, c/factor, d1, d2, surf.scale_z(factor)); }
/*...e*/
/*...sresurf:2:*/
PrimShape *PrimShapeBiPlane::resurf(Surf surf) const
	{
	return new PrimShapeBiPlane(a, b, c, d1, d2, surf);
	}
/*...e*/
/*...snormal:2:*/
// This is much like the simple plane case except that we know need to know the
// point of intersection in order to know which of the 2 half-plane surfaces
// were involved. We compute a d value, and if its closest to d2, then great,
// else the normal is the other way around.

Xyz PrimShapeBiPlane::normal(const Xyz & p) const
	{
	double d = - ( a * p.x + b * p.y + c * p.z );
	if ( fabs(d - d2) < fabs(d - d1) )
		return Xyz( a,  b,  c);
	else
		return Xyz(-a, -b, -c);
	}
/*...e*/
/*...sextent:2:*/
// Quite a lot of models have their plane faces aligned along x,y y,z or x,z
// planes. Hence this code should discover that fact quite often.

Extent PrimShapeBiPlane::extent()
	{
	Extent ext(Xyz(-LARGEST,-LARGEST,-LARGEST),
	           Xyz( LARGEST, LARGEST, LARGEST));

	if ( b == 0.0 && c == 0.0 )
		// ax>-d1 and ax<=-d2
		{
		if ( a >= 0.0 )
			{
			ext.xmin.x = -d1/a;
			ext.xmax.x = -d2/a;
			}
		else
			{
			ext.xmin.x = -d2/a;
			ext.xmax.x = -d1/a;
			}
		}
	else if ( a == 0.0 && c == 0.0 )
		// by>-d1 and by<=-d2
		{
		if ( b >= 0.0 )
			{
			ext.xmin.y = -d1/b;
			ext.xmax.y = -d2/b;
			}
		else
			{
			ext.xmin.y = -d2/b;
			ext.xmax.y = -d1/b;
			}
		}
	else if ( a == 0.0 && b == 0.0 )
		// cz>-d1 and cz<=-d2
		{
		if ( c >= 0.0 )
			{
			ext.xmin.z = -d1/c;
			ext.xmax.z = -d2/c;
			}
		else
			{
			ext.xmin.z = -d2/c;
			ext.xmax.z = -d1/c;
			}
		}

	return ext;
	}
/*...e*/
/*...spreprocess:2:*/
// If we ensure that the a*a+b*b+c*c == 1, then we can be sure we will
// return unit normal vectors when asked, without any need to unitise.

void PrimShapeBiPlane::preprocess()
	{
	double len = sqrt(a*a+b*b+c*c);
	a /= len; b /= len; c /= len; d1 /= len; d2 /= len;
	}
/*...e*/
/*...ssimplify:2:*/
// ax+by+cz+d2<=0
// ax+by+cz+d1> 0
// ax+by+cz>-d1, compare with pax+pby+pcz<=-pd

PrimShape *PrimShapeBiPlane::simplify(double pa, double pb, double pc, double pd) const
	{
	return ( pa == -a && pb == -b && pc == -c && -d1 > -pd ) ? 0 : copy();
	}
/*...e*/
/*...sisects_reqd:2:*/
int PrimShapeBiPlane::isects_reqd() const { return 2; }
/*...e*/
/*...sintersect:2:*/
// Any point along the line we are interested in is of the form p + tq.
//								~    ~
// Given:	ax+by+cz+d1 >0 and
//		ax+by+cz+d2<=0
//
// Gives:	-d1<ax+by+cz<=d2
//
// Subs:	x = x +tx		y = y +ty		z = z +tz
//		     p   q		     p   q		     p   q
//
// Gives:	(ax +by +cz )t + ax +by +cz +d = 0
//	 	   q   q   q       p   p   p

void PrimShapeBiPlane::intersect(const Xyz & p, const Xyz & q, IsectList *ils[]) const
	{
	IsectList *il = ils[0];
	double coeff_of_t = a * q.x + b * q.y + c * q.z;
	double axbycz     = a * p.x + b * p.y + c * p.z;

	if ( ZERO(coeff_of_t) )
		// No intersection with solid surface
		{
		if ( axbycz > -d1 && axbycz <= -d2 )
			// Completely in solid
			{
#ifndef NO_SKIMP
			il->n_isects = 1;
			il->isects[0].t             = -LARGEST;
			il->isects[0].entering      = TRUE;
			il->isects[0].shape         = this;
			il->isects[0].negate_normal = FALSE;
#else
			il->n_isects = 2;
			il->isects[0].t             = -LARGEST;
			il->isects[0].entering      = TRUE;
			il->isects[0].shape         = this;
			il->isects[0].negate_normal = FALSE;
			il->isects[1].t             = LARGEST;
			il->isects[1].entering      = FALSE;
			il->isects[1].shape         = this;
			il->isects[1].negate_normal = FALSE;
#endif
			}
		else
			// Completely out of solid
			il->n_isects = 0;
		}
	else
		// Intersects solid exactly twice
		{
		double t1 = - (axbycz + d1) / coeff_of_t;
		double t2 = - (axbycz + d2) / coeff_of_t;

		il->n_isects = 2;
		il->isects[0].entering      = TRUE;
		il->isects[0].shape         = this;
		il->isects[0].negate_normal = FALSE;
		il->isects[1].entering      = FALSE;
		il->isects[1].shape         = this;
		il->isects[1].negate_normal = FALSE;
		if ( t1 < t2 )
			{
			il->isects[0].t = t1;
			il->isects[1].t = t2;
			}
		else
			{
			il->isects[0].t = t2;
			il->isects[1].t = t1;
			}
		}
	}
/*...e*/
/*...e*/
/*...sPrimShapeQuad:2:*/
// 			  2   2   2
// Defined solid for :- ax +by +cz +dxy+eyz+fzx+gx+hy+iz+j<=0
//
// Ellisoids, spheres, infinitely long cylinders, double cones and planes are
// all special case of this equation.

class PrimShapeQuad : public PrimShapeSurf
	{
	double a, b, c, d, e, f, g, h, i, j;
public:
	PrimShapeQuad(
		double a, double b, double c,
		double d, double e, double f,
		double g, double h, double i,
		double j,
		Surf surf
		);
	virtual ~PrimShapeQuad();
	virtual PrimShape *copy() const;
	virtual PrimShape *trans  (Xyz    t     ) const;
	virtual PrimShape *trans_x(double t     ) const;
	virtual PrimShape *trans_y(double t     ) const;
	virtual PrimShape *trans_z(double t     ) const;
	virtual PrimShape *rot_x  (double angle ) const;
	virtual PrimShape *rot_y  (double angle ) const;
	virtual PrimShape *rot_z  (double angle ) const;
	virtual PrimShape *scale  (Xyz    factor) const;
	virtual PrimShape *scale  (double factor) const;
	virtual PrimShape *scale_x(double factor) const;
	virtual PrimShape *scale_y(double factor) const;
	virtual PrimShape *scale_z(double factor) const;
	virtual PrimShape *resurf(Surf s) const;
	virtual Xyz normal(const Xyz & p) const;
	virtual Extent extent();
	virtual int isects_reqd() const;
	virtual double evaluate(const Xyz & v) const;
	virtual void intersect(const Xyz & p, const Xyz & q, IsectList *ils[]) const;
	};

/*...sPrimShapeQuad:2:*/
PrimShapeQuad::PrimShapeQuad(
	double a, double b, double c,
	double d, double e, double f,
	double g, double h, double i,
	double j, Surf surf
	)
	:
	PrimShapeSurf(surf),
	a(a), b(b), c(c),
	d(d), e(e), f(f),
	g(g), h(h), i(i),
	j(j)
	{
	}
/*...e*/
/*...s\126\PrimShapeQuad:2:*/
PrimShapeQuad::~PrimShapeQuad() {}
/*...e*/
/*...scopy:2:*/
PrimShape *PrimShapeQuad::copy() const
	{
	return new PrimShapeQuad(a, b, c, d, e, f, g, h, i, j, surf);
	}
/*...e*/
/*...strans\39\s:2:*/
PrimShape *PrimShapeQuad::trans(Xyz t) const
	{
	PrimShape *p1 =     trans_x(t.x);
	PrimShape *p2 = p1->trans_y(t.y);
	PrimShape *p3 = p2->trans_z(t.z);
	delete p1;
	delete p2;
	return p3;
	}

PrimShape *PrimShapeQuad::trans_x(double t) const
	{
	return new PrimShapeQuad(a,b,c,d,e,f,g-2.0*a*t,h-d*t,i-f*t,j+a*t*t-g*t, surf.trans_x(t));
	}

PrimShape *PrimShapeQuad::trans_y(double t) const
	{
	return new PrimShapeQuad(a,b,c,d,e,f,g-d*t,h-2.0*b*t,i-e*t,j+b*t*t-h*t, surf.trans_y(t));
	}

PrimShape *PrimShapeQuad::trans_z(double t) const
	{
	return new PrimShapeQuad(a,b,c,d,e,f,g-f*t,h-e*t,i-2.0*c*t,j+c*t*t-i*t, surf.trans_z(t));
	}
/*...e*/
/*...srot\39\s:2:*/
PrimShape *PrimShapeQuad::rot_x(double angle) const
	{
	double ca=cos(angle), sa=sin(angle), caca=ca*ca, sasa=sa*sa, saca=sa*ca;
	return new PrimShapeQuad(
		a,
		b*caca+c*sasa-e*saca,
		b*sasa+c*caca+e*saca,
		d*ca-f*sa,
		2.0*(b-c)*saca+e*(caca-sasa),
		d*sa+f*ca,
		g,
		h*ca-i*sa,
		h*sa+i*ca,
		j,
		surf.rot_x(angle));
	}

PrimShape *PrimShapeQuad::rot_y(double angle) const
	{
	double ca=cos(angle), sa=sin(angle), caca=ca*ca, sasa=sa*sa, saca=sa*ca;
	return new PrimShapeQuad(
		a*caca+c*sasa+f*saca,
		b,
		a*sasa+c*caca-f*saca,
		e*sa+d*ca,
		e*ca-d*sa,
		2.0*(c-a)*saca+f*(caca-sasa),
		i*sa+g*ca,
		h,
		i*ca-g*sa,
		j,
		surf.rot_y(angle));
	}

PrimShape *PrimShapeQuad::rot_z(double angle) const
	{
	double ca=cos(angle), sa=sin(angle), caca=ca*ca, sasa=sa*sa, saca=sa*ca;
	return new PrimShapeQuad(
		a*caca+b*sasa-d*saca,
		a*sasa+b*caca+d*saca,
		c,
		2.0*(a-b)*saca+d*(caca-sasa),
		f*sa+e*ca,
		f*ca-e*sa,
		g*ca-h*sa,
		g*sa+h*ca,
		i,
		j,
		surf.rot_z(angle));
	}
/*...e*/
/*...sscale\39\s:2:*/
PrimShape *PrimShapeQuad::scale(Xyz factor) const
	{
	PrimShape *p1 =     scale_x(factor.x);
	PrimShape *p2 = p1->scale_y(factor.y);
	PrimShape *p3 = p2->scale_z(factor.z);
	delete p1;
	delete p2;
	return p3;
	}

PrimShape *PrimShapeQuad::scale  (double factor) const
	{ return scale(Xyz(factor,factor,factor)); }

PrimShape *PrimShapeQuad::scale_x(double factor) const
	{ return new PrimShapeQuad(a/(factor*factor),b,c,d/factor,e,f/factor,g/factor,h,i,j, surf.scale_x(factor)); }

PrimShape *PrimShapeQuad::scale_y(double factor) const
	{ return new PrimShapeQuad(a,b/(factor*factor),c,d/factor,e/factor,f,g,h/factor,i,j, surf.scale_y(factor)); }

PrimShape *PrimShapeQuad::scale_z(double factor) const
	{ return new PrimShapeQuad(a,b,c/(factor*factor),d,e/factor,f/factor,g,h,i/factor,j, surf.scale_z(factor)); }
/*...e*/
/*...sresurf:2:*/
PrimShape *PrimShapeQuad::resurf(Surf surf) const
	{
	return new PrimShapeQuad(a,b,c,d,e,f,g,h,i,j, surf);
	}
/*...e*/
/*...snormal:2:*/
// Use partial derivatives to find normal at a given point.
//
// Given:	  2   2   2
//	 	ax +by +cz +dxy+eyz+fzx+gx+hy+iz+j=0
//
// d/dx:	2ax+dy+fz+g
//
// d/dy:	2by+dx+ez+h
//
// d/dz:	2cz+ey+fx+i

Xyz PrimShapeQuad::normal(const Xyz & p) const
	{
	double ax = a*p.x, by = b*p.y, cz = c*p.z;
	return Xyz(
		ax + ax + d * p.y + f * p.z + g,
		by + by + d * p.x + e * p.z + h,
		cz + cz + e * p.y + f * p.x + i).unit();
	}
/*...e*/
/*...sextent:2:*/
// Too difficult to work this one out (well, at least for now)

Extent PrimShapeQuad::extent()
	{
	return Extent(Xyz(-LARGEST,-LARGEST,-LARGEST),
		      Xyz( LARGEST, LARGEST, LARGEST));
	}
/*...e*/
/*...sisects_reqd:2:*/
int PrimShapeQuad::isects_reqd() const { return 4; }
/*...e*/
/*...sevaluate:2:*/
// Use this to determine value of quadratic at given point in space.
// 
// 		  2   2   2
// Opt:	ax +by +cz +dxy+eyz+fzx+gx+hy+iz+j		costs 15x's and 9+'s
// 
// 	(ax+dy+g)x+(by+ez+h)y+(cz+fx+i)z+j		costs 9x's and 9+'s

double PrimShapeQuad::evaluate(const Xyz & v) const
	{
	return (a * v.x + d * v.y + g) * v.x +
	       (b * v.y + e * v.z + h) * v.y +
	       (c * v.z + f * v.x + i) * v.z +
	        j;
	}
/*...e*/
/*...sintersect:2:*/
// Those who don't like maths should page down a few times very rapidly!
// 
// Any point along the line we are interested in is of the form p + tq.
//								~    ~
//		  2   2   2
// Given:	ax +by +cz +dxy+eyz+fzx+gx+hy+iz+j=0
//
// Subs:	x = x +tx		y = y +ty		z = z +tz
//		     p   q		     p   q		     p   q
//
//		   2   2   2                    2
// Gives:	(ax +by +cz +dx y +ey z +fz x )t  +
//		   q   q   q   q q   q q   q q
//
//		(2[ax x +by y +cz z ]+d[x y +x y ]+e[y z +y z ]+f[z x +z x ]+gx +hy +iz )t +
//		     p q   p q   p q     p q  q p     p q  q p     p q  q p    q   q   q
//
//		   2   2   2
//		(ax +by +cz +dx y +ey z +fz x +gx +hy +iz +j) = 0
//		   p   p   p   p p   p p   p p   p   p   p
//
//
//		  2   2   2                   
// Opt:		ax +by +cz +dx y +ey z +fz x			costs 12x's and 5+'s
//		  q   q   q   q q   q q   q q
//
//		(ax +dy )x +(by +ez )y +(cz +fx )z		costs 9x's and 5+'s
//		   q   q  q    q   q  q    q   q  q
//
//
//		  2   2   2
// Opt:		ax +by +cz +dx y +ey z +fz x +gx +hy +iz +j	costs 15x's and 9+'s
//		  p   p   p   p p   p p   p p   p   p   p
//
//		(ax +dy +g)x +(by +ez +h)y +(cz +fx +i)z +j	costs 9x's and 9+'s
//		   p   p    p    p   p    p    p   p    p

void PrimShapeQuad::intersect(const Xyz & p, const Xyz & q, IsectList *ils[]) const
	{
	double qa = (a * q.x + d * q.y) * q.x +
		    (b * q.y + e * q.z) * q.y +
		    (c * q.z + f * q.x) * q.z;
	double xx = (a * p.x * q.x + b * p.y * q.y + c * p.z * q.z);
	double qb = xx + xx +
		    d * (p.x * q.y + q.x * p.y) +
		    e * (p.y * q.z + q.y * p.z) +
		    f * (p.z * q.x + q.z * p.x) +
		    g * q.x + h * q.y + i * q.z;
	double qc = (a * p.x + d * p.y + g) * p.x +
		    (b * p.y + e * p.z + h) * p.y +
		    (c * p.z + f * p.x + i) * p.z + j;
	intersect_quadratic_t(qa, qb, qc, p, q, this, ils[0]);
	}
/*...e*/
/*...e*/
/*...sPrimShapeSphere:2:*/
//			     2      2      2  2
// Defined solid for :- (x-a) +(y-b) +(z-c) -d <=0

class PrimShapeSphere : public PrimShapeSurf
	{
	double a, b, c, d;
public:
	PrimShapeSphere(double d, Surf surf);
	PrimShapeSphere(double a, double b, double c, double d, Surf surf);
	PrimShapeSphere(Xyz p, double r, Surf surf);
	virtual ~PrimShapeSphere();
	virtual PrimShape *copy() const;
	virtual PrimShape *trans  (Xyz    t     ) const;
	virtual PrimShape *trans_x(double t     ) const;
	virtual PrimShape *trans_y(double t     ) const;
	virtual PrimShape *trans_z(double t     ) const;
	virtual PrimShape *rot_x  (double angle ) const;
	virtual PrimShape *rot_y  (double angle ) const;
	virtual PrimShape *rot_z  (double angle ) const;
	virtual PrimShape *scale  (Xyz    factor) const;
	virtual PrimShape *scale  (double factor) const;
	virtual PrimShape *scale_x(double factor) const;
	virtual PrimShape *scale_y(double factor) const;
	virtual PrimShape *scale_z(double factor) const;
	virtual PrimShape *resurf(Surf s) const;
	virtual Xyz normal(const Xyz & p) const;
	virtual Extent extent();
	virtual void preprocess();
	virtual PrimShape *simplify(
		double pa, double pb, double pc, double pd) const;
	virtual int isects_reqd() const;
	virtual double evaluate(const Xyz & v) const;
	virtual void intersect(const Xyz & p, const Xyz & q, IsectList *ils[]) const;
	PrimShapeQuad *quad() const; // Return PrimShapeQuad equivelent
	};

/*...sPrimShapeSphere d only:2:*/
PrimShapeSphere::PrimShapeSphere(double d, Surf surf)
	: PrimShapeSurf(surf), a(0.0), b(0.0), c(0.0), d(d) {}
/*...e*/
/*...sPrimShapeSphere a\44\b\44\c\44\d:2:*/
PrimShapeSphere::PrimShapeSphere(double a, double b, double c, double d, Surf surf)
	: PrimShapeSurf(surf), a(a), b(b), c(c), d(d) {}
/*...e*/
/*...sPrimShapeSphere p\44\ r:2:*/
PrimShapeSphere::PrimShapeSphere(Xyz p, double r, Surf surf)
	: PrimShapeSurf(surf), a(p.x), b(p.y), c(p.z), d(r) {}
/*...e*/
/*...s\126\PrimShapeSphere:2:*/
PrimShapeSphere::~PrimShapeSphere() {}
/*...e*/
/*...scopy:2:*/
PrimShape *PrimShapeSphere::copy() const { return new PrimShapeSphere(a, b, c, d, surf); }
/*...e*/
/*...strans\39\s:2:*/
PrimShape *PrimShapeSphere::trans  (Xyz    t) const
	{ return new PrimShapeSphere(a+t.x,b+t.y,c+t.z,d, surf.trans(t)); }

PrimShape *PrimShapeSphere::trans_x(double t) const
	{ return new PrimShapeSphere(a+t,b,c,d, surf.trans_x(t)); }

PrimShape *PrimShapeSphere::trans_y(double t) const
	{ return new PrimShapeSphere(a,b+t,c,d, surf.trans_y(t)); }

PrimShape *PrimShapeSphere::trans_z(double t) const
	{ return new PrimShapeSphere(a,b,c+t,d, surf.trans_z(t)); }
/*...e*/
/*...srot\39\s:2:*/
PrimShape *PrimShapeSphere::rot_x(double angle) const
	{
	double ca = cos(angle), sa = sin(angle);
	return new PrimShapeSphere(a, b*ca-c*sa, b*sa+c*ca, d, surf.rot_x(angle));
	}

PrimShape *PrimShapeSphere::rot_y(double angle) const
	{
	double ca = cos(angle), sa = sin(angle);
	return new PrimShapeSphere(c*sa+a*ca, b, c*ca-a*sa, d, surf.rot_y(angle));
	}

PrimShape *PrimShapeSphere::rot_z(double angle) const
	{
	double ca = cos(angle), sa = sin(angle);
	return new PrimShapeSphere(a*ca-b*sa, a*sa+b*ca, c, d, surf.rot_z(angle));
	}
/*...e*/
/*...sscale\39\s:2:*/
// Scaling by the same amount in each direction is fine.
// Scaling by 1.0 in any direction is also fine.
// Anything else results in a non-spherical result!

PrimShape *PrimShapeSphere::scale(Xyz factor) const
	{
	if ( factor.x == factor.y && factor.y == factor.z )
		return scale(factor.x);
	PrimShapeQuad *q = quad();
	PrimShape *sq = q->scale(factor);
	delete q;
	return sq;
	}

// Suspected bugfix over RT
PrimShape *PrimShapeSphere::scale(double factor) const
	{ return new PrimShapeSphere(a*factor,b*factor,c*factor,d*factor, surf.scale(factor)); }

PrimShape *PrimShapeSphere::scale_x(double factor) const
	{
	if ( factor == 1.0 )
		return copy();
	PrimShapeQuad *q = quad();
	PrimShape *sq = q->scale_x(factor);
	delete q;
	return sq;
	}

PrimShape *PrimShapeSphere::scale_y(double factor) const
	{
	if ( factor == 1.0 )
		return copy();
	PrimShapeQuad *q = quad();
	PrimShape *sq = q->scale_y(factor);
	delete q;
	return sq;
	}

PrimShape *PrimShapeSphere::scale_z(double factor) const
	{
	if ( factor == 1.0 )
		return copy();
	PrimShapeQuad *q = quad();
	PrimShape *sq = q->scale_z(factor);
	delete q;
	return sq;
	}
/*...e*/
/*...sresurf:2:*/
PrimShape *PrimShapeSphere::resurf(Surf surf) const { return new PrimShapeSphere(a,b,c,d, surf); }
/*...e*/
/*...snormal:2:*/
// Use partial derivatives to find normal at a given point.
//
//		     2      2      2  2
// Given:	(x-a) +(y-b) +(z-c) +d =0
//
// d/dx:	2(x-a)
//
// d/dy:	2(y-b)
//
// d/dz:	2(z-c)
//
// As the intersection point p is on the surface of the sphere centred at c,
// the length of the vector p-c will be the radius of the sphere.
// This makes it easy for us to unitise the result.

Xyz PrimShapeSphere::normal(const Xyz & p) const
	{ return Xyz((p.x-a)/d, (p.y-b)/d, (p.z-c)/d); }
/*...e*/
/*...sextent:2:*/
Extent PrimShapeSphere::extent()
	{ return Extent(Xyz(a-d,b-d,c-d), Xyz(a+d,b+d,c+d)); }
/*...e*/
/*...spreprocess:2:*/
void PrimShapeSphere::preprocess() {}
/*...e*/
/*...ssimplify:2:*/
// Find distance of sphere centre from plane
// ie: compute multiple of normal from nearest point on place to sphere centre

PrimShape *PrimShapeSphere::simplify(double pa, double pb, double pc, double pd) const
	{
	return ( a*pa + b*pb + c*pc + pd <= d ) ? copy() : 0;
	}
/*...e*/
/*...sisects_reqd:2:*/
int PrimShapeSphere::isects_reqd() const { return 2; }
/*...e*/
/*...sevaluate:2:*/
double PrimShapeSphere::evaluate(const Xyz & v) const
	{
	double i = v.x - a;
	double j = v.y - b;
	double k = v.z - c;
	return i*i + j*j + k*k - d*d;
	}
/*...e*/
/*...sintersect:2:*/
// Any point along the line we are interested in is of the form p + tq.
//								~    ~
//		     2      2      2  2
// Given:	(x-a) +(y-b) +(z-c) -d =0
//
// Subs:	x = x +tx		y = y +ty		z = z +tz
//		     p   q		     p   q		     p   q
//
//		  2  2  2  2                      2  2  2  2
// Gives:	(x +y +z )t - 2(ix +jy +kz )t + (i +j +k -d ) = 0
//		  q  q  q         q   q   q
//
// Where:	i = (a-x ), j = (b-y ), k = (c-z )
//		        p           p           p

void PrimShapeSphere::intersect(const Xyz & p, const Xyz & q, IsectList *ils[]) const
	{
	double i  = a - p.x;
	double j  = b - p.y;
	double k  = c - p.z;
	double qa = q.x * q.x + q.y * q.y + q.z * q.z;
	double xx = -(i * q.x + j * q.y + k * q.z);
	double qb = xx + xx;
	double qc = i*i + j*j + k*k - d*d;
	intersect_quadratic_t_sphere(qa, qb, qc, p, q, this, ils[0]);
	}
/*...e*/
/*...squad:2:*/
// Spheres cannot be rescaled in each direction and remain a sphere.
// Hence this code will convert a sphere into the equivelent quadratic.
//
//			     2      2      2
//	Sphere:		(x-a) +(y-b) +(z-c) -d <=0
//
//			 2      2  2      2  2      2
//			x -2ax+a +y -2by+b +z -2cz+c -d <= 0
//
//			  2   2   2
//	Quadratic:	ax +by +cz +dxy+eyz+fzx+gx+hy+iz+j<=0

PrimShapeQuad *PrimShapeSphere::quad() const
	{
	return new PrimShapeQuad(1.0, 1.0, 1.0, 0.0, 0.0, 0.0,
	     -(a+a), -(b+b), -(c+c), -d, surf);
	}
/*...e*/
/*...e*/
/*...sPrimShapeQuart:2:*/
// Aim to support Torii etc..

class PrimShapeQuart : public PrimShapeSurf
	{
	double c[QCI::n_coeff];
	PrimShapeQuart(Surf surf); // used internally
public:
	PrimShapeQuart(
		const double c[],
		Surf surf
		);
	virtual ~PrimShapeQuart();
	virtual PrimShape *copy() const;
	virtual PrimShape *trans  (Xyz    t     ) const;
	virtual PrimShape *trans_x(double t     ) const;
	virtual PrimShape *trans_y(double t     ) const;
	virtual PrimShape *trans_z(double t     ) const;
	virtual PrimShape *rot_x  (double angle ) const;
	virtual PrimShape *rot_y  (double angle ) const;
	virtual PrimShape *rot_z  (double angle ) const;
	virtual PrimShape *scale  (Xyz    factor) const;
	virtual PrimShape *scale  (double factor) const;
	virtual PrimShape *scale_x(double factor) const;
	virtual PrimShape *scale_y(double factor) const;
	virtual PrimShape *scale_z(double factor) const;
	virtual PrimShape *resurf(Surf s) const;
	virtual Xyz normal(const Xyz & p) const;
	virtual Extent extent();
	virtual int isects_reqd() const;
	virtual double evaluate(const Xyz & v) const;
	virtual void intersect(const Xyz & p, const Xyz & q, IsectList *ils[]) const;
	};

/*...sPrimShapeQuart:2:*/
PrimShapeQuart::PrimShapeQuart(Surf surf) : PrimShapeSurf(surf) {}

PrimShapeQuart::PrimShapeQuart(
	const double c[],
	Surf surf
	)
	:
	PrimShapeSurf(surf)
	{
	for ( int i = 0; i < QCI::n_coeff; i++ )
		this->c[i] = c[i];
	}
/*...e*/
/*...s\126\PrimShapeQuart:2:*/
PrimShapeQuart::~PrimShapeQuart() {}
/*...e*/
/*...scopy:2:*/
PrimShape *PrimShapeQuart::copy() const
	{
	return new PrimShapeQuart(c, surf);
	}
/*...e*/
/*...strans\39\s:2:*/
PrimShape *PrimShapeQuart::trans(Xyz t) const
	{
	PrimShape *p1 =     trans_x(t.x);
	PrimShape *p2 = p1->trans_y(t.y);
	PrimShape *p3 = p2->trans_z(t.z);
	delete p1;
	delete p2;
	return p3;
	}

PrimShape *PrimShapeQuart::trans_x(double t) const
	{
	PrimShapeQuart *p = new PrimShapeQuart(surf);
/*...vsm\47\trans_x\46\C:10:*/
	p->c[QCI::C   ] = +t*t*t*t*c[QCI::XXXX] -t*t*t*c[QCI::XXX] +t*t*c[QCI::XX] -t*c[QCI::X] +c[QCI::C];
	p->c[QCI::X   ] = -4*t*t*t*c[QCI::XXXX] +3*t*t*c[QCI::XXX] -2*t*c[QCI::XX] +c[QCI::X];
	p->c[QCI::XX  ] = +6*t*t*c[QCI::XXXX] -3*t*c[QCI::XXX] +c[QCI::XX];
	p->c[QCI::XXX ] = -4*t*c[QCI::XXXX] +c[QCI::XXX];
	p->c[QCI::XXXX] = +c[QCI::XXXX];
	p->c[QCI::XXXY] = +c[QCI::XXXY];
	p->c[QCI::XXXZ] = +c[QCI::XXXZ];
	p->c[QCI::XXY ] = +c[QCI::XXY] -3*t*c[QCI::XXXY];
	p->c[QCI::XXYY] = +c[QCI::XXYY];
	p->c[QCI::XXYZ] = +c[QCI::XXYZ];
	p->c[QCI::XXZ ] = +c[QCI::XXZ] -3*t*c[QCI::XXXZ];
	p->c[QCI::XXZZ] = +c[QCI::XXZZ];
	p->c[QCI::XY  ] = +c[QCI::XY] -2*t*c[QCI::XXY] +3*t*t*c[QCI::XXXY];
	p->c[QCI::XYY ] = +c[QCI::XYY] -2*t*c[QCI::XXYY];
	p->c[QCI::XYYY] = +c[QCI::XYYY];
	p->c[QCI::XYYZ] = +c[QCI::XYYZ];
	p->c[QCI::XYZ ] = +c[QCI::XYZ] -2*t*c[QCI::XXYZ];
	p->c[QCI::XYZZ] = +c[QCI::XYZZ];
	p->c[QCI::XZ  ] = +c[QCI::XZ] -2*t*c[QCI::XXZ] +3*t*t*c[QCI::XXXZ];
	p->c[QCI::XZZ ] = +c[QCI::XZZ] -2*t*c[QCI::XXZZ];
	p->c[QCI::XZZZ] = +c[QCI::XZZZ];
	p->c[QCI::Y   ] = +c[QCI::Y] -t*c[QCI::XY] +t*t*c[QCI::XXY] -t*t*t*c[QCI::XXXY];
	p->c[QCI::YY  ] = +c[QCI::YY] -t*c[QCI::XYY] +t*t*c[QCI::XXYY];
	p->c[QCI::YYY ] = +c[QCI::YYY] -t*c[QCI::XYYY];
	p->c[QCI::YYYY] = +c[QCI::YYYY];
	p->c[QCI::YYYZ] = +c[QCI::YYYZ];
	p->c[QCI::YYZ ] = +c[QCI::YYZ] -t*c[QCI::XYYZ];
	p->c[QCI::YYZZ] = +c[QCI::YYZZ];
	p->c[QCI::YZ  ] = +c[QCI::YZ] -t*c[QCI::XYZ] +t*t*c[QCI::XXYZ];
	p->c[QCI::YZZ ] = +c[QCI::YZZ] -t*c[QCI::XYZZ];
	p->c[QCI::YZZZ] = +c[QCI::YZZZ];
	p->c[QCI::Z   ] = +c[QCI::Z] -t*c[QCI::XZ] +t*t*c[QCI::XXZ] -t*t*t*c[QCI::XXXZ];
	p->c[QCI::ZZ  ] = +c[QCI::ZZ] -t*c[QCI::XZZ] +t*t*c[QCI::XXZZ];
	p->c[QCI::ZZZ ] = +c[QCI::ZZZ] -t*c[QCI::XZZZ];
	p->c[QCI::ZZZZ] = +c[QCI::ZZZZ];
	return p;
	}

PrimShape *PrimShapeQuart::trans_y(double t) const
	{
	PrimShapeQuart *p = new PrimShapeQuart(surf);
/*...vsm\47\trans_y\46\C:10:*/
	p->c[QCI::C   ] = +t*t*t*t*c[QCI::YYYY] -t*t*t*c[QCI::YYY] +t*t*c[QCI::YY] -t*c[QCI::Y] +c[QCI::C];
	p->c[QCI::X   ] = -t*t*t*c[QCI::XYYY] +t*t*c[QCI::XYY] -t*c[QCI::XY] +c[QCI::X];
	p->c[QCI::XX  ] = +t*t*c[QCI::XXYY] -t*c[QCI::XXY] +c[QCI::XX];
	p->c[QCI::XXX ] = -t*c[QCI::XXXY] +c[QCI::XXX];
	p->c[QCI::XXXX] = +c[QCI::XXXX];
	p->c[QCI::XXXY] = +c[QCI::XXXY];
	p->c[QCI::XXXZ] = +c[QCI::XXXZ];
	p->c[QCI::XXY ] = -2*t*c[QCI::XXYY] +c[QCI::XXY];
	p->c[QCI::XXYY] = +c[QCI::XXYY];
	p->c[QCI::XXYZ] = +c[QCI::XXYZ];
	p->c[QCI::XXZ ] = +c[QCI::XXZ] -t*c[QCI::XXYZ];
	p->c[QCI::XXZZ] = +c[QCI::XXZZ];
	p->c[QCI::XY  ] = +3*t*t*c[QCI::XYYY] -2*t*c[QCI::XYY] +c[QCI::XY];
	p->c[QCI::XYY ] = -3*t*c[QCI::XYYY] +c[QCI::XYY];
	p->c[QCI::XYYY] = +c[QCI::XYYY];
	p->c[QCI::XYYZ] = +c[QCI::XYYZ];
	p->c[QCI::XYZ ] = +c[QCI::XYZ] -2*t*c[QCI::XYYZ];
	p->c[QCI::XYZZ] = +c[QCI::XYZZ];
	p->c[QCI::XZ  ] = +c[QCI::XZ] -t*c[QCI::XYZ] +t*t*c[QCI::XYYZ];
	p->c[QCI::XZZ ] = +c[QCI::XZZ] -t*c[QCI::XYZZ];
	p->c[QCI::XZZZ] = +c[QCI::XZZZ];
	p->c[QCI::Y   ] = -4*t*t*t*c[QCI::YYYY] +3*t*t*c[QCI::YYY] -2*t*c[QCI::YY] +c[QCI::Y];
	p->c[QCI::YY  ] = +6*t*t*c[QCI::YYYY] -3*t*c[QCI::YYY] +c[QCI::YY];
	p->c[QCI::YYY ] = -4*t*c[QCI::YYYY] +c[QCI::YYY];
	p->c[QCI::YYYY] = +c[QCI::YYYY];
	p->c[QCI::YYYZ] = +c[QCI::YYYZ];
	p->c[QCI::YYZ ] = +c[QCI::YYZ] -3*t*c[QCI::YYYZ];
	p->c[QCI::YYZZ] = +c[QCI::YYZZ];
	p->c[QCI::YZ  ] = +c[QCI::YZ] -2*t*c[QCI::YYZ] +3*t*t*c[QCI::YYYZ];
	p->c[QCI::YZZ ] = +c[QCI::YZZ] -2*t*c[QCI::YYZZ];
	p->c[QCI::YZZZ] = +c[QCI::YZZZ];
	p->c[QCI::Z   ] = +c[QCI::Z] -t*c[QCI::YZ] +t*t*c[QCI::YYZ] -t*t*t*c[QCI::YYYZ];
	p->c[QCI::ZZ  ] = +c[QCI::ZZ] -t*c[QCI::YZZ] +t*t*c[QCI::YYZZ];
	p->c[QCI::ZZZ ] = +c[QCI::ZZZ] -t*c[QCI::YZZZ];
	p->c[QCI::ZZZZ] = +c[QCI::ZZZZ];
	return p;
	}

PrimShape *PrimShapeQuart::trans_z(double t) const
	{
	PrimShapeQuart *p = new PrimShapeQuart(surf);
/*...vsm\47\trans_z\46\C:10:*/
	p->c[QCI::C   ] = +t*t*t*t*c[QCI::ZZZZ] -t*t*t*c[QCI::ZZZ] +t*t*c[QCI::ZZ] -t*c[QCI::Z] +c[QCI::C];
	p->c[QCI::X   ] = -t*t*t*c[QCI::XZZZ] +t*t*c[QCI::XZZ] -t*c[QCI::XZ] +c[QCI::X];
	p->c[QCI::XX  ] = +t*t*c[QCI::XXZZ] -t*c[QCI::XXZ] +c[QCI::XX];
	p->c[QCI::XXX ] = -t*c[QCI::XXXZ] +c[QCI::XXX];
	p->c[QCI::XXXX] = +c[QCI::XXXX];
	p->c[QCI::XXXY] = +c[QCI::XXXY];
	p->c[QCI::XXXZ] = +c[QCI::XXXZ];
	p->c[QCI::XXY ] = -t*c[QCI::XXYZ] +c[QCI::XXY];
	p->c[QCI::XXYY] = +c[QCI::XXYY];
	p->c[QCI::XXYZ] = +c[QCI::XXYZ];
	p->c[QCI::XXZ ] = -2*t*c[QCI::XXZZ] +c[QCI::XXZ];
	p->c[QCI::XXZZ] = +c[QCI::XXZZ];
	p->c[QCI::XY  ] = +t*t*c[QCI::XYZZ] -t*c[QCI::XYZ] +c[QCI::XY];
	p->c[QCI::XYY ] = -t*c[QCI::XYYZ] +c[QCI::XYY];
	p->c[QCI::XYYY] = +c[QCI::XYYY];
	p->c[QCI::XYYZ] = +c[QCI::XYYZ];
	p->c[QCI::XYZ ] = -2*t*c[QCI::XYZZ] +c[QCI::XYZ];
	p->c[QCI::XYZZ] = +c[QCI::XYZZ];
	p->c[QCI::XZ  ] = +3*t*t*c[QCI::XZZZ] -2*t*c[QCI::XZZ] +c[QCI::XZ];
	p->c[QCI::XZZ ] = -3*t*c[QCI::XZZZ] +c[QCI::XZZ];
	p->c[QCI::XZZZ] = +c[QCI::XZZZ];
	p->c[QCI::Y   ] = -t*t*t*c[QCI::YZZZ] +t*t*c[QCI::YZZ] -t*c[QCI::YZ] +c[QCI::Y];
	p->c[QCI::YY  ] = +t*t*c[QCI::YYZZ] -t*c[QCI::YYZ] +c[QCI::YY];
	p->c[QCI::YYY ] = -t*c[QCI::YYYZ] +c[QCI::YYY];
	p->c[QCI::YYYY] = +c[QCI::YYYY];
	p->c[QCI::YYYZ] = +c[QCI::YYYZ];
	p->c[QCI::YYZ ] = -2*t*c[QCI::YYZZ] +c[QCI::YYZ];
	p->c[QCI::YYZZ] = +c[QCI::YYZZ];
	p->c[QCI::YZ  ] = +3*t*t*c[QCI::YZZZ] -2*t*c[QCI::YZZ] +c[QCI::YZ];
	p->c[QCI::YZZ ] = -3*t*c[QCI::YZZZ] +c[QCI::YZZ];
	p->c[QCI::YZZZ] = +c[QCI::YZZZ];
	p->c[QCI::Z   ] = -4*t*t*t*c[QCI::ZZZZ] +3*t*t*c[QCI::ZZZ] -2*t*c[QCI::ZZ] +c[QCI::Z];
	p->c[QCI::ZZ  ] = +6*t*t*c[QCI::ZZZZ] -3*t*c[QCI::ZZZ] +c[QCI::ZZ];
	p->c[QCI::ZZZ ] = -4*t*c[QCI::ZZZZ] +c[QCI::ZZZ];
	p->c[QCI::ZZZZ] = +c[QCI::ZZZZ];
	return p;
	}
/*...e*/
/*...srot\39\s:2:*/
PrimShape *PrimShapeQuart::rot_x(double angle) const
	{
	double ca=cos(angle), sa=sin(angle);
	PrimShapeQuart *p = new PrimShapeQuart(surf);
/*...vsm\47\rot_x\46\C:10:*/
	p->c[QCI::C   ] = +c[QCI::C];
	p->c[QCI::X   ] = +c[QCI::X];
	p->c[QCI::XX  ] = +c[QCI::XX];
	p->c[QCI::XXX ] = +c[QCI::XXX];
	p->c[QCI::XXXX] = +c[QCI::XXXX];
	p->c[QCI::XXXY] = -sa*c[QCI::XXXZ] +ca*c[QCI::XXXY];
	p->c[QCI::XXXZ] = +ca*c[QCI::XXXZ] +sa*c[QCI::XXXY];
	p->c[QCI::XXY ] = -sa*c[QCI::XXZ] +ca*c[QCI::XXY];
	p->c[QCI::XXYY] = +sa*sa*c[QCI::XXZZ] -ca*sa*c[QCI::XXYZ] +ca*ca*c[QCI::XXYY];
	p->c[QCI::XXYZ] = -2*ca*sa*c[QCI::XXZZ] -sa*sa*c[QCI::XXYZ] +ca*ca*c[QCI::XXYZ] +2*ca*sa*c[QCI::XXYY];
	p->c[QCI::XXZ ] = +ca*c[QCI::XXZ] +sa*c[QCI::XXY];
	p->c[QCI::XXZZ] = +ca*ca*c[QCI::XXZZ] +ca*sa*c[QCI::XXYZ] +sa*sa*c[QCI::XXYY];
	p->c[QCI::XY  ] = -sa*c[QCI::XZ] +ca*c[QCI::XY];
	p->c[QCI::XYY ] = +sa*sa*c[QCI::XZZ] -ca*sa*c[QCI::XYZ] +ca*ca*c[QCI::XYY];
	p->c[QCI::XYYY] = -sa*sa*sa*c[QCI::XZZZ] +ca*sa*sa*c[QCI::XYZZ] -ca*ca*sa*c[QCI::XYYZ] +ca*ca*ca*c[QCI::XYYY];
	p->c[QCI::XYYZ] = +3*ca*sa*sa*c[QCI::XZZZ] +sa*sa*sa*c[QCI::XYZZ] -2*ca*ca*sa*c[QCI::XYZZ] -2*ca*sa*sa*c[QCI::XYYZ] +ca*ca*ca*c[QCI::XYYZ] +3*ca*ca*sa*c[QCI::XYYY];
	p->c[QCI::XYZ ] = -2*ca*sa*c[QCI::XZZ] -sa*sa*c[QCI::XYZ] +ca*ca*c[QCI::XYZ] +2*ca*sa*c[QCI::XYY];
	p->c[QCI::XYZZ] = -3*ca*ca*sa*c[QCI::XZZZ] -2*ca*sa*sa*c[QCI::XYZZ] +ca*ca*ca*c[QCI::XYZZ] -sa*sa*sa*c[QCI::XYYZ] +2*ca*ca*sa*c[QCI::XYYZ] +3*ca*sa*sa*c[QCI::XYYY];
	p->c[QCI::XZ  ] = +ca*c[QCI::XZ] +sa*c[QCI::XY];
	p->c[QCI::XZZ ] = +ca*ca*c[QCI::XZZ] +ca*sa*c[QCI::XYZ] +sa*sa*c[QCI::XYY];
	p->c[QCI::XZZZ] = +ca*ca*ca*c[QCI::XZZZ] +ca*ca*sa*c[QCI::XYZZ] +ca*sa*sa*c[QCI::XYYZ] +sa*sa*sa*c[QCI::XYYY];
	p->c[QCI::Y   ] = -sa*c[QCI::Z] +ca*c[QCI::Y];
	p->c[QCI::YY  ] = +sa*sa*c[QCI::ZZ] -ca*sa*c[QCI::YZ] +ca*ca*c[QCI::YY];
	p->c[QCI::YYY ] = -sa*sa*sa*c[QCI::ZZZ] +ca*sa*sa*c[QCI::YZZ] -ca*ca*sa*c[QCI::YYZ] +ca*ca*ca*c[QCI::YYY];
	p->c[QCI::YYYY] = +sa*sa*sa*sa*c[QCI::ZZZZ] -ca*sa*sa*sa*c[QCI::YZZZ] +ca*ca*sa*sa*c[QCI::YYZZ] -ca*ca*ca*sa*c[QCI::YYYZ] +ca*ca*ca*ca*c[QCI::YYYY];
	p->c[QCI::YYYZ] = -4*ca*sa*sa*sa*c[QCI::ZZZZ] -sa*sa*sa*sa*c[QCI::YZZZ] +3*ca*ca*sa*sa*c[QCI::YZZZ] +2*ca*sa*sa*sa*c[QCI::YYZZ] -2*ca*ca*ca*sa*c[QCI::YYZZ] -3*ca*ca*sa*sa*c[QCI::YYYZ] +ca*ca*ca*ca*c[QCI::YYYZ] +4*ca*ca*ca*sa*c[QCI::YYYY];
	p->c[QCI::YYZ ] = +3*ca*sa*sa*c[QCI::ZZZ] +sa*sa*sa*c[QCI::YZZ] -2*ca*ca*sa*c[QCI::YZZ] -2*ca*sa*sa*c[QCI::YYZ] +ca*ca*ca*c[QCI::YYZ] +3*ca*ca*sa*c[QCI::YYY];
	p->c[QCI::YYZZ] = +6*ca*ca*sa*sa*c[QCI::ZZZZ] +3*ca*sa*sa*sa*c[QCI::YZZZ] -3*ca*ca*ca*sa*c[QCI::YZZZ] +sa*sa*sa*sa*c[QCI::YYZZ] -4*ca*ca*sa*sa*c[QCI::YYZZ] +ca*ca*ca*ca*c[QCI::YYZZ] -3*ca*sa*sa*sa*c[QCI::YYYZ] +3*ca*ca*ca*sa*c[QCI::YYYZ] +6*ca*ca*sa*sa*c[QCI::YYYY];
	p->c[QCI::YZ  ] = -2*ca*sa*c[QCI::ZZ] -sa*sa*c[QCI::YZ] +ca*ca*c[QCI::YZ] +2*ca*sa*c[QCI::YY];
	p->c[QCI::YZZ ] = -3*ca*ca*sa*c[QCI::ZZZ] -2*ca*sa*sa*c[QCI::YZZ] +ca*ca*ca*c[QCI::YZZ] -sa*sa*sa*c[QCI::YYZ] +2*ca*ca*sa*c[QCI::YYZ] +3*ca*sa*sa*c[QCI::YYY];
	p->c[QCI::YZZZ] = -4*ca*ca*ca*sa*c[QCI::ZZZZ] -3*ca*ca*sa*sa*c[QCI::YZZZ] +ca*ca*ca*ca*c[QCI::YZZZ] -2*ca*sa*sa*sa*c[QCI::YYZZ] +2*ca*ca*ca*sa*c[QCI::YYZZ] -sa*sa*sa*sa*c[QCI::YYYZ] +3*ca*ca*sa*sa*c[QCI::YYYZ] +4*ca*sa*sa*sa*c[QCI::YYYY];
	p->c[QCI::Z   ] = +ca*c[QCI::Z] +sa*c[QCI::Y];
	p->c[QCI::ZZ  ] = +ca*ca*c[QCI::ZZ] +ca*sa*c[QCI::YZ] +sa*sa*c[QCI::YY];
	p->c[QCI::ZZZ ] = +ca*ca*ca*c[QCI::ZZZ] +ca*ca*sa*c[QCI::YZZ] +ca*sa*sa*c[QCI::YYZ] +sa*sa*sa*c[QCI::YYY];
	p->c[QCI::ZZZZ] = +ca*ca*ca*ca*c[QCI::ZZZZ] +ca*ca*ca*sa*c[QCI::YZZZ] +ca*ca*sa*sa*c[QCI::YYZZ] +ca*sa*sa*sa*c[QCI::YYYZ] +sa*sa*sa*sa*c[QCI::YYYY];
	return p;
	}

PrimShape *PrimShapeQuart::rot_y(double angle) const
	{
	double ca=cos(angle), sa=sin(angle);
	PrimShapeQuart *p = new PrimShapeQuart(surf);
/*...vsm\47\rot_y\46\C:10:*/
	p->c[QCI::C   ] = +c[QCI::C];
	p->c[QCI::X   ] = +sa*c[QCI::Z] +ca*c[QCI::X];
	p->c[QCI::XX  ] = +sa*sa*c[QCI::ZZ] +ca*sa*c[QCI::XZ] +ca*ca*c[QCI::XX];
	p->c[QCI::XXX ] = +sa*sa*sa*c[QCI::ZZZ] +ca*sa*sa*c[QCI::XZZ] +ca*ca*sa*c[QCI::XXZ] +ca*ca*ca*c[QCI::XXX];
	p->c[QCI::XXXX] = +sa*sa*sa*sa*c[QCI::ZZZZ] +ca*sa*sa*sa*c[QCI::XZZZ] +ca*ca*sa*sa*c[QCI::XXZZ] +ca*ca*ca*sa*c[QCI::XXXZ] +ca*ca*ca*ca*c[QCI::XXXX];
	p->c[QCI::XXXY] = +sa*sa*sa*c[QCI::YZZZ] +ca*sa*sa*c[QCI::XYZZ] +ca*ca*sa*c[QCI::XXYZ] +ca*ca*ca*c[QCI::XXXY];
	p->c[QCI::XXXZ] = +4*ca*sa*sa*sa*c[QCI::ZZZZ] -sa*sa*sa*sa*c[QCI::XZZZ] +3*ca*ca*sa*sa*c[QCI::XZZZ] -2*ca*sa*sa*sa*c[QCI::XXZZ] +2*ca*ca*ca*sa*c[QCI::XXZZ] -3*ca*ca*sa*sa*c[QCI::XXXZ] +ca*ca*ca*ca*c[QCI::XXXZ] -4*ca*ca*ca*sa*c[QCI::XXXX];
	p->c[QCI::XXY ] = +sa*sa*c[QCI::YZZ] +ca*sa*c[QCI::XYZ] +ca*ca*c[QCI::XXY];
	p->c[QCI::XXYY] = +sa*sa*c[QCI::YYZZ] +ca*sa*c[QCI::XYYZ] +ca*ca*c[QCI::XXYY];
	p->c[QCI::XXYZ] = +3*ca*sa*sa*c[QCI::YZZZ] -sa*sa*sa*c[QCI::XYZZ] +2*ca*ca*sa*c[QCI::XYZZ] -2*ca*sa*sa*c[QCI::XXYZ] +ca*ca*ca*c[QCI::XXYZ] -3*ca*ca*sa*c[QCI::XXXY];
	p->c[QCI::XXZ ] = +3*ca*sa*sa*c[QCI::ZZZ] -sa*sa*sa*c[QCI::XZZ] +2*ca*ca*sa*c[QCI::XZZ] -2*ca*sa*sa*c[QCI::XXZ] +ca*ca*ca*c[QCI::XXZ] -3*ca*ca*sa*c[QCI::XXX];
	p->c[QCI::XXZZ] = +6*ca*ca*sa*sa*c[QCI::ZZZZ] -3*ca*sa*sa*sa*c[QCI::XZZZ] +3*ca*ca*ca*sa*c[QCI::XZZZ] +sa*sa*sa*sa*c[QCI::XXZZ] -4*ca*ca*sa*sa*c[QCI::XXZZ] +ca*ca*ca*ca*c[QCI::XXZZ] +3*ca*sa*sa*sa*c[QCI::XXXZ] -3*ca*ca*ca*sa*c[QCI::XXXZ] +6*ca*ca*sa*sa*c[QCI::XXXX];
	p->c[QCI::XY  ] = +sa*c[QCI::YZ] +ca*c[QCI::XY];
	p->c[QCI::XYY ] = +sa*c[QCI::YYZ] +ca*c[QCI::XYY];
	p->c[QCI::XYYY] = +sa*c[QCI::YYYZ] +ca*c[QCI::XYYY];
	p->c[QCI::XYYZ] = +2*ca*sa*c[QCI::YYZZ] -sa*sa*c[QCI::XYYZ] +ca*ca*c[QCI::XYYZ] -2*ca*sa*c[QCI::XXYY];
	p->c[QCI::XYZ ] = +2*ca*sa*c[QCI::YZZ] -sa*sa*c[QCI::XYZ] +ca*ca*c[QCI::XYZ] -2*ca*sa*c[QCI::XXY];
	p->c[QCI::XYZZ] = +3*ca*ca*sa*c[QCI::YZZZ] -2*ca*sa*sa*c[QCI::XYZZ] +ca*ca*ca*c[QCI::XYZZ] +sa*sa*sa*c[QCI::XXYZ] -2*ca*ca*sa*c[QCI::XXYZ] +3*ca*sa*sa*c[QCI::XXXY];
	p->c[QCI::XZ  ] = +2*ca*sa*c[QCI::ZZ] -sa*sa*c[QCI::XZ] +ca*ca*c[QCI::XZ] -2*ca*sa*c[QCI::XX];
	p->c[QCI::XZZ ] = +3*ca*ca*sa*c[QCI::ZZZ] -2*ca*sa*sa*c[QCI::XZZ] +ca*ca*ca*c[QCI::XZZ] +sa*sa*sa*c[QCI::XXZ] -2*ca*ca*sa*c[QCI::XXZ] +3*ca*sa*sa*c[QCI::XXX];
	p->c[QCI::XZZZ] = +4*ca*ca*ca*sa*c[QCI::ZZZZ] -3*ca*ca*sa*sa*c[QCI::XZZZ] +ca*ca*ca*ca*c[QCI::XZZZ] +2*ca*sa*sa*sa*c[QCI::XXZZ] -2*ca*ca*ca*sa*c[QCI::XXZZ] -sa*sa*sa*sa*c[QCI::XXXZ] +3*ca*ca*sa*sa*c[QCI::XXXZ] -4*ca*sa*sa*sa*c[QCI::XXXX];
	p->c[QCI::Y   ] = +c[QCI::Y];
	p->c[QCI::YY  ] = +c[QCI::YY];
	p->c[QCI::YYY ] = +c[QCI::YYY];
	p->c[QCI::YYYY] = +c[QCI::YYYY];
	p->c[QCI::YYYZ] = +ca*c[QCI::YYYZ] -sa*c[QCI::XYYY];
	p->c[QCI::YYZ ] = +ca*c[QCI::YYZ] -sa*c[QCI::XYY];
	p->c[QCI::YYZZ] = +ca*ca*c[QCI::YYZZ] -ca*sa*c[QCI::XYYZ] +sa*sa*c[QCI::XXYY];
	p->c[QCI::YZ  ] = +ca*c[QCI::YZ] -sa*c[QCI::XY];
	p->c[QCI::YZZ ] = +ca*ca*c[QCI::YZZ] -ca*sa*c[QCI::XYZ] +sa*sa*c[QCI::XXY];
	p->c[QCI::YZZZ] = +ca*ca*ca*c[QCI::YZZZ] -ca*ca*sa*c[QCI::XYZZ] +ca*sa*sa*c[QCI::XXYZ] -sa*sa*sa*c[QCI::XXXY];
	p->c[QCI::Z   ] = +ca*c[QCI::Z] -sa*c[QCI::X];
	p->c[QCI::ZZ  ] = +ca*ca*c[QCI::ZZ] -ca*sa*c[QCI::XZ] +sa*sa*c[QCI::XX];
	p->c[QCI::ZZZ ] = +ca*ca*ca*c[QCI::ZZZ] -ca*ca*sa*c[QCI::XZZ] +ca*sa*sa*c[QCI::XXZ] -sa*sa*sa*c[QCI::XXX];
	p->c[QCI::ZZZZ] = +ca*ca*ca*ca*c[QCI::ZZZZ] -ca*ca*ca*sa*c[QCI::XZZZ] +ca*ca*sa*sa*c[QCI::XXZZ] -ca*sa*sa*sa*c[QCI::XXXZ] +sa*sa*sa*sa*c[QCI::XXXX];
	return p;
	}

PrimShape *PrimShapeQuart::rot_z(double angle) const
	{
	double ca=cos(angle), sa=sin(angle);
	PrimShapeQuart *p = new PrimShapeQuart(surf);
/*...vsm\47\rot_z\46\C:10:*/
	p->c[QCI::C   ] = +c[QCI::C];
	p->c[QCI::X   ] = -sa*c[QCI::Y] +ca*c[QCI::X];
	p->c[QCI::XX  ] = +sa*sa*c[QCI::YY] -ca*sa*c[QCI::XY] +ca*ca*c[QCI::XX];
	p->c[QCI::XXX ] = -sa*sa*sa*c[QCI::YYY] +ca*sa*sa*c[QCI::XYY] -ca*ca*sa*c[QCI::XXY] +ca*ca*ca*c[QCI::XXX];
	p->c[QCI::XXXX] = +sa*sa*sa*sa*c[QCI::YYYY] -ca*sa*sa*sa*c[QCI::XYYY] +ca*ca*sa*sa*c[QCI::XXYY] -ca*ca*ca*sa*c[QCI::XXXY] +ca*ca*ca*ca*c[QCI::XXXX];
	p->c[QCI::XXXY] = -4*ca*sa*sa*sa*c[QCI::YYYY] -sa*sa*sa*sa*c[QCI::XYYY] +3*ca*ca*sa*sa*c[QCI::XYYY] +2*ca*sa*sa*sa*c[QCI::XXYY] -2*ca*ca*ca*sa*c[QCI::XXYY] -3*ca*ca*sa*sa*c[QCI::XXXY] +ca*ca*ca*ca*c[QCI::XXXY] +4*ca*ca*ca*sa*c[QCI::XXXX];
	p->c[QCI::XXXZ] = -sa*sa*sa*c[QCI::YYYZ] +ca*sa*sa*c[QCI::XYYZ] -ca*ca*sa*c[QCI::XXYZ] +ca*ca*ca*c[QCI::XXXZ];
	p->c[QCI::XXY ] = +3*ca*sa*sa*c[QCI::YYY] +sa*sa*sa*c[QCI::XYY] -2*ca*ca*sa*c[QCI::XYY] -2*ca*sa*sa*c[QCI::XXY] +ca*ca*ca*c[QCI::XXY] +3*ca*ca*sa*c[QCI::XXX];
	p->c[QCI::XXYY] = +6*ca*ca*sa*sa*c[QCI::YYYY] +3*ca*sa*sa*sa*c[QCI::XYYY] -3*ca*ca*ca*sa*c[QCI::XYYY] +sa*sa*sa*sa*c[QCI::XXYY] -4*ca*ca*sa*sa*c[QCI::XXYY] +ca*ca*ca*ca*c[QCI::XXYY] -3*ca*sa*sa*sa*c[QCI::XXXY] +3*ca*ca*ca*sa*c[QCI::XXXY] +6*ca*ca*sa*sa*c[QCI::XXXX];
	p->c[QCI::XXYZ] = +3*ca*sa*sa*c[QCI::YYYZ] +sa*sa*sa*c[QCI::XYYZ] -2*ca*ca*sa*c[QCI::XYYZ] -2*ca*sa*sa*c[QCI::XXYZ] +ca*ca*ca*c[QCI::XXYZ] +3*ca*ca*sa*c[QCI::XXXZ];
	p->c[QCI::XXZ ] = +sa*sa*c[QCI::YYZ] -ca*sa*c[QCI::XYZ] +ca*ca*c[QCI::XXZ];
	p->c[QCI::XXZZ] = +sa*sa*c[QCI::YYZZ] -ca*sa*c[QCI::XYZZ] +ca*ca*c[QCI::XXZZ];
	p->c[QCI::XY  ] = -2*ca*sa*c[QCI::YY] -sa*sa*c[QCI::XY] +ca*ca*c[QCI::XY] +2*ca*sa*c[QCI::XX];
	p->c[QCI::XYY ] = -3*ca*ca*sa*c[QCI::YYY] -2*ca*sa*sa*c[QCI::XYY] +ca*ca*ca*c[QCI::XYY] -sa*sa*sa*c[QCI::XXY] +2*ca*ca*sa*c[QCI::XXY] +3*ca*sa*sa*c[QCI::XXX];
	p->c[QCI::XYYY] = -4*ca*ca*ca*sa*c[QCI::YYYY] -3*ca*ca*sa*sa*c[QCI::XYYY] +ca*ca*ca*ca*c[QCI::XYYY] -2*ca*sa*sa*sa*c[QCI::XXYY] +2*ca*ca*ca*sa*c[QCI::XXYY] -sa*sa*sa*sa*c[QCI::XXXY] +3*ca*ca*sa*sa*c[QCI::XXXY] +4*ca*sa*sa*sa*c[QCI::XXXX];
	p->c[QCI::XYYZ] = -3*ca*ca*sa*c[QCI::YYYZ] -2*ca*sa*sa*c[QCI::XYYZ] +ca*ca*ca*c[QCI::XYYZ] -sa*sa*sa*c[QCI::XXYZ] +2*ca*ca*sa*c[QCI::XXYZ] +3*ca*sa*sa*c[QCI::XXXZ];
	p->c[QCI::XYZ ] = -2*ca*sa*c[QCI::YYZ] -sa*sa*c[QCI::XYZ] +ca*ca*c[QCI::XYZ] +2*ca*sa*c[QCI::XXZ];
	p->c[QCI::XYZZ] = -2*ca*sa*c[QCI::YYZZ] -sa*sa*c[QCI::XYZZ] +ca*ca*c[QCI::XYZZ] +2*ca*sa*c[QCI::XXZZ];
	p->c[QCI::XZ  ] = -sa*c[QCI::YZ] +ca*c[QCI::XZ];
	p->c[QCI::XZZ ] = -sa*c[QCI::YZZ] +ca*c[QCI::XZZ];
	p->c[QCI::XZZZ] = -sa*c[QCI::YZZZ] +ca*c[QCI::XZZZ];
	p->c[QCI::Y   ] = +ca*c[QCI::Y] +sa*c[QCI::X];
	p->c[QCI::YY  ] = +ca*ca*c[QCI::YY] +ca*sa*c[QCI::XY] +sa*sa*c[QCI::XX];
	p->c[QCI::YYY ] = +ca*ca*ca*c[QCI::YYY] +ca*ca*sa*c[QCI::XYY] +ca*sa*sa*c[QCI::XXY] +sa*sa*sa*c[QCI::XXX];
	p->c[QCI::YYYY] = +ca*ca*ca*ca*c[QCI::YYYY] +ca*ca*ca*sa*c[QCI::XYYY] +ca*ca*sa*sa*c[QCI::XXYY] +ca*sa*sa*sa*c[QCI::XXXY] +sa*sa*sa*sa*c[QCI::XXXX];
	p->c[QCI::YYYZ] = +ca*ca*ca*c[QCI::YYYZ] +ca*ca*sa*c[QCI::XYYZ] +ca*sa*sa*c[QCI::XXYZ] +sa*sa*sa*c[QCI::XXXZ];
	p->c[QCI::YYZ ] = +ca*ca*c[QCI::YYZ] +ca*sa*c[QCI::XYZ] +sa*sa*c[QCI::XXZ];
	p->c[QCI::YYZZ] = +ca*ca*c[QCI::YYZZ] +ca*sa*c[QCI::XYZZ] +sa*sa*c[QCI::XXZZ];
	p->c[QCI::YZ  ] = +ca*c[QCI::YZ] +sa*c[QCI::XZ];
	p->c[QCI::YZZ ] = +ca*c[QCI::YZZ] +sa*c[QCI::XZZ];
	p->c[QCI::YZZZ] = +ca*c[QCI::YZZZ] +sa*c[QCI::XZZZ];
	p->c[QCI::Z   ] = +c[QCI::Z];
	p->c[QCI::ZZ  ] = +c[QCI::ZZ];
	p->c[QCI::ZZZ ] = +c[QCI::ZZZ];
	p->c[QCI::ZZZZ] = +c[QCI::ZZZZ];
	return p;
	}
/*...e*/
/*...sscale\39\s:2:*/
PrimShape *PrimShapeQuart::scale(Xyz factor) const
	{
	PrimShape *p1 =     scale_x(factor.x);
	PrimShape *p2 = p1->scale_y(factor.y);
	PrimShape *p3 = p2->scale_z(factor.z);
	delete p1;
	delete p2;
	return p3;
	}

PrimShape *PrimShapeQuart::scale  (double factor) const
	{ return scale(Xyz(factor,factor,factor)); }

PrimShape *PrimShapeQuart::scale_x(double factor) const
	{
	PrimShapeQuart *p = new PrimShapeQuart(c, surf.scale_x(factor));
	double factor2 = factor*factor, factor3 = factor2*factor, factor4 = factor3*factor;
	double *c = p->c;
	c[QCI::XXXX] /= factor4;
	c[QCI::XXXY] /= factor3;
	c[QCI::XXXZ] /= factor3;
	c[QCI::XXX ] /= factor3;
	c[QCI::XXYY] /= factor2;
	c[QCI::XXYZ] /= factor2;
	c[QCI::XXY ] /= factor2;
	c[QCI::XXZZ] /= factor2;
	c[QCI::XXZ ] /= factor2;
	c[QCI::XX  ] /= factor2;
	c[QCI::XYYY] /= factor;
	c[QCI::XYYZ] /= factor;
	c[QCI::XYY ] /= factor;
	c[QCI::XYZZ] /= factor;
	c[QCI::XYZ ] /= factor;
	c[QCI::XY  ] /= factor;
	c[QCI::XZZZ] /= factor;
	c[QCI::XZZ ] /= factor;
	c[QCI::XZ  ] /= factor;
	c[QCI::X   ] /= factor;
	return p;
	}

PrimShape *PrimShapeQuart::scale_y(double factor) const
	{
	PrimShapeQuart *p = new PrimShapeQuart(c, surf.scale_y(factor));
	double factor2 = factor*factor, factor3 = factor2*factor, factor4 = factor3*factor;
	double *c = p->c;
	c[QCI::XXXY] /= factor;
	c[QCI::XXYY] /= factor2;
	c[QCI::XXYZ] /= factor;
	c[QCI::XXY ] /= factor;
	c[QCI::XYYY] /= factor3;
	c[QCI::XYYZ] /= factor2;
	c[QCI::XYY ] /= factor2;
	c[QCI::XYZZ] /= factor;
	c[QCI::XYZ ] /= factor;
	c[QCI::XY  ] /= factor;
	c[QCI::YYYY] /= factor4;
	c[QCI::YYYZ] /= factor3;
	c[QCI::YYY ] /= factor3;
	c[QCI::YYZZ] /= factor2;
	c[QCI::YYZ ] /= factor2;
	c[QCI::YY  ] /= factor2;
	c[QCI::YZZZ] /= factor;
	c[QCI::YZZ ] /= factor;
	c[QCI::YZ  ] /= factor;
	c[QCI::Y   ] /= factor;
	return p;
	}

PrimShape *PrimShapeQuart::scale_z(double factor) const
	{
	PrimShapeQuart *p = new PrimShapeQuart(c, surf.scale_z(factor));
	double factor2 = factor*factor, factor3 = factor2*factor, factor4 = factor3*factor;
	double *c = p->c;
	c[QCI::XXXZ] /= factor;
	c[QCI::XXYZ] /= factor;
	c[QCI::XXZZ] /= factor2;
	c[QCI::XXZ ] /= factor;
	c[QCI::XYYZ] /= factor;
	c[QCI::XYZZ] /= factor2;
	c[QCI::XYZ ] /= factor;
	c[QCI::XZZZ] /= factor3;
	c[QCI::XZZ ] /= factor2;
	c[QCI::XZ  ] /= factor;
	c[QCI::YYYZ] /= factor;
	c[QCI::YYZZ] /= factor2;
	c[QCI::YYZ ] /= factor;
	c[QCI::YZZZ] /= factor3;
	c[QCI::YZZ ] /= factor2;
	c[QCI::YZ  ] /= factor;
	c[QCI::ZZZZ] /= factor4;
	c[QCI::ZZZ ] /= factor3;
	c[QCI::ZZ  ] /= factor2;
	c[QCI::Z   ] /= factor;
	return p;
	}
/*...e*/
/*...sresurf:2:*/
PrimShape *PrimShapeQuart::resurf(Surf surf) const
	{
	return new PrimShapeQuart(c, surf);
	}
/*...e*/
/*...snormal:2:*/
// Use partial derivatives to find normal at a given point.

Xyz PrimShapeQuart::normal(const Xyz & p) const
	{
	double x = p.x, x2 = x*x, x3 = x2*x; 
	double y = p.y, y2 = y*y, y3 = y2*y; 
	double z = p.z, z2 = z*z, z3 = z2*z; 
	return Xyz(
		  c[QCI::XXXX] * 4.0*x3
		+ c[QCI::XXXY] * 3.0*x2*y
		+ c[QCI::XXXZ] * 3.0*x2*z
		+ c[QCI::XXX ] * 3.0*x2 
		+ c[QCI::XXYY] * 2.0*x*y2
		+ c[QCI::XXYZ] * 2.0*x*y*z
		+ c[QCI::XXY ] * 2.0*x*y
		+ c[QCI::XXZZ] * 2.0*x*z2
		+ c[QCI::XXZ ] * 2.0*x*z
		+ c[QCI::XX  ] * 2.0*x
		+ c[QCI::XYYY] * y3
		+ c[QCI::XYYZ] * y2*z
		+ c[QCI::XYY ] * y2
		+ c[QCI::XYZZ] * y*z2
		+ c[QCI::XYZ ] * y*z
		+ c[QCI::XY  ] * y
		+ c[QCI::XZZZ] * z3
		+ c[QCI::XZZ ] * z2
		+ c[QCI::XZ  ] * z
		+ c[QCI::X   ]
		,
		  c[QCI::XXXY] * x3
		+ c[QCI::XXYY] * 2.0*x2*y
		+ c[QCI::XXYZ] * x2*z
		+ c[QCI::XXY ] * x2
		+ c[QCI::XYYY] * 3.0*x*y2
		+ c[QCI::XYYZ] * 2.0*x*y*z
		+ c[QCI::XYY ] * 2.0*x*y
		+ c[QCI::XYZZ] * x*z2
		+ c[QCI::XYZ ] * x*z
		+ c[QCI::XY  ] * x
		+ c[QCI::YYYY] * 4.0*y3
		+ c[QCI::YYYZ] * 3.0*y2*z
		+ c[QCI::YYY ] * 3.0*y2
		+ c[QCI::YYZZ] * 2.0*y*z2
		+ c[QCI::YYZ ] * 2.0*y*z
		+ c[QCI::YY  ] * 2.0*y
		+ c[QCI::YZZZ] * z3
		+ c[QCI::YZZ ] * z2
		+ c[QCI::YZ  ] * z
		+ c[QCI::Y   ]
		,
		  c[QCI::XXXZ] * x3
		+ c[QCI::XXYZ] * x2*y
		+ c[QCI::XXZZ] * 2.0*x2*z
		+ c[QCI::XXZ ] * x2
		+ c[QCI::XYYZ] * x*y2
		+ c[QCI::XYZZ] * 2.0*x*y*z
		+ c[QCI::XYZ ] * x*y
		+ c[QCI::XZZZ] * 3.0*x*z2
		+ c[QCI::XZZ ] * 2.0*x*z
		+ c[QCI::XZ  ] * x
		+ c[QCI::YYYZ] * y3
		+ c[QCI::YYZZ] * 2.0*y2*z
		+ c[QCI::YYZ ] * y2
		+ c[QCI::YZZZ] * 3.0*y*z2
		+ c[QCI::YZZ ] * 2.0*y*z
		+ c[QCI::YZ  ] * y
		+ c[QCI::ZZZZ] * 4.0*z3
		+ c[QCI::ZZZ ] * 3.0*z2
		+ c[QCI::ZZ  ] * 2.0*z
		+ c[QCI::Z   ]
		).unit();
	}
/*...e*/
/*...sextent:2:*/
// Too difficult to work this one out (well, at least for now)

Extent PrimShapeQuart::extent()
	{
	return Extent(Xyz(-LARGEST,-LARGEST,-LARGEST),
		      Xyz( LARGEST, LARGEST, LARGEST));
	}
/*...e*/
/*...sisects_reqd:2:*/
int PrimShapeQuart::isects_reqd() const { return 6; }
/*...e*/
/*...sevaluate:2:*/
double PrimShapeQuart::evaluate(const Xyz & v) const
	{
	double x = v.x, x2 = x*x, x3 = x2*x, x4 = x3*x;
	double y = v.y, y2 = y*y, y3 = y2*y, y4 = y3*y;
	double z = v.z, z2 = z*z, z3 = z2*z, z4 = z3*z;
	return
		  c[QCI::XXXX] * x4 
		+ c[QCI::XXXY] * x3*y
		+ c[QCI::XXXZ] * x3*z
		+ c[QCI::XXX ] * x3
		+ c[QCI::XXYY] * x2*y2
		+ c[QCI::XXYZ] * x2*y*z
		+ c[QCI::XXY ] * x2*y
		+ c[QCI::XXZZ] * x2*z2
		+ c[QCI::XXZ ] * x2*z
		+ c[QCI::XX  ] * x2
		+ c[QCI::XYYY] * x*y3
		+ c[QCI::XYYZ] * x*y2*z
		+ c[QCI::XYY ] * x*y2
		+ c[QCI::XYZZ] * x*y*z2
		+ c[QCI::XYZ ] * x*y*z
		+ c[QCI::XY  ] * x*y
		+ c[QCI::XZZZ] * x*z3
		+ c[QCI::XZZ ] * x*z2
		+ c[QCI::XZ  ] * x*z
		+ c[QCI::X   ] * x
		+ c[QCI::YYYY] * y4
		+ c[QCI::YYYZ] * y3*z
		+ c[QCI::YYY ] * y3
		+ c[QCI::YYZZ] * y2*z2
		+ c[QCI::YYZ ] * y2*z
		+ c[QCI::YY  ] * y2
		+ c[QCI::YZZZ] * y*z3
		+ c[QCI::YZZ ] * y*z2
		+ c[QCI::YZ  ] * y*z
		+ c[QCI::Y   ] * y
		+ c[QCI::ZZZZ] * z4
		+ c[QCI::ZZZ ] * z3
		+ c[QCI::ZZ  ] * z2
		+ c[QCI::Z   ] * z
		+ c[QCI::C   ];
	}
/*...e*/
/*...sintersect:2:*/
void PrimShapeQuart::intersect(const Xyz & p, const Xyz & q, IsectList *ils[]) const
	{
/*...vsm\47\int\46\C:10:*/
	double k = 
		+p.z*p.z*p.z*p.z*c[QCI::ZZZZ] 
		+p.z*p.z*p.z*c[QCI::ZZZ] 
		+p.z*p.z*c[QCI::ZZ] 
		+p.z*c[QCI::Z] 
		+p.y*p.z*p.z*p.z*c[QCI::YZZZ] 
		+p.y*p.z*p.z*c[QCI::YZZ] 
		+p.y*p.z*c[QCI::YZ] 
		+p.y*p.y*p.z*p.z*c[QCI::YYZZ] 
		+p.y*p.y*p.z*c[QCI::YYZ] 
		+p.y*p.y*p.y*p.z*c[QCI::YYYZ] 
		+p.y*p.y*p.y*p.y*c[QCI::YYYY] 
		+p.y*p.y*p.y*c[QCI::YYY] 
		+p.y*p.y*c[QCI::YY] 
		+p.y*c[QCI::Y] 
		+p.x*p.z*p.z*p.z*c[QCI::XZZZ] 
		+p.x*p.z*p.z*c[QCI::XZZ] 
		+p.x*p.z*c[QCI::XZ] 
		+p.x*p.y*p.z*p.z*c[QCI::XYZZ] 
		+p.x*p.y*p.z*c[QCI::XYZ] 
		+p.x*p.y*p.y*p.z*c[QCI::XYYZ] 
		+p.x*p.y*p.y*p.y*c[QCI::XYYY] 
		+p.x*p.y*p.y*c[QCI::XYY] 
		+p.x*p.y*c[QCI::XY] 
		+p.x*p.x*p.z*p.z*c[QCI::XXZZ] 
		+p.x*p.x*p.z*c[QCI::XXZ] 
		+p.x*p.x*p.y*p.z*c[QCI::XXYZ] 
		+p.x*p.x*p.y*p.y*c[QCI::XXYY] 
		+p.x*p.x*p.y*c[QCI::XXY] 
		+p.x*p.x*p.x*p.z*c[QCI::XXXZ] 
		+p.x*p.x*p.x*p.y*c[QCI::XXXY] 
		+p.x*p.x*p.x*p.x*c[QCI::XXXX] 
		+p.x*p.x*p.x*c[QCI::XXX] 
		+p.x*p.x*c[QCI::XX] 
		+p.x*c[QCI::X] 
		+c[QCI::C];
	double t = 
		+4*p.z*p.z*p.z*q.z*c[QCI::ZZZZ] 
		+3*p.z*p.z*q.z*c[QCI::ZZZ] 
		+2*p.z*q.z*c[QCI::ZZ] 
		+q.z*c[QCI::Z] 
		+3*p.y*p.z*p.z*q.z*c[QCI::YZZZ] 
		+p.z*p.z*p.z*q.y*c[QCI::YZZZ] 
		+2*p.y*p.z*q.z*c[QCI::YZZ] 
		+p.z*p.z*q.y*c[QCI::YZZ] 
		+p.y*q.z*c[QCI::YZ] 
		+p.z*q.y*c[QCI::YZ] 
		+2*p.y*p.y*p.z*q.z*c[QCI::YYZZ] 
		+2*p.y*p.z*p.z*q.y*c[QCI::YYZZ] 
		+p.y*p.y*q.z*c[QCI::YYZ] 
		+2*p.y*p.z*q.y*c[QCI::YYZ] 
		+p.y*p.y*p.y*q.z*c[QCI::YYYZ] 
		+3*p.y*p.y*p.z*q.y*c[QCI::YYYZ] 
		+4*p.y*p.y*p.y*q.y*c[QCI::YYYY] 
		+3*p.y*p.y*q.y*c[QCI::YYY] 
		+2*p.y*q.y*c[QCI::YY] 
		+q.y*c[QCI::Y] 
		+3*p.x*p.z*p.z*q.z*c[QCI::XZZZ] 
		+p.z*p.z*p.z*q.x*c[QCI::XZZZ] 
		+2*p.x*p.z*q.z*c[QCI::XZZ] 
		+p.z*p.z*q.x*c[QCI::XZZ] 
		+p.x*q.z*c[QCI::XZ] 
		+p.z*q.x*c[QCI::XZ] 
		+2*p.x*p.y*p.z*q.z*c[QCI::XYZZ] 
		+p.x*p.z*p.z*q.y*c[QCI::XYZZ] 
		+p.y*p.z*p.z*q.x*c[QCI::XYZZ] 
		+p.x*p.y*q.z*c[QCI::XYZ] 
		+p.x*p.z*q.y*c[QCI::XYZ] 
		+p.y*p.z*q.x*c[QCI::XYZ] 
		+p.x*p.y*p.y*q.z*c[QCI::XYYZ] 
		+2*p.x*p.y*p.z*q.y*c[QCI::XYYZ] 
		+p.y*p.y*p.z*q.x*c[QCI::XYYZ] 
		+3*p.x*p.y*p.y*q.y*c[QCI::XYYY] 
		+p.y*p.y*p.y*q.x*c[QCI::XYYY] 
		+2*p.x*p.y*q.y*c[QCI::XYY] 
		+p.y*p.y*q.x*c[QCI::XYY] 
		+p.x*q.y*c[QCI::XY] 
		+p.y*q.x*c[QCI::XY] 
		+2*p.x*p.x*p.z*q.z*c[QCI::XXZZ] 
		+2*p.x*p.z*p.z*q.x*c[QCI::XXZZ] 
		+p.x*p.x*q.z*c[QCI::XXZ] 
		+2*p.x*p.z*q.x*c[QCI::XXZ] 
		+p.x*p.x*p.y*q.z*c[QCI::XXYZ] 
		+p.x*p.x*p.z*q.y*c[QCI::XXYZ] 
		+2*p.x*p.y*p.z*q.x*c[QCI::XXYZ] 
		+2*p.x*p.x*p.y*q.y*c[QCI::XXYY] 
		+2*p.x*p.y*p.y*q.x*c[QCI::XXYY] 
		+p.x*p.x*q.y*c[QCI::XXY] 
		+2*p.x*p.y*q.x*c[QCI::XXY] 
		+p.x*p.x*p.x*q.z*c[QCI::XXXZ] 
		+3*p.x*p.x*p.z*q.x*c[QCI::XXXZ] 
		+p.x*p.x*p.x*q.y*c[QCI::XXXY] 
		+3*p.x*p.x*p.y*q.x*c[QCI::XXXY] 
		+4*p.x*p.x*p.x*q.x*c[QCI::XXXX] 
		+3*p.x*p.x*q.x*c[QCI::XXX] 
		+2*p.x*q.x*c[QCI::XX] 
		+q.x*c[QCI::X];
	double tt = 
		+6*p.z*p.z*q.z*q.z*c[QCI::ZZZZ] 
		+3*p.z*q.z*q.z*c[QCI::ZZZ] 
		+q.z*q.z*c[QCI::ZZ] 
		+3*p.y*p.z*q.z*q.z*c[QCI::YZZZ] 
		+3*p.z*p.z*q.y*q.z*c[QCI::YZZZ] 
		+p.y*q.z*q.z*c[QCI::YZZ] 
		+2*p.z*q.y*q.z*c[QCI::YZZ] 
		+q.y*q.z*c[QCI::YZ] 
		+p.y*p.y*q.z*q.z*c[QCI::YYZZ] 
		+4*p.y*p.z*q.y*q.z*c[QCI::YYZZ] 
		+p.z*p.z*q.y*q.y*c[QCI::YYZZ] 
		+2*p.y*q.y*q.z*c[QCI::YYZ] 
		+p.z*q.y*q.y*c[QCI::YYZ] 
		+3*p.y*p.y*q.y*q.z*c[QCI::YYYZ] 
		+3*p.y*p.z*q.y*q.y*c[QCI::YYYZ] 
		+6*p.y*p.y*q.y*q.y*c[QCI::YYYY] 
		+3*p.y*q.y*q.y*c[QCI::YYY] 
		+q.y*q.y*c[QCI::YY] 
		+3*p.x*p.z*q.z*q.z*c[QCI::XZZZ] 
		+3*p.z*p.z*q.x*q.z*c[QCI::XZZZ] 
		+p.x*q.z*q.z*c[QCI::XZZ] 
		+2*p.z*q.x*q.z*c[QCI::XZZ] 
		+q.x*q.z*c[QCI::XZ] 
		+p.x*p.y*q.z*q.z*c[QCI::XYZZ] 
		+2*p.x*p.z*q.y*q.z*c[QCI::XYZZ] 
		+2*p.y*p.z*q.x*q.z*c[QCI::XYZZ] 
		+p.z*p.z*q.x*q.y*c[QCI::XYZZ] 
		+p.x*q.y*q.z*c[QCI::XYZ] 
		+p.y*q.x*q.z*c[QCI::XYZ] 
		+p.z*q.x*q.y*c[QCI::XYZ] 
		+2*p.x*p.y*q.y*q.z*c[QCI::XYYZ] 
		+p.y*p.y*q.x*q.z*c[QCI::XYYZ] 
		+p.x*p.z*q.y*q.y*c[QCI::XYYZ] 
		+2*p.y*p.z*q.x*q.y*c[QCI::XYYZ] 
		+3*p.x*p.y*q.y*q.y*c[QCI::XYYY] 
		+3*p.y*p.y*q.x*q.y*c[QCI::XYYY] 
		+p.x*q.y*q.y*c[QCI::XYY] 
		+2*p.y*q.x*q.y*c[QCI::XYY] 
		+q.x*q.y*c[QCI::XY] 
		+p.x*p.x*q.z*q.z*c[QCI::XXZZ] 
		+4*p.x*p.z*q.x*q.z*c[QCI::XXZZ] 
		+p.z*p.z*q.x*q.x*c[QCI::XXZZ] 
		+2*p.x*q.x*q.z*c[QCI::XXZ] 
		+p.z*q.x*q.x*c[QCI::XXZ] 
		+p.x*p.x*q.y*q.z*c[QCI::XXYZ] 
		+2*p.x*p.y*q.x*q.z*c[QCI::XXYZ] 
		+2*p.x*p.z*q.x*q.y*c[QCI::XXYZ] 
		+p.y*p.z*q.x*q.x*c[QCI::XXYZ] 
		+p.x*p.x*q.y*q.y*c[QCI::XXYY] 
		+4*p.x*p.y*q.x*q.y*c[QCI::XXYY] 
		+p.y*p.y*q.x*q.x*c[QCI::XXYY] 
		+2*p.x*q.x*q.y*c[QCI::XXY] 
		+p.y*q.x*q.x*c[QCI::XXY] 
		+3*p.x*p.x*q.x*q.z*c[QCI::XXXZ] 
		+3*p.x*p.z*q.x*q.x*c[QCI::XXXZ] 
		+3*p.x*p.x*q.x*q.y*c[QCI::XXXY] 
		+3*p.x*p.y*q.x*q.x*c[QCI::XXXY] 
		+6*p.x*p.x*q.x*q.x*c[QCI::XXXX] 
		+3*p.x*q.x*q.x*c[QCI::XXX] 
		+q.x*q.x*c[QCI::XX];
	double ttt = 
		+4*p.z*q.z*q.z*q.z*c[QCI::ZZZZ] 
		+q.z*q.z*q.z*c[QCI::ZZZ] 
		+p.y*q.z*q.z*q.z*c[QCI::YZZZ] 
		+3*p.z*q.y*q.z*q.z*c[QCI::YZZZ] 
		+q.y*q.z*q.z*c[QCI::YZZ] 
		+2*p.y*q.y*q.z*q.z*c[QCI::YYZZ] 
		+2*p.z*q.y*q.y*q.z*c[QCI::YYZZ] 
		+q.y*q.y*q.z*c[QCI::YYZ] 
		+3*p.y*q.y*q.y*q.z*c[QCI::YYYZ] 
		+p.z*q.y*q.y*q.y*c[QCI::YYYZ] 
		+4*p.y*q.y*q.y*q.y*c[QCI::YYYY] 
		+q.y*q.y*q.y*c[QCI::YYY] 
		+p.x*q.z*q.z*q.z*c[QCI::XZZZ] 
		+3*p.z*q.x*q.z*q.z*c[QCI::XZZZ] 
		+q.x*q.z*q.z*c[QCI::XZZ] 
		+p.x*q.y*q.z*q.z*c[QCI::XYZZ] 
		+p.y*q.x*q.z*q.z*c[QCI::XYZZ] 
		+2*p.z*q.x*q.y*q.z*c[QCI::XYZZ] 
		+q.x*q.y*q.z*c[QCI::XYZ] 
		+p.x*q.y*q.y*q.z*c[QCI::XYYZ] 
		+2*p.y*q.x*q.y*q.z*c[QCI::XYYZ] 
		+p.z*q.x*q.y*q.y*c[QCI::XYYZ] 
		+p.x*q.y*q.y*q.y*c[QCI::XYYY] 
		+3*p.y*q.x*q.y*q.y*c[QCI::XYYY] 
		+q.x*q.y*q.y*c[QCI::XYY] 
		+2*p.x*q.x*q.z*q.z*c[QCI::XXZZ] 
		+2*p.z*q.x*q.x*q.z*c[QCI::XXZZ] 
		+q.x*q.x*q.z*c[QCI::XXZ] 
		+2*p.x*q.x*q.y*q.z*c[QCI::XXYZ] 
		+p.y*q.x*q.x*q.z*c[QCI::XXYZ] 
		+p.z*q.x*q.x*q.y*c[QCI::XXYZ] 
		+2*p.x*q.x*q.y*q.y*c[QCI::XXYY] 
		+2*p.y*q.x*q.x*q.y*c[QCI::XXYY] 
		+q.x*q.x*q.y*c[QCI::XXY] 
		+3*p.x*q.x*q.x*q.z*c[QCI::XXXZ] 
		+p.z*q.x*q.x*q.x*c[QCI::XXXZ] 
		+3*p.x*q.x*q.x*q.y*c[QCI::XXXY] 
		+p.y*q.x*q.x*q.x*c[QCI::XXXY] 
		+4*p.x*q.x*q.x*q.x*c[QCI::XXXX] 
		+q.x*q.x*q.x*c[QCI::XXX];
	double tttt = 
		+q.z*q.z*q.z*q.z*c[QCI::ZZZZ] 
		+q.y*q.z*q.z*q.z*c[QCI::YZZZ] 
		+q.y*q.y*q.z*q.z*c[QCI::YYZZ] 
		+q.y*q.y*q.y*q.z*c[QCI::YYYZ] 
		+q.y*q.y*q.y*q.y*c[QCI::YYYY] 
		+q.x*q.z*q.z*q.z*c[QCI::XZZZ] 
		+q.x*q.y*q.z*q.z*c[QCI::XYZZ] 
		+q.x*q.y*q.y*q.z*c[QCI::XYYZ] 
		+q.x*q.y*q.y*q.y*c[QCI::XYYY] 
		+q.x*q.x*q.z*q.z*c[QCI::XXZZ] 
		+q.x*q.x*q.y*q.z*c[QCI::XXYZ] 
		+q.x*q.x*q.y*q.y*c[QCI::XXYY] 
		+q.x*q.x*q.x*q.z*c[QCI::XXXZ] 
		+q.x*q.x*q.x*q.y*c[QCI::XXXY] 
		+q.x*q.x*q.x*q.x*c[QCI::XXXX];
	intersect_quartic_t(tttt, ttt, tt, t, k, p, q, this, ils[0]);
	}
/*...e*/
/*...e*/
/*...sPrimShapeBinOp:0:*/
// Binary combination of 2 shapes.
// Again, like PrimShape, this is an abstract base class.
// You'd never actually have an instance of this.

class PrimShapeBinOp : public PrimShape
	{
protected:
	PrimShape *l, *r;
	Boolean overlap;		// Could l and r overlap
					// Set up by extent()
					// Can accelerate Ray-Tracing
public:
	PrimShapeBinOp(PrimShape *l, PrimShape *r)
		: l(l), r(r), overlap(TRUE) {}
	virtual ~PrimShapeBinOp() { delete l; delete r; }
	virtual Xyz normal(const Xyz & p) const { return Xyz(0.0,0.0,0.0); }
	virtual void preprocess();
	virtual int isect_depth() const;
	};

int PrimShapeBinOp::isect_depth() const
	{
#ifdef BETTER_DIFF
	// This new variation allows us to choose to intersect with
	// either the lhs or rhs first, and then the other. This is important,
	// as with lhs-rhs, it appears to be quicker to do the lhs first.
	int depth_l = l->isect_depth();
	int depth_r = r->isect_depth();
	return 2 + Max(depth_l, depth_r);
#else
	// This variation assumes we always intersect with the rhs first.
	// Can use slightly less memory, but can be slower.
	int depth_l = 1 + l->isect_depth();
	int depth_r =     r->isect_depth();
	return 1 + Max(depth_l, depth_r);
#endif
	}

void PrimShapeBinOp::preprocess()
	{
	l->preprocess();
	r->preprocess();
	}
/*...e*/
/*...sPrimShapeUnion:2:*/
class PrimShapeUnion : public PrimShapeBinOp
	{
public:
	PrimShapeUnion(PrimShape *l, PrimShape *r);
	virtual ~PrimShapeUnion();
	virtual PrimShape *copy() const;
	virtual PrimShape *trans  (Xyz    t     ) const;
	virtual PrimShape *trans_x(double t     ) const;
	virtual PrimShape *trans_y(double t     ) const;
	virtual PrimShape *trans_z(double t     ) const;
	virtual PrimShape *rot_x  (double angle ) const;
	virtual PrimShape *rot_y  (double angle ) const;
	virtual PrimShape *rot_z  (double angle ) const;
	virtual PrimShape *scale  (Xyz    factor) const;
	virtual PrimShape *scale  (double factor) const;
	virtual PrimShape *scale_x(double factor) const;
	virtual PrimShape *scale_y(double factor) const;
	virtual PrimShape *scale_z(double factor) const;
	virtual PrimShape *resurf(Surf s) const;
	virtual Extent extent();
	virtual PrimShape *simplify(
		double pa, double pb, double pc, double pd) const;
	virtual int isects_reqd() const;
	virtual void intersect(const Xyz & p, const Xyz & q, IsectList *ils[]) const;
	};

/*...sPrimShapeUnion:2:*/
PrimShapeUnion::PrimShapeUnion(PrimShape *l, PrimShape *r) : PrimShapeBinOp(l, r) {}
/*...e*/
/*...s\126\PrimShapeUnion:2:*/
PrimShapeUnion::~PrimShapeUnion() {}
/*...e*/
/*...scopy:2:*/
PrimShape *PrimShapeUnion::copy() const { return new PrimShapeUnion(l->copy(), r->copy()); }
/*...e*/
/*...strans\39\s:2:*/
PrimShape *PrimShapeUnion::trans  (Xyz    t) const { return new PrimShapeUnion(l->trans  (t),r->trans  (t)); }
PrimShape *PrimShapeUnion::trans_x(double t) const { return new PrimShapeUnion(l->trans_x(t),r->trans_x(t)); }
PrimShape *PrimShapeUnion::trans_y(double t) const { return new PrimShapeUnion(l->trans_y(t),r->trans_y(t)); }
PrimShape *PrimShapeUnion::trans_z(double t) const { return new PrimShapeUnion(l->trans_z(t),r->trans_z(t)); }
/*...e*/
/*...srot\39\s:2:*/
PrimShape *PrimShapeUnion::rot_x(double angle) const { return new PrimShapeUnion(l->rot_x(angle),r->rot_x(angle)); }
PrimShape *PrimShapeUnion::rot_y(double angle) const { return new PrimShapeUnion(l->rot_y(angle),r->rot_y(angle)); }
PrimShape *PrimShapeUnion::rot_z(double angle) const { return new PrimShapeUnion(l->rot_z(angle),r->rot_z(angle)); }
/*...e*/
/*...sscale\39\s:2:*/
PrimShape *PrimShapeUnion::scale  (Xyz    factor) const { return new PrimShapeUnion(l->scale  (factor),r->scale  (factor)); }
PrimShape *PrimShapeUnion::scale  (double factor) const { return new PrimShapeUnion(l->scale  (factor),r->scale  (factor)); }
PrimShape *PrimShapeUnion::scale_x(double factor) const { return new PrimShapeUnion(l->scale_x(factor),r->scale_x(factor)); }
PrimShape *PrimShapeUnion::scale_y(double factor) const { return new PrimShapeUnion(l->scale_y(factor),r->scale_y(factor)); }
PrimShape *PrimShapeUnion::scale_z(double factor) const { return new PrimShapeUnion(l->scale_z(factor),r->scale_z(factor)); }
/*...e*/
/*...sresurf:2:*/
PrimShape *PrimShapeUnion::resurf(Surf surf) const { return new PrimShapeUnion(l->resurf(surf),r->resurf(surf)); }
/*...e*/
/*...sextent:2:*/
Extent PrimShapeUnion::extent()
	{
	Extent extl(l->extent());
	Extent extr(r->extent());
	overlap = (extl && extr);
	return extl | extr;
	}
/*...e*/
/*...ssimplify:2:*/
// If either are on the wrong side of plane, they're removed from the result

PrimShape *PrimShapeUnion::simplify(double pa, double pb, double pc, double pd) const
	{
	PrimShape *sl = simpex( l->simplify(pa,pb,pc,pd), pa,pb,pc,pd);
	PrimShape *sr = simpex( r->simplify(pa,pb,pc,pd), pa,pb,pc,pd);
	if ( sl == 0 )
		return sr;
	if ( sr == 0 )
		return sl;
	return new PrimShapeUnion(sl, sr);
	}
/*...e*/
/*...sisects_reqd:2:*/
int PrimShapeUnion::isects_reqd() const
	{
	return l->isects_reqd() + r->isects_reqd();
	}
/*...e*/
/*...sintersect:2:*/
/*...scombine_union:2:*/
static Boolean combine_union(Boolean in_a, Boolean in_b)
	{
	return in_a || in_b;
	}
/*...e*/

void PrimShapeUnion::intersect(const Xyz & p, const Xyz & q, IsectList *ils[]) const
	{
	if ( overlap )
		{
		r->intersect(p, q, ils+1);
		if ( ils[1]->is_empty() )
			l->intersect(p, q, ils);
		else if ( ils[1]->is_solid() )
			swap_isectls(ils[0], ils[1]);
		else
			{
			l->intersect(p, q, ils+2);
			if ( ils[2]->is_empty() )
				swap_isectls(ils[0], ils[1]);
			else if ( ils[2]->is_solid() )
				swap_isectls(ils[0], ils[2]);
			else
				combine_isectls(combine_union, ils[2], ils[1], ils[0]);
			}
		}
	else
		// No overlap, treat like concatenation
		{
		r->intersect(p, q, ils+1);
		l->intersect(p, q, ils+2);
		concat_isectls(ils[2], ils[1], ils[0]);
		}
	}
/*...e*/
/*...e*/
/*...sPrimShapeIsect:2:*/
class PrimShapeIsect : public PrimShapeBinOp
	{
public:
	PrimShapeIsect(PrimShape *l, PrimShape *r);
	virtual ~PrimShapeIsect();
	virtual PrimShape *copy() const;
	virtual PrimShape *trans  (Xyz    t     ) const;
	virtual PrimShape *trans_x(double t     ) const;
	virtual PrimShape *trans_y(double t     ) const;
	virtual PrimShape *trans_z(double t     ) const;
	virtual PrimShape *rot_x  (double angle ) const;
	virtual PrimShape *rot_y  (double angle ) const;
	virtual PrimShape *rot_z  (double angle ) const;
	virtual PrimShape *scale  (Xyz    factor) const;
	virtual PrimShape *scale  (double factor) const;
	virtual PrimShape *scale_x(double factor) const;
	virtual PrimShape *scale_y(double factor) const;
	virtual PrimShape *scale_z(double factor) const;
	virtual PrimShape *resurf(Surf s) const;
	virtual Extent extent();
	virtual PrimShape *simplify(
		double pa, double pb, double pc, double pd) const;
	virtual int isects_reqd() const;
	virtual void intersect(const Xyz & p, const Xyz & q, IsectList *ils[]) const;
	};

/*...sPrimShapeIsect:2:*/
PrimShapeIsect::PrimShapeIsect(PrimShape *l, PrimShape *r) : PrimShapeBinOp(l, r) {}
/*...e*/
/*...s\126\PrimShapeIsect:2:*/
PrimShapeIsect::~PrimShapeIsect() {}
/*...e*/
/*...scopy:2:*/
PrimShape *PrimShapeIsect::copy() const { return new PrimShapeIsect(l->copy(), r->copy()); }
/*...e*/
/*...strans\39\s:2:*/
PrimShape *PrimShapeIsect::trans  (Xyz    t) const { return new PrimShapeIsect(l->trans  (t),r->trans  (t)); }
PrimShape *PrimShapeIsect::trans_x(double t) const { return new PrimShapeIsect(l->trans_x(t),r->trans_x(t)); }
PrimShape *PrimShapeIsect::trans_y(double t) const { return new PrimShapeIsect(l->trans_y(t),r->trans_y(t)); }
PrimShape *PrimShapeIsect::trans_z(double t) const { return new PrimShapeIsect(l->trans_z(t),r->trans_z(t)); }
/*...e*/
/*...srot\39\s:2:*/
PrimShape *PrimShapeIsect::rot_x(double angle) const { return new PrimShapeIsect(l->rot_x(angle),r->rot_x(angle)); }
PrimShape *PrimShapeIsect::rot_y(double angle) const { return new PrimShapeIsect(l->rot_y(angle),r->rot_y(angle)); }
PrimShape *PrimShapeIsect::rot_z(double angle) const { return new PrimShapeIsect(l->rot_z(angle),r->rot_z(angle)); }
/*...e*/
/*...sscale\39\s:2:*/
PrimShape *PrimShapeIsect::scale  (Xyz    factor) const { return new PrimShapeIsect(l->scale  (factor),r->scale  (factor)); }
PrimShape *PrimShapeIsect::scale  (double factor) const { return new PrimShapeIsect(l->scale  (factor),r->scale  (factor)); }
PrimShape *PrimShapeIsect::scale_x(double factor) const { return new PrimShapeIsect(l->scale_x(factor),r->scale_x(factor)); }
PrimShape *PrimShapeIsect::scale_y(double factor) const { return new PrimShapeIsect(l->scale_y(factor),r->scale_y(factor)); }
PrimShape *PrimShapeIsect::scale_z(double factor) const { return new PrimShapeIsect(l->scale_z(factor),r->scale_z(factor)); }
/*...e*/
/*...sresurf:2:*/
PrimShape *PrimShapeIsect::resurf(Surf surf) const { return new PrimShapeIsect(l->resurf(surf),r->resurf(surf)); }
/*...e*/
/*...sextent:2:*/
Extent PrimShapeIsect::extent()
	{
	Extent extl(l->extent());
	Extent extr(r->extent());
	overlap = (extl && extr);
	if ( overlap )
		return extl & extr;
	else
		return Extent(); // Empty extent
	}
/*...e*/
/*...ssimplify:2:*/
// If either are the wrong side of the plane, the result is a void

PrimShape *PrimShapeIsect::simplify(double pa, double pb, double pc, double pd) const
	{
	PrimShape *sl = simpex( l->simplify(pa,pb,pc,pd), pa,pb,pc,pd);
	if ( sl == 0 )
		return 0;
	PrimShape *sr = simpex( r->simplify(pa,pb,pc,pd), pa,pb,pc,pd);
	if ( sr == 0 )
		{
		delete sl;
		return 0;
		}
	return new PrimShapeIsect(sl, sr);
	}
/*...e*/
/*...sisects_reqd:2:*/
int PrimShapeIsect::isects_reqd() const
	{
	return overlap ? l->isects_reqd() + r->isects_reqd() : 0;
	}
/*...e*/
/*...sintersect:2:*/
/*...scombine_isect:2:*/
static Boolean combine_isect(Boolean in_a, Boolean in_b)
	{
	return in_a && in_b;
	}
/*...e*/

void PrimShapeIsect::intersect(const Xyz & p, const Xyz & q, IsectList *ils[]) const
	{
	if ( overlap )
		{
		r->intersect(p, q, ils+1);
		if ( ils[1]->is_empty() )
			ils[0]->n_isects = 0;
		else if ( ils[1]->is_solid() )
			l->intersect(p, q, ils);
		else
			{
			l->intersect(p, q, ils+2);
			if ( ils[2]->is_empty() )
				ils[0]->n_isects = 0;
			else if ( ils[2]->is_solid() )
				swap_isectls(ils[0], ils[1]);
			else
				combine_isectls(combine_isect, ils[2], ils[1], ils[0]);
			}
		}
	else
		// No overlap, empty list is the result
		ils[0]->n_isects = 0;
	}
/*...e*/
/*...e*/
/*...sPrimShapeDiff:2:*/
class PrimShapeDiff : public PrimShapeBinOp
	{
public:
	PrimShapeDiff(PrimShape *l, PrimShape *r);
	virtual ~PrimShapeDiff();
	virtual PrimShape *copy() const;
	virtual PrimShape *trans  (Xyz    t     ) const;
	virtual PrimShape *trans_x(double t     ) const;
	virtual PrimShape *trans_y(double t     ) const;
	virtual PrimShape *trans_z(double t     ) const;
	virtual PrimShape *rot_x  (double angle ) const;
	virtual PrimShape *rot_y  (double angle ) const;
	virtual PrimShape *rot_z  (double angle ) const;
	virtual PrimShape *scale  (Xyz    factor) const;
	virtual PrimShape *scale  (double factor) const;
	virtual PrimShape *scale_x(double factor) const;
	virtual PrimShape *scale_y(double factor) const;
	virtual PrimShape *scale_z(double factor) const;
	virtual PrimShape *resurf(Surf s) const;
	virtual Extent extent();
	virtual PrimShape *simplify(
		double pa, double pb, double pc, double pd) const;
	virtual int isects_reqd() const;
	virtual void intersect(const Xyz & p, const Xyz & q, IsectList *ils[]) const;
	};

/*...sPrimShapeDiff:2:*/
PrimShapeDiff::PrimShapeDiff(PrimShape *l, PrimShape *r) : PrimShapeBinOp(l, r) {}
/*...e*/
/*...s\126\PrimShapeDiff:2:*/
PrimShapeDiff::~PrimShapeDiff() {}
/*...e*/
/*...scopy:2:*/
PrimShape *PrimShapeDiff::copy() const { return new PrimShapeDiff(l->copy(), r->copy()); }
/*...e*/
/*...strans\39\s:2:*/
PrimShape *PrimShapeDiff::trans  (Xyz    t) const { return new PrimShapeDiff(l->trans  (t),r->trans  (t)); }
PrimShape *PrimShapeDiff::trans_x(double t) const { return new PrimShapeDiff(l->trans_x(t),r->trans_x(t)); }
PrimShape *PrimShapeDiff::trans_y(double t) const { return new PrimShapeDiff(l->trans_y(t),r->trans_y(t)); }
PrimShape *PrimShapeDiff::trans_z(double t) const { return new PrimShapeDiff(l->trans_z(t),r->trans_z(t)); }
/*...e*/
/*...srot\39\s:2:*/
PrimShape *PrimShapeDiff::rot_x  (double angle) const { return new PrimShapeDiff(l->rot_x(angle),r->rot_x(angle)); }
PrimShape *PrimShapeDiff::rot_y  (double angle) const { return new PrimShapeDiff(l->rot_y(angle),r->rot_y(angle)); }
PrimShape *PrimShapeDiff::rot_z  (double angle) const { return new PrimShapeDiff(l->rot_z(angle),r->rot_z(angle)); }
/*...e*/
/*...sscale\39\s:2:*/
PrimShape *PrimShapeDiff::scale  (Xyz    factor) const { return new PrimShapeDiff(l->scale  (factor),r->scale  (factor)); }
PrimShape *PrimShapeDiff::scale  (double factor) const { return new PrimShapeDiff(l->scale  (factor),r->scale  (factor)); }
PrimShape *PrimShapeDiff::scale_x(double factor) const { return new PrimShapeDiff(l->scale_x(factor),r->scale_x(factor)); }
PrimShape *PrimShapeDiff::scale_y(double factor) const { return new PrimShapeDiff(l->scale_y(factor),r->scale_y(factor)); }
PrimShape *PrimShapeDiff::scale_z(double factor) const { return new PrimShapeDiff(l->scale_z(factor),r->scale_z(factor)); }
/*...e*/
/*...sresurf:2:*/
PrimShape *PrimShapeDiff::resurf(Surf surf) const
	{ return new PrimShapeDiff(l->resurf(surf),r->resurf(surf)); }
/*...e*/
/*...sextent:2:*/
Extent PrimShapeDiff::extent()
	{
	Extent extl(l->extent());
	Extent extr(r->extent());
	overlap = (extl && extr);
	return extl;
	}
/*...e*/
/*...ssimplify:2:*/
// If l is on the wrong side, the answer is a void
// If r is on the wrong side, the answer is l

PrimShape *PrimShapeDiff::simplify(double pa, double pb, double pc, double pd) const
	{
	PrimShape *sl = simpex( l->simplify(pa,pb,pc,pd), pa,pb,pc,pd);
	if ( sl == 0 )
		return 0;
	PrimShape *sr = simpex( r->simplify(pa,pb,pc,pd), pa,pb,pc,pd);
	if ( sr == 0 )
		return sl;
	return new PrimShapeDiff(sl, sr);
	}
/*...e*/
/*...sisects_reqd:2:*/
int PrimShapeDiff::isects_reqd() const
	{
	return overlap ? l->isects_reqd() + r->isects_reqd() : l->isects_reqd();
	}
/*...e*/
/*...sintersect:2:*/
/*...scombine_diff:2:*/
static Boolean combine_diff(Boolean in_a, Boolean in_b)
	{
	return in_a && !in_b;
	}
/*...e*/

void PrimShapeDiff::intersect(const Xyz & p, const Xyz & q, IsectList *ils[]) const
	{
	if ( overlap )
		{
#ifdef BETTER_DIFF
		// This variation appears to be quicker, as the lhs
		// forms a natural "extent" around the whole result.
		l->intersect(p, q, ils+1);
		if ( ils[1]->is_empty() )
			ils[0]->n_isects = 0;
		else
			{
			r->intersect(p, q, ils+2);
			if ( ils[2]->is_empty() )
				swap_isectls(ils[0], ils[1]);
			else if ( ils[2]->is_solid() )
				ils[0]->n_isects = 0;
			else
				combine_isectls(combine_diff, ils[1], ils[2], ils[0]);
			}
#else
		// This is the version I originally coded
		r->intersect(p, q, ils+1);
		if ( ils[1]->is_empty() )
			l->intersect(p, q, ils);
		else if ( ils[1]->is_solid() )
			ils[0]->n_isects = 0;
		else
			{
			l->intersect(p, q, ils+2);
			if ( ils[2]->is_empty() )
				ils[0]->n_isects = 0;
			else
				combine_isectls(combine_diff, ils[2], ils[1], ils[0]);
			}
#endif
		}
	else
		// No possibility of overlap, lhs is result
		l->intersect(p, q, ils);
	}
/*...e*/
/*...e*/
/*...sPrimShapeSdiff:2:*/
class PrimShapeSdiff : public PrimShapeBinOp
	{
public:
	PrimShapeSdiff(PrimShape *l, PrimShape *r);
	virtual ~PrimShapeSdiff();
	virtual PrimShape *copy() const;
	virtual PrimShape *trans  (Xyz    t     ) const;
	virtual PrimShape *trans_x(double t     ) const;
	virtual PrimShape *trans_y(double t     ) const;
	virtual PrimShape *trans_z(double t     ) const;
	virtual PrimShape *rot_x  (double angle ) const;
	virtual PrimShape *rot_y  (double angle ) const;
	virtual PrimShape *rot_z  (double angle ) const;
	virtual PrimShape *scale  (Xyz    factor) const;
	virtual PrimShape *scale  (double factor) const;
	virtual PrimShape *scale_x(double factor) const;
	virtual PrimShape *scale_y(double factor) const;
	virtual PrimShape *scale_z(double factor) const;
	virtual PrimShape *resurf(Surf s) const;
	virtual Extent extent();
	virtual PrimShape *simplify(
		double pa, double pb, double pc, double pd) const;
	virtual int isects_reqd() const;
	virtual void intersect(const Xyz & p, const Xyz & q, IsectList *ils[]) const;
	};

/*...sPrimShapeSdiff:2:*/
PrimShapeSdiff::PrimShapeSdiff(PrimShape *l, PrimShape *r) : PrimShapeBinOp(l, r) {}
/*...e*/
/*...s\126\PrimShapeSdiff:2:*/
PrimShapeSdiff::~PrimShapeSdiff() {}
/*...e*/
/*...scopy:2:*/
PrimShape *PrimShapeSdiff::copy() const { return new PrimShapeSdiff(l->copy(), r->copy()); }
/*...e*/
/*...strans\39\s:2:*/
PrimShape *PrimShapeSdiff::trans  (Xyz    t) const { return new PrimShapeSdiff(l->trans  (t),r->trans  (t)); }
PrimShape *PrimShapeSdiff::trans_x(double t) const { return new PrimShapeSdiff(l->trans_x(t),r->trans_x(t)); }
PrimShape *PrimShapeSdiff::trans_y(double t) const { return new PrimShapeSdiff(l->trans_y(t),r->trans_y(t)); }
PrimShape *PrimShapeSdiff::trans_z(double t) const { return new PrimShapeSdiff(l->trans_z(t),r->trans_z(t)); }
/*...e*/
/*...srot\39\s:2:*/
PrimShape *PrimShapeSdiff::rot_x  (double angle) const { return new PrimShapeSdiff(l->rot_x(angle),r->rot_x(angle)); }
PrimShape *PrimShapeSdiff::rot_y  (double angle) const { return new PrimShapeSdiff(l->rot_y(angle),r->rot_y(angle)); }
PrimShape *PrimShapeSdiff::rot_z  (double angle) const { return new PrimShapeSdiff(l->rot_z(angle),r->rot_z(angle)); }
/*...e*/
/*...sscale\39\s:2:*/
PrimShape *PrimShapeSdiff::scale  (Xyz    factor) const { return new PrimShapeSdiff(l->scale  (factor),r->scale  (factor)); }
PrimShape *PrimShapeSdiff::scale  (double factor) const { return new PrimShapeSdiff(l->scale  (factor),r->scale  (factor)); }
PrimShape *PrimShapeSdiff::scale_x(double factor) const { return new PrimShapeSdiff(l->scale_x(factor),r->scale_x(factor)); }
PrimShape *PrimShapeSdiff::scale_y(double factor) const { return new PrimShapeSdiff(l->scale_y(factor),r->scale_y(factor)); }
PrimShape *PrimShapeSdiff::scale_z(double factor) const { return new PrimShapeSdiff(l->scale_z(factor),r->scale_z(factor)); }
/*...e*/
/*...sresurf:2:*/
PrimShape *PrimShapeSdiff::resurf(Surf surf) const { return new PrimShapeSdiff(l->resurf(surf),r->resurf(surf)); }
/*...e*/
/*...sextent:2:*/
Extent PrimShapeSdiff::extent()
	{
	Extent extl(l->extent());
	Extent extr(r->extent());
	overlap = (extl && extr);
	return extl | extr;
	}
/*...e*/
/*...ssimplify:2:*/
// Treat much like PrimShapeUnion

PrimShape *PrimShapeSdiff::simplify(double pa, double pb, double pc, double pd) const
	{
	PrimShape *sl = simpex( l->simplify(pa,pb,pc,pd), pa,pb,pc,pd);
	PrimShape *sr = simpex( r->simplify(pa,pb,pc,pd), pa,pb,pc,pd);
	if ( sl == 0 )
		return sr;
	if ( sr == 0 )
		return sl;
	return new PrimShapeSdiff(sl, sr);
	}
/*...e*/
/*...sisects_reqd:2:*/
int PrimShapeSdiff::isects_reqd() const
	{
	return l->isects_reqd() + r->isects_reqd();
	}
/*...e*/
/*...sintersect:2:*/
/*...scombine_sdiff:2:*/
static Boolean combine_sdiff(Boolean in_a, Boolean in_b)
	{
	return in_a ^ in_b;
	}
/*...e*/

void PrimShapeSdiff::intersect(const Xyz & p, const Xyz & q, IsectList *ils[]) const
	{
	if ( overlap )
		{
		r->intersect(p, q, ils+1);
		if ( ils[1]->is_empty() )
			l->intersect(p, q, ils);
		else if ( ils[1]->is_solid() )
			ils[0]->n_isects = 0;
		else
			{
			l->intersect(p, q, ils+2);
			if ( ils[2]->is_empty() )
				swap_isectls(ils[0], ils[1]);
			else
				combine_isectls(combine_sdiff, ils[2], ils[1], ils[0]);
			}
		}
	else
		// No possibility of overlap, treat as concatenation
		{
		r->intersect(p, q, ils+1);
		l->intersect(p, q, ils+2);
		combine_isectls(combine_sdiff, ils[2], ils[1], ils[0]);
		}
	}
/*...e*/
/*...e*/
/*...sPrimShapeExtent:2:*/
class PrimShapeExtent : public PrimShapeBinOp
	{
public:
	PrimShapeExtent(PrimShape *l, PrimShape *r);
	virtual ~PrimShapeExtent();
	virtual PrimShape *copy() const;
	virtual PrimShape *trans  (Xyz    t     ) const;
	virtual PrimShape *trans_x(double t     ) const;
	virtual PrimShape *trans_y(double t     ) const;
	virtual PrimShape *trans_z(double t     ) const;
	virtual PrimShape *rot_x  (double angle ) const;
	virtual PrimShape *rot_y  (double angle ) const;
	virtual PrimShape *rot_z  (double angle ) const;
	virtual PrimShape *scale  (Xyz    factor) const;
	virtual PrimShape *scale  (double factor) const;
	virtual PrimShape *scale_x(double factor) const;
	virtual PrimShape *scale_y(double factor) const;
	virtual PrimShape *scale_z(double factor) const;
	virtual PrimShape *resurf(Surf s) const;
	virtual Extent extent();
	virtual PrimShape *simplify(
		double pa, double pb, double pc, double pd) const;
	virtual int isects_reqd() const;
	virtual void intersect(const Xyz & p, const Xyz & q, IsectList *ils[]) const;
	};

/*...sPrimShapeExtent:2:*/
PrimShapeExtent::PrimShapeExtent(PrimShape *l, PrimShape *r) : PrimShapeBinOp(l, r) {}
/*...e*/
/*...s\126\PrimShapeExtent:2:*/
PrimShapeExtent::~PrimShapeExtent() {}
/*...e*/
/*...scopy:2:*/
PrimShape *PrimShapeExtent::copy() const { return new PrimShapeExtent(l->copy(), r->copy()); }
/*...e*/
/*...strans\39\s:2:*/
PrimShape *PrimShapeExtent::trans  (Xyz    t) const { return new PrimShapeExtent(l->trans  (t),r->trans  (t)); }
PrimShape *PrimShapeExtent::trans_x(double t) const { return new PrimShapeExtent(l->trans_x(t),r->trans_x(t)); }
PrimShape *PrimShapeExtent::trans_y(double t) const { return new PrimShapeExtent(l->trans_y(t),r->trans_y(t)); }
PrimShape *PrimShapeExtent::trans_z(double t) const { return new PrimShapeExtent(l->trans_z(t),r->trans_z(t)); }
/*...e*/
/*...srot\39\s:2:*/
PrimShape *PrimShapeExtent::rot_x  (double angle) const { return new PrimShapeExtent(l->rot_x(angle),r->rot_x(angle)); }
PrimShape *PrimShapeExtent::rot_y  (double angle) const { return new PrimShapeExtent(l->rot_y(angle),r->rot_y(angle)); }
PrimShape *PrimShapeExtent::rot_z  (double angle) const { return new PrimShapeExtent(l->rot_z(angle),r->rot_z(angle)); }
/*...e*/
/*...sscale\39\s:2:*/
PrimShape *PrimShapeExtent::scale  (Xyz    factor) const { return new PrimShapeExtent(l->scale  (factor),r->scale  (factor)); }
PrimShape *PrimShapeExtent::scale  (double factor) const { return new PrimShapeExtent(l->scale  (factor),r->scale  (factor)); }
PrimShape *PrimShapeExtent::scale_x(double factor) const { return new PrimShapeExtent(l->scale_x(factor),r->scale_x(factor)); }
PrimShape *PrimShapeExtent::scale_y(double factor) const { return new PrimShapeExtent(l->scale_y(factor),r->scale_y(factor)); }
PrimShape *PrimShapeExtent::scale_z(double factor) const { return new PrimShapeExtent(l->scale_z(factor),r->scale_z(factor)); }
/*...e*/
/*...sresurf:2:*/
PrimShape *PrimShapeExtent::resurf(Surf surf) const { return new PrimShapeExtent(l->resurf(surf),r->resurf(surf)); }
/*...e*/
/*...sextent:2:*/
Extent PrimShapeExtent::extent()
	{
	l->extent(); // Performed to cause side effect of labelling lhs
	return r->extent();
	}
/*...e*/
/*...ssimplify:2:*/
// If simple enclosing shape is on the wrong side, then result is a void
// else process complicated shape

PrimShape *PrimShapeExtent::simplify(double pa, double pb, double pc, double pd) const
	{
	PrimShape *sr = simpex( r->simplify(pa,pb,pc,pd), pa,pb,pc,pd);
	if ( sr == 0 )
		return 0;
	PrimShape *sl = simpex( l->simplify(pa,pb,pc,pd), pa,pb,pc,pd);
	if ( sl == 0 )
		// Can get here if extent has part of itself on the right side
		// of the plane, and yet the actual shape does not
		{
		delete sr;
		return 0;
		}
	return new PrimShapeExtent(sl, sr);
	}
/*...e*/
/*...sisects_reqd:2:*/
int PrimShapeExtent::isects_reqd() const
	{
	int li = l->isects_reqd(), ri = r->isects_reqd();
	return Max(li, ri);
	}
/*...e*/
/*...sintersect:2:*/
void PrimShapeExtent::intersect(const Xyz & p, const Xyz & q, IsectList *ils[]) const
	{
	r->intersect(p, q, ils);
	if ( ! ( ils[0]->is_empty() ) )
		l->intersect(p, q, ils);
	}
/*...e*/
/*...e*/

/*...sShape:0:*/
/*...sconstruction \47\ destruction etc\46\:0:*/
Shape::Shape() { p = new ShapeRefCnt(new PrimShapeEmpty()); }

Shape::Shape(PrimShape *prim) { p = new ShapeRefCnt(prim); }

Shape::Shape(const Shape & shape) { p = shape.p; p->n++; }

Shape & Shape::operator=(const Shape & shape)
	{
	++shape.p->n;
	if ( --(p->n) == 0 )
		delete p;
	p = shape.p;
	return *this;
	}

Shape::~Shape()
	{
	if ( --(p->n) == 0 )
		delete p;
	}
/*...e*/
/*...smethods used to construct shapes:0:*/
// Making half-planes

Shape Shape::plane(
	double a, double b, double c, double d,
	Surf surf
	)
	{ return Shape(new PrimShapePlane(a,b,c,d, surf)); }

Shape Shape::plane(const Xyz & p, const Xyz & n, Surf surf)
	{ return plane(n.x, n.y, n.z, -(n.x*p.x+n.y*p.y+n.z*p.z), surf); }

Shape Shape::x_lt(double x, Surf surf)
	{ return Shape(new PrimShapePlane( 1.0,0.0,0.0,-x, surf)); }

Shape Shape::x_gt(double x, Surf surf)
	{ return Shape(new PrimShapePlane(-1.0,0.0,0.0, x, surf)); }

Shape Shape::y_lt(double y, Surf surf)
	{ return Shape(new PrimShapePlane(0.0, 1.0,0.0,-y, surf)); }

Shape Shape::y_gt(double y, Surf surf)
	{ return Shape(new PrimShapePlane(0.0,-1.0,0.0, y, surf)); }

Shape Shape::z_lt(double z, Surf surf)
	{ return Shape(new PrimShapePlane(0.0,0.0, 1.0,-z, surf)); }

Shape Shape::z_gt(double z, Surf surf)
	{ return Shape(new PrimShapePlane(0.0,0.0,-1.0, z, surf)); }

// Making bi-planes

Shape Shape::biplane(
	double a, double b, double c, double d1, double d2,
	Surf surf
	)
	{ return Shape(new PrimShapeBiPlane(a,b,c,d1,d2, surf)); }

Shape Shape::x_in(double x1, double x2, Surf surf)
	{ return Shape(new PrimShapeBiPlane(1.0,0.0,0.0,-x1,-x2,surf)); }

Shape Shape::y_in(double y1, double y2, Surf surf)
	{ return Shape(new PrimShapeBiPlane(0.0,1.0,0.0,-y1,-y2,surf)); }

Shape Shape::z_in(double z1, double z2, Surf surf)
	{ return Shape(new PrimShapeBiPlane(0.0,0.0,1.0,-z1,-z2,surf)); }

// Making quadratics

Shape Shape::quad(
	double a, double b, double c,
	double d, double e, double f,
	double g, double h, double i,
	double j, Surf surf
	)
	{ return Shape(new PrimShapeQuad(a,b,c,d,e,f,g,h,i,j,surf)); }

Shape Shape::ellipsoid(double rx, double ry, double rz, Surf surf)
	{
	return Shape::quad(1.0/(rx*rx), 1.0/(ry*ry), 1.0/(rz*rz),
			   0.0, 0.0, 0.0,
			   0.0, 0.0, 0.0,
			   -1.0, surf);
	}

Shape Shape::x_ell_cyl(double ry, double rz, Surf surf)
	{
	return Shape::quad(0.0, 1.0 / (ry*ry), 1.0 / (rz*rz),
			   0.0, 0.0, 0.0,
			   0.0, 0.0, 0.0,
			   -1.0, surf);
	}

Shape Shape::y_ell_cyl(double rx, double rz, Surf surf)
	{
	return Shape::quad(1.0 / (rx*rx), 0.0, 1.0 / (rz*rz),
			   0.0, 0.0, 0.0,
			   0.0, 0.0, 0.0,
			   -1.0, surf);
	}

Shape Shape::z_ell_cyl(double rx, double ry, Surf surf)
	{
	return Shape::quad(1.0 / (rx*rx), 1.0 / (ry*ry), 0.0,
			   0.0, 0.0, 0.0,
			   0.0, 0.0, 0.0,
			   -1.0, surf);
	}

Shape Shape::x_cyl(double r, Surf surf) { return Shape::x_ell_cyl(r,r,surf); }
Shape Shape::y_cyl(double r, Surf surf) { return Shape::y_ell_cyl(r,r,surf); }
Shape Shape::z_cyl(double r, Surf surf) { return Shape::z_ell_cyl(r,r,surf); }

Shape Shape::x_ell_cone(double ky, double kz, Surf surf)
	{
	return Shape::quad(-1.0, ky*ky, kz*kz,
			   0.0, 0.0, 0.0,
			   0.0, 0.0, 0.0,
			   0.0, surf);
	}

Shape Shape::y_ell_cone(double kx, double kz, Surf surf)
	{
	return Shape::quad(kx*kx, -1.0, kz*kz,
			   0.0, 0.0, 0.0,
			   0.0, 0.0, 0.0,
			   0.0, surf);
	}

Shape Shape::z_ell_cone(double kx, double ky, Surf surf)
	{
	return Shape::quad(kx*kx, ky*ky, -1.0,
			   0.0, 0.0, 0.0,
			   0.0, 0.0, 0.0,
			   0.0, surf);
	}

Shape Shape::x_cone(double k, Surf surf) { return Shape::x_ell_cone(k,k,surf); }
Shape Shape::y_cone(double k, Surf surf) { return Shape::y_ell_cone(k,k,surf); }
Shape Shape::z_cone(double k, Surf surf) { return Shape::z_ell_cone(k,k,surf); }

// Making a sphere

Shape Shape::sphere(double r, Surf surf)
	{ return Shape(new PrimShapeSphere(r, surf)); }

Shape Shape::sphere(Xyz p, double r, Surf surf)
	{ return Shape(new PrimShapeSphere(p,r, surf)); }

// Making quartics

Shape Shape::quartic(double c[], Surf surf)
	{ return Shape(new PrimShapeQuart(c, surf)); }

// Making circular torii
// See: http://en.wikipedia.org/wiki/Torus

Shape Shape::x_torus(double r, double R, Surf surf)
	{
	double c[QCI::n_coeff];
	double s = R*R-r*r;
	c[QCI::XXXX] = 1.0;
	c[QCI::XXXY] = 0.0;
	c[QCI::XXXZ] = 0.0;
	c[QCI::XXX ] = 0.0;
	c[QCI::XXYY] = 2.0;
	c[QCI::XXYZ] = 0.0;
	c[QCI::XXY ] = 0.0;
	c[QCI::XXZZ] = 2.0;
	c[QCI::XXZ ] = 0.0;
	c[QCI::XX  ] = 2.0*s;
	c[QCI::XYYY] = 0.0;
	c[QCI::XYYZ] = 0.0;
	c[QCI::XYY ] = 0.0;
	c[QCI::XYZZ] = 0.0;
	c[QCI::XYZ ] = 0.0;
	c[QCI::XY  ] = 0.0;
	c[QCI::XZZZ] = 0.0;
	c[QCI::XZZ ] = 0.0;
	c[QCI::XZ  ] = 0.0;
	c[QCI::X   ] = 0.0;
	c[QCI::YYYY] = 1.0;
	c[QCI::YYYZ] = 0.0;
	c[QCI::YYY ] = 0.0;
	c[QCI::YYZZ] = 2.0;
	c[QCI::YYZ ] = 0.0;
	c[QCI::YY  ] = 2.0*s -4.0*R*R;
	c[QCI::YZZZ] = 0.0;
	c[QCI::YZZ ] = 0.0;
	c[QCI::YZ  ] = 0.0;
	c[QCI::Y   ] = 0.0;
	c[QCI::ZZZZ] = 1.0;
	c[QCI::ZZZ ] = 0.0;
	c[QCI::ZZ  ] = 2.0*s -4.0*R*R;
	c[QCI::Z   ] = 0.0;
	c[QCI::C   ] = s*s;
	return Shape(new PrimShapeQuart(c, surf));
	}

Shape Shape::y_torus(double r, double R, Surf surf)
	{
	double c[QCI::n_coeff];
	double s = R*R-r*r;
	c[QCI::XXXX] = 1.0;
	c[QCI::XXXY] = 0.0;
	c[QCI::XXXZ] = 0.0;
	c[QCI::XXX ] = 0.0;
	c[QCI::XXYY] = 2.0;
	c[QCI::XXYZ] = 0.0;
	c[QCI::XXY ] = 0.0;
	c[QCI::XXZZ] = 2.0;
	c[QCI::XXZ ] = 0.0;
	c[QCI::XX  ] = 2.0*s -4.0*R*R;
	c[QCI::XYYY] = 0.0;
	c[QCI::XYYZ] = 0.0;
	c[QCI::XYY ] = 0.0;
	c[QCI::XYZZ] = 0.0;
	c[QCI::XYZ ] = 0.0;
	c[QCI::XY  ] = 0.0;
	c[QCI::XZZZ] = 0.0;
	c[QCI::XZZ ] = 0.0;
	c[QCI::XZ  ] = 0.0;
	c[QCI::X   ] = 0.0;
	c[QCI::YYYY] = 1.0;
	c[QCI::YYYZ] = 0.0;
	c[QCI::YYY ] = 0.0;
	c[QCI::YYZZ] = 2.0;
	c[QCI::YYZ ] = 0.0;
	c[QCI::YY  ] = 2.0*s;
	c[QCI::YZZZ] = 0.0;
	c[QCI::YZZ ] = 0.0;
	c[QCI::YZ  ] = 0.0;
	c[QCI::Y   ] = 0.0;
	c[QCI::ZZZZ] = 1.0;
	c[QCI::ZZZ ] = 0.0;
	c[QCI::ZZ  ] = 2.0*s -4.0*R*R;
	c[QCI::Z   ] = 0.0;
	c[QCI::C   ] = s*s;
	return Shape(new PrimShapeQuart(c, surf));
	}

Shape Shape::z_torus(double r, double R, Surf surf)
	{
	double c[QCI::n_coeff];
	double s = R*R-r*r;
	c[QCI::XXXX] = 1.0;
	c[QCI::XXXY] = 0.0;
	c[QCI::XXXZ] = 0.0;
	c[QCI::XXX ] = 0.0;
	c[QCI::XXYY] = 2.0;
	c[QCI::XXYZ] = 0.0;
	c[QCI::XXY ] = 0.0;
	c[QCI::XXZZ] = 2.0;
	c[QCI::XXZ ] = 0.0;
	c[QCI::XX  ] = 2.0*s -4.0*R*R;
	c[QCI::XYYY] = 0.0;
	c[QCI::XYYZ] = 0.0;
	c[QCI::XYY ] = 0.0;
	c[QCI::XYZZ] = 0.0;
	c[QCI::XYZ ] = 0.0;
	c[QCI::XY  ] = 0.0;
	c[QCI::XZZZ] = 0.0;
	c[QCI::XZZ ] = 0.0;
	c[QCI::XZ  ] = 0.0;
	c[QCI::X   ] = 0.0;
	c[QCI::YYYY] = 1.0;
	c[QCI::YYYZ] = 0.0;
	c[QCI::YYY ] = 0.0;
	c[QCI::YYZZ] = 2.0;
	c[QCI::YYZ ] = 0.0;
	c[QCI::YY  ] = 2.0*s -4.0*R*R;
	c[QCI::YZZZ] = 0.0;
	c[QCI::YZZ ] = 0.0;
	c[QCI::YZ  ] = 0.0;
	c[QCI::Y   ] = 0.0;
	c[QCI::ZZZZ] = 1.0;
	c[QCI::ZZZ ] = 0.0;
	c[QCI::ZZ  ] = 2.0*s;
	c[QCI::Z   ] = 0.0;
	c[QCI::C   ] = s*s;
	return Shape(new PrimShapeQuart(c, surf));
	}
/*...e*/
/*...soperators used to construct compound shapes:0:*/
Shape operator|(const Shape & l, const Shape & r)
	{ return Shape(new PrimShapeUnion (((PrimShape *)l)->copy(),((PrimShape *)r)->copy())); }

Shape operator&(const Shape & l, const Shape & r)
	{ return Shape(new PrimShapeIsect (((PrimShape *)l)->copy(),((PrimShape *)r)->copy())); }

Shape operator-(const Shape & l, const Shape & r)
	{ return Shape(new PrimShapeDiff  (((PrimShape *)l)->copy(),((PrimShape *)r)->copy())); }

Shape operator^(const Shape & l, const Shape & r)
	{ return Shape(new PrimShapeSdiff (((PrimShape *)l)->copy(),((PrimShape *)r)->copy())); }

Shape operator<(const Shape & l, const Shape & r)
	{ return Shape(new PrimShapeExtent(((PrimShape *)l)->copy(),((PrimShape *)r)->copy())); }

Shape operator>(const Shape & l, const Shape & r) { return r < l; }

Shape operator|=(Shape & l, const Shape & r) { return l = (l | r); }
Shape operator&=(Shape & l, const Shape & r) { return l = (l & r); }
Shape operator-=(Shape & l, const Shape & r) { return l = (l - r); }
Shape operator^=(Shape & l, const Shape & r) { return l = (l ^ r); }
Shape operator<=(Shape & l, const Shape & r) { return l = (l < r); }
Shape operator>=(Shape & l, const Shape & r) { return l = (l > r); }
/*...e*/
/*...sgeneral manipulation:0:*/
Shape Shape::trans  (Xyz    t     ) const { return Shape(p->prim->trans  (t     )); }
Shape Shape::trans_x(double t     ) const { return Shape(p->prim->trans_x(t     )); }
Shape Shape::trans_y(double t     ) const { return Shape(p->prim->trans_y(t     )); }
Shape Shape::trans_z(double t     ) const { return Shape(p->prim->trans_z(t     )); }
Shape Shape::rot_x  (double angle ) const { return Shape(p->prim->rot_x  (angle )); }
Shape Shape::rot_y  (double angle ) const { return Shape(p->prim->rot_y  (angle )); }
Shape Shape::rot_z  (double angle ) const { return Shape(p->prim->rot_z  (angle )); }
Shape Shape::scale  (Xyz    factor) const { return Shape(p->prim->scale  (factor)); }
Shape Shape::scale  (double factor) const { return Shape(p->prim->scale  (factor)); }
Shape Shape::scale_x(double factor) const { return Shape(p->prim->scale_x(factor)); }
Shape Shape::scale_y(double factor) const { return Shape(p->prim->scale_y(factor)); }
Shape Shape::scale_z(double factor) const { return Shape(p->prim->scale_z(factor)); }
Shape Shape::resurf(Surf surf) const { return Shape(p->prim->resurf(surf)); }

/*...srot \45\ rotate about arbitrary axis through origin:0:*/
// Rotate about axis through origin, direction ad, by angle
// Determine what it takes to move the axis into the x-z plane
// Determine what it takes to move it onto the x axis
// Rotate about X axis
Shape Shape::rot(const Xyz & ad, double angle) const
	{
	double angle_z = atan2(ad.y, ad.x);
	Xyz ad2(ad.rot_z(-angle_z));
	double angle_y = atan2(ad2.z, ad2.x);
	return rot_z(-angle_z).rot_y(angle_y)
	      .rot_x(angle)
	      .rot_y(-angle_y).rot_z(angle_z);
	}
/*...e*/
/*...srot \45\ rotate about arbitrary axis:0:*/
// Rotate about axis through ao, direction ad, by angle
// Translate to origin, rotate, translate back
Shape Shape::rot(const Xyz & ao, const Xyz & ad, double angle) const
	{
	return trans(-ao).rot(ad, angle).trans(ao);
	}
/*...e*/
/*...e*/
/*...e*/
