//
// shape.h - Geometric Shapes
//

#ifndef SHAPE_H
#define	SHAPE_H

/*...sincludes:0:*/
#include "types.h"
#include "insane.h"
#include "xyz.h"
#include "rgb.h"
#include "col.h"
#include "surf.h"
#include "extent.h"
#include "roots.h"

/*...vtypes\46\h:0:*/
/*...vinsane\46\h:0:*/
/*...vxyz\46\h:0:*/
/*...vrgb\46\h:0:*/
/*...vcol\46\h:0:*/
/*...vsurf\46\h:0:*/
/*...vextent\46\h:0:*/
/*...vroots\46\h:0:*/
/*...e*/

/*...sIsect\44\ IsectList:0:*/
// A single point of intersection

class PrimShapeSurf;

class Isect
	{
public:
	double t;			// t value at point of intersection
	const PrimShapeSurf *shape;	// Shape intersection is with
	Boolean	entering;		// Are we entering the shape?
	Boolean	negate_normal;		// Is normal to be negated?
	};

class IsectList
	{
public:
	int n_isects;			// # of intersection points
	Isect *isects;			// Array of intersection points

	IsectList(int n_isects_max);
	~IsectList();

	Boolean is_empty() const
		{ return n_isects == 0; }
	Boolean is_solid() const
		{ return ( n_isects == 1 && isects[0].t == -LARGEST ) ||
			 ( n_isects == 2 && isects[0].t == -LARGEST
			 		 && isects[1].t ==  LARGEST ) ; }
	void t_after(double t = 0.0);
	};
/*...e*/

/*...sprivate PrimShape\44\ PrimShapeSurf:0:*/
// Abstract primitive shape class.
// You'd never actually have an instance of this.
// All raytracable shapes are derived from this.

class PrimShape
	{
public:
	PrimShape() {}
	virtual ~PrimShape() {}
	virtual PrimShape *copy() const = 0;
	virtual PrimShape *trans  (Xyz    t     ) const = 0;
	virtual PrimShape *trans_x(double t     ) const = 0;
	virtual PrimShape *trans_y(double t     ) const = 0;
	virtual PrimShape *trans_z(double t     ) const = 0;
	virtual PrimShape *rot_x  (double angle ) const = 0;
	virtual PrimShape *rot_y  (double angle ) const = 0;
	virtual PrimShape *rot_z  (double angle ) const = 0;
	virtual PrimShape *scale  (Xyz    factor) const = 0;
	virtual PrimShape *scale  (double factor) const = 0;
	virtual PrimShape *scale_x(double factor) const = 0;
	virtual PrimShape *scale_y(double factor) const = 0;
	virtual PrimShape *scale_z(double factor) const = 0;
	virtual PrimShape *resurf(Surf s) const = 0;
	virtual Xyz normal(const Xyz & p) const = 0;
	virtual Extent extent() = 0;
	virtual void preprocess() {};
		// preprocess is called before a raytrace, and tends to do
		// things like unitise vectors etc..
	virtual PrimShape *simplify(
		double pa, double pb, double pc, double pd) const
			{ pa=pa; pb=pb; pc=pc; pd=pd; return copy(); }
		// ensure preprocess called before using simplify.
		// simplify returns 0 if the entire shape does not satisfy
		// the plane equation pax+pby+pcz<=0, otherwise it returns
		// a copy of the shape (possibly simplified)
		// Note: assume (pa,pb,pc) is a unit vector
		// Hence sqrt(pa*pa+pb*pb+pc*pc) == 1.0
	virtual int isects_reqd() const = 0;
	virtual int isect_depth() const = 0;
	virtual void intersect(const Xyz & p, const Xyz & q, IsectList *ils[]) const = 0;
	};

// PrimShape that has a surface.
// Abstract base class.
// Because it inherits some pure virtual functions from PrimShape.
// You'd never have an instance of this.
// The surfaces which are referenced within the IsectList passed to the
// intersect function will all be derived from PrimShapeSurf.

class PrimShapeSurf : public PrimShape
	{
public:
	Surf surf;
	PrimShapeSurf(Surf surf) : surf(surf) {}
	virtual ~PrimShapeSurf() {}
	virtual Xyz normal(const Xyz & p) const { return Xyz(0.0,0.0,0.0); }
	virtual double evaluate(const Xyz & v) const { return 0.0; }
		// Any class derived from PrimShapeSurf which implements
		// the evaluate method may use the intersect_* functions
		// defined in this module.
	virtual int isect_depth() const { return 1; }
	};
/*...e*/

/*...senum QCI:0:*/
// Ordering of Quartic Coefficient Indices from POV-Ray
// See: http://www.povray.org/documentation/view/3.6.1/298/

class QCI
	{
public:
	enum QCIE
		{
		XXXX,
		XXXY,
		XXXZ,
		XXX,
		XXYY,
		XXYZ,
		XXY,
		XXZZ,
		XXZ,
		XX,
		XYYY,
		XYYZ,
		XYY,
		XYZZ,
		XYZ,
		XY,
		XZZZ,
		XZZ,
		XZ,
		X,
		YYYY,
		YYYZ,
		YYY,
		YYZZ,
		YYZ,
		YY,
		YZZZ,
		YZZ,
		YZ,
		Y,
		ZZZZ,
		ZZZ,
		ZZ,
		Z,
		C,
		n_coeff
		};
	};
/*...e*/

class Shape
	{
/*...sprivate:8:*/
// Because shapes will be copy constructed a lot, and also assigned a lot,
// and because Shape trees can be large, I have implemented reference counts.

class ShapeRefCnt
	{
public:
	PrimShape *prim;
	int n;
	ShapeRefCnt(PrimShape *prim) : n(1), prim(prim) {}
	~ShapeRefCnt() { delete prim; }
	};
ShapeRefCnt *p;	
/*...e*/
public:
	Shape();
	Shape(const Shape & shape);
	Shape & operator=(const Shape & shape);
	~Shape();

	// In the absence of a decent multiple constructor mechanism in C++
	// use these static member functions instead...

	static Shape plane(
		double a, double b, double c, double d,
		Surf surf
		);
	static Shape plane(const Xyz & p, const Xyz & n, Surf surf);
		// Construct plane passing through point p, with normal n
	static Shape x_lt(double x, Surf surf);
	static Shape x_gt(double x, Surf surf);
	static Shape y_lt(double y, Surf surf);
	static Shape y_gt(double y, Surf surf);
	static Shape z_lt(double z, Surf surf);
	static Shape z_gt(double z, Surf surf);

	static Shape biplane(
		double a, double b, double c, double d1, double d2,
		Surf surf
		);
	static Shape x_in(double x1, double x2, Surf surf);
	static Shape y_in(double y1, double y2, Surf surf);
	static Shape z_in(double z1, double z2, Surf surf);

	static Shape quad(
		double a, double b, double c,
		double d, double e, double f,
		double g, double h, double i,
		double j,
		Surf surf
		);
	static Shape ellipsoid(double rx, double ry, double rz, Surf surf);
	static Shape x_ell_cyl(double ry, double rz, Surf surf);
	static Shape y_ell_cyl(double rx, double rz, Surf surf);
	static Shape z_ell_cyl(double rx, double ry, Surf surf);
	static Shape x_cyl(double r, Surf surf);
	static Shape y_cyl(double r, Surf surf);
	static Shape z_cyl(double r, Surf surf);
	static Shape x_ell_cone(double ky, double kz, Surf surf);
	static Shape y_ell_cone(double kx, double kz, Surf surf);
	static Shape z_ell_cone(double kx, double ky, Surf surf);
	static Shape x_cone(double k, Surf surf);
	static Shape y_cone(double k, Surf surf);
	static Shape z_cone(double k, Surf surf);

	static Shape sphere(double r, Surf surf);
	static Shape sphere(Xyz p, double r, Surf surf);

	static Shape quartic(double c[], Surf surf);
	static Shape x_torus(double r, double R, Surf surf);
	static Shape y_torus(double r, double R, Surf surf);
	static Shape z_torus(double r, double R, Surf surf);

	// To allow manipulation of shapes

	Shape trans  (Xyz t) const;
	Shape trans_x(double t) const;
	Shape trans_y(double t) const;
	Shape trans_z(double t) const;
	Shape rot_x(double angle) const;
	Shape rot_y(double angle) const;
	Shape rot_z(double angle) const;
	Shape scale  (Xyz factor) const;
	Shape scale  (double factor) const;
	Shape scale_x(double factor) const;
	Shape scale_y(double factor) const;
	Shape scale_z(double factor) const;
	Shape resurf(Surf surf) const;

	Shape rot(const Xyz & ad, double angle) const;
		// Rotate about axis through origin, direction ad, by angle
	Shape rot(const Xyz & ao, const Xyz & ad, double angle) const;
		// Rotate about axis through ao, direction ad, by angle

	// Internal use only

	Shape(PrimShape *prim);
	operator PrimShape *() const { return p->prim; };
	};

// CSG union, isect, diff, sdiff, and extent operators
Shape operator|(const Shape & l, const Shape & r);
Shape operator&(const Shape & l, const Shape & r);
Shape operator-(const Shape & l, const Shape & r);
Shape operator^(const Shape & l, const Shape & r);
Shape operator<(const Shape & l, const Shape & r);
Shape operator>(const Shape & l, const Shape & r);

Shape operator|=(Shape & l, const Shape & r);
Shape operator&=(Shape & l, const Shape & r);
Shape operator-=(Shape & l, const Shape & r);
Shape operator^=(Shape & l, const Shape & r);
Shape operator<=(Shape & l, const Shape & r);
Shape operator>=(Shape & l, const Shape & r);

#endif
