//
// texture.h - 3D Solid Texture Block
//
// I go to special lengths to handle the fact that solid textures may consume
// large amounts of memory (often megabytes).
//
// Each PrimTexture includes a reference count. Therefore, passing Textures by
// value does not cause multiple copies of the bitmap data to exist.
//
// All PrimTextures are kept in a list, and remember the filename they
// originated from. Then it becomes possible to detect that we are loading a
// texture that we have previously loaded, and simply re-reference it.
// This 'check_cache' feature may be disabled. eg:
//
//	Texture t1("backdrop.tex");		// Loads bitmap
//	Texture t2("backdrop.tex");		// Points to t1 internally
//	Texture t3("backdrop.tex", FALSE);	// Does load another copy
//

#ifndef TEXTURE_H
#define	TEXTURE_H

/*...sincludes:0:*/
#include "types.h"
#include "rgb.h"

/*...vtypes\46\h:0:*/
/*...vrgb\46\h:0:*/
/*...e*/

/*...sprivate PrimTexture:0:*/
class PrimTexture
	{
	int w, h, d, bpv, stride;
	typedef struct { Byte b, g, r, dummy; } PAL;
	PAL pal[0x100];
	Byte *data;
public:

	static PrimTexture *head;	// Head of list of PrimTextures
	PrimTexture *next;		// Next PrimTexture in list
	char fn_origin[500+1];		// Filename constructed from or ""
	int n;				// Reference count (initially 1)

	PrimTexture(int w, int h, int d, int bpv = 24);
	PrimTexture(const char *fn);
	~PrimTexture();
	Boolean write(const char *fn) const;
	int width () const { return w; }
	int height() const { return h; }
	int depth () const { return d; }
	Rgb get_voxel(int x, int y, int z) const;
	void set_voxel(int x, int y, int z, const Rgb & rgb); // if bpv==24
	void set_voxel(int x, int y, int z, int index);	// if bpv!=24
	void set_pal(int index, const Rgb & rgb);	// if bpv!=24
	};
/*...e*/

class Texture
	{
/*...sprivate:8:*/
PrimTexture *p;
Texture(PrimTexture *p);
/*...e*/
public:
	Texture();
	Texture(int w, int h, int d, int bpv = 24);
	Texture(const char *fn, Boolean check_cache = TRUE);
	Texture(const Texture & t);
	Texture & operator=(const Texture & t);
	~Texture();
	Boolean write(const char *fn) const { return p->write(fn); }
	int width () const { return p->width (); }
	int height() const { return p->height(); }
	int depth () const { return p->depth (); }
	Rgb get_voxel(int x, int y, int z) const
		{ return p->get_voxel(x,y,z); }
	void set_voxel(int x, int y, int z, const Rgb & rgb) // Use if bpv==24
		{ p->set_voxel(x,y,z,rgb); }
	void set_voxel(int x, int y, int z, int index)	// Use if bpv!=24
		{ p->set_voxel(x,y,z,index); }
	void set_pal(int index, const Rgb & rgb)	// Use if bpv!=24
		{ p->set_pal(index, rgb); }
	};

#endif
