//
// viewwin.C - Viewer for a bitmap using a Windows window
//

/*...sincludes:0:*/
#ifdef NO_CINCLUDES
  #include <stdio.h>
  #include <stdlib.h>
  #include <string.h>
  #include <time.h>
#else
  #include <cstdio>
  #include <cstdlib>
  #include <cstring>
  #include <ctime>
#endif
#include <process.h>

#define	BOOLEAN BOOLEANx
#define	WIN32_LEAN_AND_MEAN
#include <windows.h>
#undef BOOLEAN

extern "C" {
#include "gbm.h"
#include "gbmerr.h"
}

#include "bitmap.h"
#include "viewwin.h"

/*...vbitmap\46\h:0:*/
/*...vviewwin\46\h:0:*/
/*...e*/

static HINSTANCE hinst;
static int cxScreen, cyScreen, cyCaption, cxFixedFrame, cyFixedFrame;
static int iBppErrDiff;
static GBMRGB gbmrgbErrDiff[0x100];

/*...sclass ViewerWinInternal:0:*/
#define	WC_VIEWWIN "ViewerWindow"

#define	N_CHARS 10

class ViewerWinInternal
	{
public:
	HANDLE hmutex;        // Control access to an instance of this class
	HANDLE heventCreated; // Signalled when window initialised etc.
	HANDLE heventUpdated; // Signalled when update window done
	HANDLE heventDeleted; // Signalled when window destroyed
	HANDLE hsemChars;
	ViewerWin::WinFlags wf;

	HWND hwnd;
	BITMAPINFOHEADER bmih;
	RGBQUAD rgbqPalette[0x100];
	HBITMAP hbitmap;
	HPALETTE hpalette;
	char wtitle[200+1];
	Boolean window_closed;

	int nChars;
	char acChars[N_CHARS];

	GBM *gbm; GBMRGB *gbmrgb; Byte *data, *data_d;
	int ncols;

	ViewerWinInternal(Bitmap & b, int ncols, const char *title, ViewerWin::WinFlags wf);
	~ViewerWinInternal();

	void redraw();
	void refresh();
	int keypress();
	int wait_keypress();
	};

/*...sViewerWinWindowProc:0:*/
#define	WM_UPDATEWINDOW  WM_USER
#define	WM_DELETEWINDOW (WM_USER+1)

/*...sMakePalette:0:*/
static HPALETTE MakePalette(GBMRGB *gbmrgb, int ncols)
	{
	LOGPALETTE *pal = (LOGPALETTE *) malloc(sizeof(LOGPALETTE)+ncols*sizeof(PALETTEENTRY));
	pal->palNumEntries = ncols;
	pal->palVersion = 0x300;
	for ( int i = 0; i < ncols; i++ )
		{
		pal->palPalEntry[i].peRed   = (BYTE) gbmrgb[i].r;
		pal->palPalEntry[i].peGreen = (BYTE) gbmrgb[i].g;
		pal->palPalEntry[i].peBlue  = (BYTE) gbmrgb[i].b;
		pal->palPalEntry[i].peFlags = (BYTE) 0;
		}
	HPALETTE hpal = CreatePalette(pal);
	free(pal);
	return hpal;
	}
/*...e*/
/*...sRealizeOurPalette:0:*/
static void RealizeOurPalette(HWND hwnd, HPALETTE hpalette, BOOL bForceBackground)
	{
	HDC hdc = GetWindowDC(hwnd);
	HPALETTE hpaletteOld = SelectPalette(hdc, hpalette, FALSE);
	UINT nChanged = RealizePalette(hdc);
	SelectPalette(hdc, hpaletteOld, TRUE);
	ReleaseDC(hwnd, hdc);
	if ( nChanged != 0 )
		// Palette changes occured, we should repaint all
		// so as to get the best rendition to the new palette
		InvalidateRect(hwnd, NULL, TRUE);
	}
/*...e*/

// was FAR PASCAL
LRESULT CALLBACK ViewerWinWindowProc(
	HWND hwnd,
	unsigned message,
	WPARAM wParam,
	LPARAM lParam
	)
	{
	switch ( message )
		{
/*...sWM_CREATE          \45\ create:16:*/
case WM_CREATE:
	{
	CREATESTRUCT *cs = (CREATESTRUCT *) lParam;
	ViewerWinInternal *vwi = (ViewerWinInternal *) cs->lpCreateParams;
	SetWindowLongPtr(hwnd, 0, (LONG_PTR) vwi);
	memset(&(vwi->bmih), 0, sizeof(vwi->bmih));
	vwi->bmih.biSize        = sizeof(vwi->bmih);
	vwi->bmih.biWidth       = vwi->gbm->w;
	vwi->bmih.biHeight      = vwi->gbm->h;
	vwi->bmih.biPlanes      = 1;
	vwi->bmih.biCompression = BI_RGB;
	if ( vwi->gbm->bpp == 24 )
		{
		switch ( iBppErrDiff )
			{
			case 8:
				{
				for ( int i = 0; i < 7*8*4; i++ )
					{
					vwi->rgbqPalette[i].rgbRed      = (BYTE) gbmrgbErrDiff[i].r;
					vwi->rgbqPalette[i].rgbGreen    = (BYTE) gbmrgbErrDiff[i].g;
					vwi->rgbqPalette[i].rgbBlue     = (BYTE) gbmrgbErrDiff[i].b;
					vwi->rgbqPalette[i].rgbReserved = 0;
					}
				vwi->hpalette = MakePalette(gbmrgbErrDiff, 7*8*4);
				}
				break;
			default:
				vwi->hpalette = (HPALETTE) NULL;
				break;
			}
		vwi->bmih.biBitCount = iBppErrDiff;
		}
	else
		{
		for ( int i = 0; i < vwi->ncols; i++ )
			{
			vwi->rgbqPalette[i].rgbRed      = (BYTE) vwi->gbmrgb[i].r;
			vwi->rgbqPalette[i].rgbGreen    = (BYTE) vwi->gbmrgb[i].g;
			vwi->rgbqPalette[i].rgbBlue     = (BYTE) vwi->gbmrgb[i].b;
			vwi->rgbqPalette[i].rgbReserved = 0;
			}
		vwi->hpalette = MakePalette(vwi->gbmrgb, vwi->ncols);
		vwi->bmih.biBitCount = vwi->gbm->bpp;
		}
	HDC hdcScreen = GetWindowDC((HWND) NULL);
	vwi->hbitmap = CreateDIBSection(
		hdcScreen,
		(BITMAPINFO *) &(vwi->bmih),
		DIB_RGB_COLORS,
		NULL,
		(HANDLE) NULL,
		0);
	ReleaseDC((HWND) NULL, hdcScreen);
	}
	return 0;
/*...e*/
/*...sWM_PAINT           \45\ paint client:16:*/
case WM_PAINT:
	{
	PAINTSTRUCT ps;
	HDC hdc = BeginPaint(hwnd, &ps);

	RECT rcClient;
	GetClientRect(hwnd, &rcClient);
	int cxClient = rcClient.right  - rcClient.left;
	int cyClient = rcClient.bottom - rcClient.top ;

	ViewerWinInternal *vwi = (ViewerWinInternal *) GetWindowLongPtr(hwnd, 0);

	WaitForSingleObject(vwi->hmutex, INFINITE);

	HDC hdcCompatible = CreateCompatibleDC(hdc);
	HBITMAP hbitmapOld = (HBITMAP) SelectObject(hdcCompatible, vwi->hbitmap);

	HPALETTE hpaletteOld;
	if ( vwi->hpalette != (HPALETTE) NULL )
		hpaletteOld = SelectPalette(hdc, vwi->hpalette, FALSE);

	vwi->bmih.biHeight = vwi->gbm->h;
	SetDIBits(hdcCompatible, vwi->hbitmap,
		0, vwi->gbm->h, vwi->data_d,
		(CONST BITMAPINFO *) &(vwi->bmih), DIB_RGB_COLORS);

	BitBlt(hdc, 0, 0, vwi->gbm->w, vwi->gbm->h,
		hdcCompatible, 0, 0, SRCCOPY);

	if ( vwi->hpalette != (HPALETTE) NULL )
		SelectPalette(hdc, hpaletteOld, TRUE);

	SelectObject(hdcCompatible, hbitmapOld);
	DeleteDC(hdcCompatible);

	ReleaseMutex(vwi->hmutex);

	EndPaint(hwnd, &ps);
	}
	return 0;
/*...e*/
/*...sWM_QUERYNEWPALETTE \45\ got focus\44\ set our palette:16:*/
case WM_QUERYNEWPALETTE:
	// We've got input focus, set our choice of palette
	{
	ViewerWinInternal *vwi = (ViewerWinInternal *) GetWindowLongPtr(hwnd, 0);
	if ( vwi->hpalette != (HPALETTE) NULL )
		RealizeOurPalette(hwnd, vwi->hpalette, FALSE);
	}
	return 0;
/*...e*/
/*...sWM_PALETTECHANGED  \45\ other changed palette:16:*/
case WM_PALETTECHANGED:
	if ( (HWND) wParam != hwnd )
		// Someone else got palette
		// Aquire what entries we can
		{
		ViewerWinInternal *vwi = (ViewerWinInternal *) GetWindowLongPtr(hwnd, 0);
		if ( vwi->hpalette != (HPALETTE) NULL )
			RealizeOurPalette(hwnd, vwi->hpalette, FALSE);
		}
	return 0;
/*...e*/
/*...sWM_CLOSE           \45\ close:16:*/
case WM_CLOSE:
	{
	ViewerWinInternal *vwi = (ViewerWinInternal *) GetWindowLongPtr(hwnd, 0);
	WaitForSingleObject(vwi->hmutex, INFINITE);
	vwi->window_closed = TRUE;
	DestroyWindow(hwnd);
	ReleaseMutex(vwi->hmutex);
	}
	return 0;
/*...e*/
/*...sWM_DESTROY         \45\ destroy:16:*/
case WM_DESTROY:
	{
	ViewerWinInternal *vwi = (ViewerWinInternal *) GetWindowLongPtr(hwnd, 0);
	WaitForSingleObject(vwi->hmutex, INFINITE);
	vwi->hwnd = (HWND) NULL;
	if ( vwi->hpalette != (HPALETTE) NULL )
		DeleteObject(vwi->hpalette);
	DeleteObject(vwi->hbitmap);
	ReleaseMutex(vwi->hmutex);
	LONG lPrevReleaseCount;
	ReleaseSemaphore(vwi->hsemChars, 1000, &lPrevReleaseCount);
	SetEvent(vwi->heventUpdated);
	SetEvent(vwi->heventDeleted);
	PostQuitMessage(0);
	}
	return 0;
/*...e*/
/*...sWM_CHAR            \45\ capture keys typed:16:*/
case WM_CHAR:
	if ( wParam < 0x100 )
		{
		ViewerWinInternal *vwi = (ViewerWinInternal *) GetWindowLongPtr(hwnd, 0);
		WaitForSingleObject(vwi->hmutex, INFINITE);
		if ( vwi->nChars < N_CHARS )
			{
			vwi->acChars[vwi->nChars++] = (char) wParam;
			LONG lPrevReleaseCount;
			ReleaseSemaphore(vwi->hsemChars, 1, &lPrevReleaseCount);
			}
		ReleaseMutex(vwi->hmutex);
		return 0L;
		}
	break;
/*...e*/
/*...sWM_UPDATEWINDOW    \45\ update window:16:*/
case WM_UPDATEWINDOW:
	{
	ViewerWinInternal *vwi = (ViewerWinInternal *) GetWindowLongPtr(hwnd, 0);
	UpdateWindow(hwnd);
	SetEvent(vwi->heventUpdated);
	}
	return 0;
/*...e*/
/*...sWM_DELETEWINDOW    \45\ delete window:16:*/
case WM_DELETEWINDOW:
	DestroyWindow(hwnd);
	return 0;
/*...e*/
		}
	return DefWindowProc(hwnd, message, wParam, lParam);
	}
/*...e*/
/*...sUiThread:0:*/
// The main thread of the program isn't necessarily a windows thread.
// So we create a second thread to create and watch the bitmap viewer window.

static void _cdecl UiThread(void *p)
	{
	ViewerWinInternal *vwi = (ViewerWinInternal *) p;
	int x, y;
	if ( vwi->wf & ViewerWin::centralized )
		{
		x = (cxScreen - vwi->gbm->w)/2 - cxFixedFrame;
		y = (cyScreen - vwi->gbm->h)/2 - cyFixedFrame - cyCaption;
		// Now ensure user can easily get to system menu
		if ( x < 0 || y < 0 ) x = y = 10;
		}
	else
		x = y = CW_USEDEFAULT;
	vwi->hwnd = CreateWindow(
		WC_VIEWWIN,		// Window class name
		vwi->wtitle,		// Window title
		WS_BORDER|WS_SYSMENU|WS_MINIMIZEBOX|WS_OVERLAPPED,
		x,
		y,
		cxFixedFrame + vwi->gbm->w + cxFixedFrame,
		cyFixedFrame + cyCaption + vwi->gbm->h + cyFixedFrame,
		NULL,			// No parent for this window
		NULL,			// Use the class menu (not our own)
		hinst,			// Who created the window
		vwi);			// Parameters to pass on
	ShowWindow(vwi->hwnd,
		( vwi->wf & ViewerWin::minimized ) != 0
			? SW_SHOWMINIMIZED
			: SW_SHOWNORMAL);
	SetWindowPos(vwi->hwnd, HWND_TOP, 0, 0, 0, 0,
		SWP_NOMOVE|SWP_NOSIZE|SWP_SHOWWINDOW);
	SetEvent(vwi->heventCreated);
	MSG msg;
	while ( GetMessage(&msg, NULL, 0, 0) )
		{
		TranslateMessage(&msg); // So that WM_CHAR works!
		DispatchMessage(&msg);
		}
	}
/*...e*/

/*...sViewerWinInternal:0:*/
ViewerWinInternal::ViewerWinInternal(
	Bitmap & b, int ncols, const char *title, ViewerWin::WinFlags wf
	)
	:
	ncols(ncols), wf(wf)
	{
	// Mutex used to ensure either main tracing thread or windows
	// thread can only access vwi at any one time.
	hmutex = CreateMutex(NULL, FALSE, NULL);

	// Event which gets signalled when initialisation completes
	heventCreated = CreateEvent(NULL, TRUE, FALSE, NULL);

	// Event which gets signalled when update window occurs
	heventUpdated = CreateEvent(NULL, TRUE, FALSE, NULL);

	// Event which gets signalled when window destroyed
	heventDeleted = CreateEvent(NULL, TRUE, FALSE, NULL);

	b.get_gbm_internals(gbm, gbmrgb, data);

	if ( gbm->bpp == 24 )
		switch ( iBppErrDiff )
			{
			case 8:
				data_d = new Byte[((gbm->w+3)&~3)*gbm->h];
				ncols = 7*8*4;
				break;
			case 4:
				data_d = new Byte[((gbm->w*4+31)/32*4)*gbm->h];
				ncols = 16;
				break;
			case 24:
				data_d = data;
				ncols = 0;
				break;
			}
	else
		// Palettised bitmap to display,
		// so simply use it as is, and realise palette
		data_d = data;

	redraw();

	if ( title != 0 )
		strcpy(wtitle, title);
	else
		strcpy(wtitle, "Bitmap Viewer");
	window_closed = FALSE;

	nChars = 0;

	// Semaphore which gets 'incremented' whenever a key is queued
	// or by 1000 when the window is destroyed.
	hsemChars = CreateSemaphore(NULL, 0, N_CHARS+1000, NULL);

	_beginthread(UiThread, 0x10000, this);
	WaitForSingleObject(heventCreated, INFINITE);
	}
/*...e*/
/*...s\126\ViewerWinInternal:0:*/
ViewerWinInternal::~ViewerWinInternal()
	{
	WaitForSingleObject(hmutex, INFINITE);
	if ( hwnd != (HWND) NULL )
		PostMessage(hwnd, WM_DELETEWINDOW, 0, 0);
	ReleaseMutex(hmutex);
	WaitForSingleObject(heventDeleted, INFINITE);
	if ( data_d != data )
		delete[] data_d;
	}
/*...e*/
/*...sredraw:0:*/
void ViewerWinInternal::redraw()
	{
	if ( gbm->bpp == 24 )
		switch ( iBppErrDiff )
			{
			case 8:
				gbm_errdiff_7R8G4B(gbm, data, data_d);
				break;
			case 4:
				gbm_errdiff_VGA(gbm, data, data_d);
				break;
			default:
				break;
			}
	}
/*...e*/
/*...srefresh:0:*/
void ViewerWinInternal::refresh()
	{
	WaitForSingleObject(hmutex, INFINITE);
	if ( hwnd != (HWND) NULL )
		{
		redraw();
		ResetEvent(heventUpdated);
		InvalidateRect(hwnd, NULL, TRUE);
		PostMessage(hwnd, WM_UPDATEWINDOW, 0, 0); // delayed redraw
		ReleaseMutex(hmutex);
		WaitForSingleObject(heventUpdated, INFINITE);
		}
	else
		ReleaseMutex(hmutex);
	}
/*...e*/
/*...skeypress:0:*/
int ViewerWinInternal::keypress()
	{
	int k;
	if ( WaitForSingleObject(hsemChars, 0) == 0 )
		// Characters have been placed in array
		{
		WaitForSingleObject(hmutex, INFINITE);
		if ( window_closed )
			k = -1;
		else if ( nChars > 0 )
			{
			k = (int) (unsigned int) (unsigned char) acChars[0];
			memcpy(acChars, acChars+1, --nChars);
			}
		else
			k = 0;
		ReleaseMutex(hmutex);
		}
	else
		// No characters available
		{
		WaitForSingleObject(hmutex, INFINITE);
		if ( window_closed )
			k = -1;
		else
			k = 0;
		ReleaseMutex(hmutex);
		}
	return k;
	}
/*...e*/
/*...swait_keypress:0:*/
int ViewerWinInternal::wait_keypress()
	{
	int k;
	WaitForSingleObject(hsemChars, INFINITE);
		// Wait until character available (or window destroyed)
	WaitForSingleObject(hmutex, INFINITE);
	if ( window_closed )
		k = -1;
	else if ( nChars > 0 )
		{
		k = (int) (unsigned int) (unsigned char) acChars[0];
		memcpy(acChars, acChars+1, --nChars);
		}
	else
		k = 0;
	ReleaseMutex(hmutex);
	return k;
	}
/*...e*/
/*...e*/
/*...sclass ViewerWin:0:*/
// Conceal internal details (needed by window class) from external users.

ViewerWin::ViewerWin(Bitmap & b, int ncols, const char *title, WinFlags wf)
	{ vwi = new ViewerWinInternal(b, ncols, title, wf); }
ViewerWin::~ViewerWin()
	{ delete vwi; }
void ViewerWin::refresh()
	{ vwi->refresh(); }
int ViewerWin::keypress()
	{ return vwi->keypress(); }
int ViewerWin::wait_keypress()
	{ return vwi->wait_keypress(); }
/*...e*/
/*...sclass ViewwinWindows:0:*/
// A mechanism to ensure global Windows initialisation is done prior to
// any bitmap viewer windows being created.

class ViewwinWindows
	{
public:
	ViewwinWindows();
	};

ViewwinWindows::ViewwinWindows()
	{
	hinst = (HINSTANCE) GetModuleHandle((LPCSTR) NULL);

	// Register the window class
	WNDCLASS wc;
	wc.lpszClassName = WC_VIEWWIN;
	wc.hInstance     = hinst;
	wc.lpfnWndProc   = ViewerWinWindowProc;
	wc.hCursor       = LoadCursor(NULL, IDC_ARROW);
	wc.hIcon         = (HICON) NULL;
	wc.lpszMenuName  = NULL;
	wc.hbrBackground = (HBRUSH) GetStockObject(NULL_BRUSH);
	wc.style         = CS_HREDRAW | CS_VREDRAW;
	wc.cbClsExtra    = 0;
	wc.cbWndExtra    = sizeof(void *);
	RegisterClass(&wc);

	// Learn some system metrics
	cxScreen     = GetSystemMetrics(SM_CXSCREEN);
	cyScreen     = GetSystemMetrics(SM_CYSCREEN);
	cyCaption    = GetSystemMetrics(SM_CYCAPTION);
	cxFixedFrame = GetSystemMetrics(SM_CXFIXEDFRAME);
	cyFixedFrame = GetSystemMetrics(SM_CYFIXEDFRAME);

	// Decide what we are going to error-diffuse down to (or not)
	HDC hdcScreen = GetWindowDC((HWND) NULL);
	LONG lRasterCaps = GetDeviceCaps( hdcScreen, RASTERCAPS );
	LONG lPlanes     = GetDeviceCaps( hdcScreen, PLANES     );
	LONG lBitCount   = GetDeviceCaps( hdcScreen, BITSPIXEL  );
	ReleaseDC((HWND) NULL, hdcScreen);
	if ( (lRasterCaps & RC_PALETTE) != 0 && lPlanes == 1 && lBitCount == 8 )
		{
		gbm_errdiff_pal_7R8G4B(gbmrgbErrDiff);
		iBppErrDiff = 8;
		}
	else if ( lPlanes == 1 && lBitCount == 4 )
		{
		gbm_errdiff_pal_VGA(gbmrgbErrDiff);
		iBppErrDiff = 4;
		}
	else
		iBppErrDiff = 24;
	}

static ViewwinWindows windows;
/*...e*/
