//
// wire.C - Support for Wire Frame models
//

/*...sincludes:0:*/
#include "wire.h"

/*...vwire\46\h:0:*/
/*...e*/

/*...sclass OpCol:0:*/
// This type of op specifies the drawing colour

class OpCol : public Op
	{
public:
	int index;
	OpCol(int index) : index(index) {}
	virtual ~OpCol() {}
	virtual Op *copy() const { return new OpCol(index); }
	virtual Op *trans  (Xyz t) const { t=t; return copy(); }
	virtual Op *trans_x(double t) const { t=t; return copy(); }
	virtual Op *trans_y(double t) const { t=t; return copy(); }
	virtual Op *trans_z(double t) const { t=t; return copy(); }
	virtual Op *rot_x(double angle) const { angle=angle; return copy(); }
	virtual Op *rot_y(double angle) const { angle=angle; return copy(); }
	virtual Op *rot_z(double angle) const { angle=angle; return copy(); }
	virtual Op *scale  (Xyz factor) const { factor=factor; return copy(); }
	virtual Op *scale  (double factor) const { factor=factor; return copy(); }
	virtual Op *scale_x(double factor) const { factor=factor; return copy(); }
	virtual Op *scale_y(double factor) const { factor=factor; return copy(); }
	virtual Op *scale_z(double factor) const { factor=factor; return copy(); }
	virtual Op *mult_matrix(const Matrix3 & m) const { return copy(); }
	virtual Op *mult_matrix(const Matrix4 & m) const { return copy(); }
	virtual int opcode() const { return 0; }
	};
/*...e*/
/*...sclass OpVertex:0:*/
class OpVertex : public Op
	{
public:
	Xyz xyz;
	OpVertex(const Xyz & xyz) : xyz(xyz) {}
	virtual ~OpVertex() {}
	virtual Op *copy() const { return new OpVertex(xyz); }
	virtual Op *trans  (Xyz t)    const { return new OpVertex(xyz+t); }
	virtual Op *trans_x(double t) const { return new OpVertex(Xyz(xyz.x+t,xyz.y,xyz.z)); }
	virtual Op *trans_y(double t) const { return new OpVertex(Xyz(xyz.x,xyz.y+t,xyz.z)); }
	virtual Op *trans_z(double t) const { return new OpVertex(Xyz(xyz.x,xyz.y,xyz.z+t)); }
	virtual Op *rot_x(double angle) const { return new OpVertex(xyz.rot_x(angle)); }
	virtual Op *rot_y(double angle) const { return new OpVertex(xyz.rot_y(angle)); }
	virtual Op *rot_z(double angle) const { return new OpVertex(xyz.rot_z(angle)); }
	virtual Op *scale  (Xyz factor)    const { return new OpVertex(scale_by(xyz,factor)); }
	virtual Op *scale  (double factor) const { return new OpVertex(xyz*factor); }
	virtual Op *scale_x(double t) const { return new OpVertex(Xyz(xyz.x*t,xyz.y,xyz.z)); }
	virtual Op *scale_y(double t) const { return new OpVertex(Xyz(xyz.x,xyz.y*t,xyz.z)); }
	virtual Op *scale_z(double t) const { return new OpVertex(Xyz(xyz.x,xyz.y,xyz.z*t)); }
	virtual Op *mult_matrix(const Matrix3 & m) const
		{ return new OpVertex(m * xyz); }
	virtual Op *mult_matrix(const Matrix4 & m) const
		{ return new OpVertex(m * xyz); }
	virtual int opcode() const { return 1; }
	};
/*...e*/
/*...sclass OpEdge:0:*/
class OpEdge : public Op
	{
public:
	int vertex1, vertex2;
	OpEdge(int vertex1, int vertex2)
		: vertex1(vertex1), vertex2(vertex2) {}
	virtual ~OpEdge() {}
	virtual Op *copy() const { return new OpEdge(vertex1, vertex2); }
	virtual Op *trans  (Xyz t) const { t=t; return copy(); }
	virtual Op *trans_x(double t) const { t=t; return copy(); }
	virtual Op *trans_y(double t) const { t=t; return copy(); }
	virtual Op *trans_z(double t) const { t=t; return copy(); }
	virtual Op *rot_x(double angle) const { angle=angle; return copy(); }
	virtual Op *rot_y(double angle) const { angle=angle; return copy(); }
	virtual Op *rot_z(double angle) const { angle=angle; return copy(); }
	virtual Op *scale  (Xyz factor) const { factor=factor; return copy(); }
	virtual Op *scale  (double factor) const { factor=factor; return copy(); }
	virtual Op *scale_x(double factor) const { factor=factor; return copy(); }
	virtual Op *scale_y(double factor) const { factor=factor; return copy(); }
	virtual Op *scale_z(double factor) const { factor=factor; return copy(); }
	virtual Op *mult_matrix(const Matrix3 & m) const { return copy(); }
	virtual Op *mult_matrix(const Matrix4 & m) const { return copy(); }
	virtual int opcode() const { return 2; }
	};
/*...e*/

/*...sWireOpList:0:*/
WireOpList::WireOpList() {}
/*...e*/
/*...sWireOpList \40\copy\41\:0:*/
WireOpList::WireOpList(const WireOpList & ol)
	{
	append(ol);
	}
/*...e*/
/*...soperator\61\:0:*/
WireOpList & WireOpList::operator=(const WireOpList & ol)
	{
	if ( &ol != this )
		{
		empty();
		append(ol);
		}
	return *this;
	}
/*...e*/
/*...soperator\43\\61\:0:*/
WireOpList & WireOpList::operator+=(const WireOpList & ol)
	{
	append(ol);
	return *this;
	}
/*...e*/
/*...s\126\WireOpList:0:*/
WireOpList::~WireOpList() { empty(); }
/*...e*/

/*...scol:0:*/
WireOpList & WireOpList::col(int index)
	{
	Op *op = new OpCol(index);
	if ( op != 0 )
		append(op);
	return *this;
	}
/*...e*/
/*...svertex:0:*/
WireOpList & WireOpList::vertex(const Xyz & xyz)
	{
	Op *op = new OpVertex(xyz);
	if ( op != 0 )
		append(op);
	return *this;
	}
/*...e*/
/*...sedge:0:*/
WireOpList & WireOpList::edge(int vertex1, int vertex2)
	{
	Op *op = new OpEdge(vertex1, vertex2);
	if ( op != 0 )
		append(op);
	return *this;
	}
/*...e*/

/*...strans:0:*/
WireOpList WireOpList::trans(Xyz t) const
	{
	WireOpList ol;
	for ( Op *op = first; op != 0; op = op->next )
		{
		Op *op2 = op->trans(t);
		if ( op2 == 0 )
			break;
		ol.append(op2);
		}
	return ol;
	}
/*...e*/
/*...strans_x:0:*/
WireOpList WireOpList::trans_x(double t) const
	{
	WireOpList ol;
	for ( Op *op = first; op != 0; op = op->next )
		{
		Op *op2 = op->trans_x(t);
		if ( op2 == 0 )
			break;
		ol.append(op2);
		}
	return ol;
	}
/*...e*/
/*...strans_y:0:*/
WireOpList WireOpList::trans_y(double t) const
	{
	WireOpList ol;
	for ( Op *op = first; op != 0; op = op->next )
		{
		Op *op2 = op->trans_y(t);
		if ( op2 == 0 )
			break;
		ol.append(op2);
		}
	return ol;
	}
/*...e*/
/*...strans_z:0:*/
WireOpList WireOpList::trans_z(double t) const
	{
	WireOpList ol;
	for ( Op *op = first; op != 0; op = op->next )
		{
		Op *op2 = op->trans_z(t);
		if ( op2 == 0 )
			break;
		ol.append(op2);
		}
	return ol;
	}
/*...e*/
/*...srot_x:0:*/
WireOpList WireOpList::rot_x(double angle) const
	{
	WireOpList ol;
	for ( Op *op = first; op != 0; op = op->next )
		{
		Op *op2 = op->rot_x(angle);
		if ( op2 == 0 )
			break;
		ol.append(op2);
		}
	return ol;
	}
/*...e*/
/*...srot_y:0:*/
WireOpList WireOpList::rot_y(double angle) const
	{
	WireOpList ol;
	for ( Op *op = first; op != 0; op = op->next )
		{
		Op *op2 = op->rot_y(angle);
		if ( op2 == 0 )
			break;
		ol.append(op2);
		}
	return ol;
	}
/*...e*/
/*...srot_z:0:*/
WireOpList WireOpList::rot_z(double angle) const
	{
	WireOpList ol;
	for ( Op *op = first; op != 0; op = op->next )
		{
		Op *op2 = op->rot_z(angle);
		if ( op2 == 0 )
			break;
		ol.append(op2);
		}
	return ol;
	}
/*...e*/
/*...sscale:0:*/
WireOpList WireOpList::scale(Xyz factor) const
	{
	WireOpList ol;
	for ( Op *op = first; op != 0; op = op->next )
		{
		Op *op2 = op->scale(factor);
		if ( op2 == 0 )
			break;
		ol.append(op2);
		}
	return ol;
	}
/*...e*/
/*...sscale:0:*/
WireOpList WireOpList::scale(double factor) const
	{
	WireOpList ol;
	for ( Op *op = first; op != 0; op = op->next )
		{
		Op *op2 = op->scale(factor);
		if ( op2 == 0 )
			break;
		ol.append(op2);
		}
	return ol;
	}
/*...e*/
/*...sscale_x:0:*/
WireOpList WireOpList::scale_x(double factor) const
	{
	WireOpList ol;
	for ( Op *op = first; op != 0; op = op->next )
		{
		Op *op2 = op->scale_x(factor);
		if ( op2 == 0 )
			break;
		ol.append(op2);
		}
	return ol;
	}
/*...e*/
/*...sscale_y:0:*/
WireOpList WireOpList::scale_y(double factor) const
	{
	WireOpList ol;
	for ( Op *op = first; op != 0; op = op->next )
		{
		Op *op2 = op->scale_y(factor);
		if ( op2 == 0 )
			break;
		ol.append(op2);
		}
	return ol;
	}
/*...e*/
/*...sscale_z:0:*/
WireOpList WireOpList::scale_z(double factor) const
	{
	WireOpList ol;
	for ( Op *op = first; op != 0; op = op->next )
		{
		Op *op2 = op->scale_z(factor);
		if ( op2 == 0 )
			break;
		ol.append(op2);
		}
	return ol;
	}
/*...e*/
/*...smult_matrix:0:*/
WireOpList WireOpList::mult_matrix(const Matrix3 & m) const
	{
	WireOpList ol;
	for ( Op *op = first; op != 0; op = op->next )
		{
		Op *op2 = op->mult_matrix(m);
		if ( op2 == 0 )
			break;
		ol.append(op2);
		}
	return ol;
	}
/*...e*/
/*...smult_matrix:0:*/
WireOpList WireOpList::mult_matrix(const Matrix4 & m) const
	{
	WireOpList ol;
	for ( Op *op = first; op != 0; op = op->next )
		{
		Op *op2 = op->mult_matrix(m);
		if ( op2 == 0 )
			break;
		ol.append(op2);
		}
	return ol;
	}
/*...e*/

/*...srender:0:*/
// Scan along the opcode list. Map coordinates from world coordinates to
// the canonical view volume. Clip against this view volume. Optionally
// output the resulting line-segments, mapped to viewport (screen) coordinates.
//
// Note the clever use of a rolling loop of vertexes to cash in on the fact
// that most vertexes have 3 or more edges ending at them. We only do a mapping
// to the veiwpoint once for each vertex, although we still have to map to
// the viewport per clipped coordinate.

/*...sclip_to_view_volume:0:*/
// Clip to the canonical perspective view volume using Liang-Barsky 2D clipping
// algorithm, extended to 3D, as per F&vD&F&H p274.
//
// Visible points have :-
//   z >= zmin (near plane)
//   -1.0 <= x/z <= 1.0  or  -z <= x <= z
//   -1.0 <= y/z <= 1.0  or  -z <= y <= z
//
// Any point on the line from p0 to p1 may be described as
//   p0 + t * (p1-p0),  where 0.0 <= t <= 1.0
//
// Intersecting p0+t*dir with ax+by+cz+d==0 gives :-
//   t = -(a*p0.x+b*p0.y+c*p0.z+d)/(a*dir.x+b*dir.y+c*dir.z)
//
// Clipping planes, a, b, c, d,     t=
//   zmin <= z      0  0  1  -zmin  -( p0.z-zmin)/dir.z
//   -z <= x        1  0  1  0      -( p0.x+p0.z)/( dir.x+dir.z)
//   x <= z        -1  0  1  0      -(-p0.x+p0.z)/(-dir.x+dir.z)
//   -z <= y        0  1  1  0      -( p0.y+p0.z)/( dir.y+dir.z)
//   y <= z         0 -1  1  0      -(-p0.y+p0.z)/(-dir.y+dir.z)
//
// The signs of num and demon are used to determine whether we lines are
// potentially entering or leaving the view volume. See p117. Observe that
// the normals for each of the clipping planes point inward.

/*...sclipt:0:*/
// t may be computed as num/denom
// if denom is 0, then the line is parallel to the clipping plane

static Boolean clipt(
	double num, double denom,
	double & tmin, double & tmax
	)
	{
	if ( denom > 0.0 )
		{
		double t = num/denom;
		if ( t > tmax )
			return FALSE;
		if ( t > tmin )
			tmin = t;
		}
	else if ( denom < 0.0 )
		{
		double t = num/denom;
		if ( t < tmin )
			return FALSE;
		if ( t < tmax )
			tmax = t;
		}
	else
		if ( num > 0 )
			return FALSE;
	return TRUE;
	}
/*...e*/

#define	NEARPLANE (1e-10)

static Boolean clip_to_view_volume(Xyz & p0, Xyz & p1)
	{
	Xyz dir(p1-p0);
	double tmin = 0.0, tmax = 1.0;
	if ( clipt(-(p0.x+p0.z), dir.x+dir.z, tmin, tmax) )
		if ( clipt(-(-p0.x+p0.z), -dir.x+dir.z, tmin, tmax) )
			if ( clipt(-(p0.y+p0.z), dir.y+dir.z, tmin, tmax) )
				if ( clipt(-(-p0.y+p0.z), -dir.y+dir.z, tmin, tmax) )
					if ( clipt(-(p0.z-NEARPLANE), dir.z, tmin, tmax) )
						{
						if ( tmax < 1.0 )
							p1 = p0 + tmax*dir;
						if ( tmin > 0.0 )
							p0 = p0 + tmin*dir;
						return TRUE;
						}
	return FALSE;
	}
/*...e*/

#define	N_VERTEXES 32 // power of 2
#define	WRAP(n) ((n)&(N_VERTEXES-1))

Boolean WireOpList::render(const View & view, Bitmap & bitmap) const
	{
	Xyz vs[N_VERTEXES];
	int v = -1;
	int w = bitmap.width();
	int h = bitmap.height();
	if ( w < 1 || h < 1 )
		return FALSE;
	Matrix4 to_view(view.to_view_volume());
	Matrix4 to_port(to_viewport(w, h));
	int index = 0; // Start with colour 0
	for ( Op *op = first; op != 0; op = op->next )
		switch ( op->opcode() )		
			{
/*...s0 \45\ set colour:24:*/
case 0:
	{
	OpCol *op2 = (OpCol *) op;
	index = op2->index;
	}
	break;
/*...e*/
/*...s1 \45\ define vertex:24:*/
case 1:
	{
	OpVertex *op2 = (OpVertex *) op;
	vs[WRAP(++v)] = to_view * op2->xyz;
	}
	break;
/*...e*/
/*...s2 \45\ define edge:24:*/
case 2:
	{
	OpEdge *op2 = (OpEdge *) op;
	Xyz p1(vs[WRAP(v-op2->vertex1)]);
	Xyz p2(vs[WRAP(v-op2->vertex2)]);
	if ( clip_to_view_volume(p1, p2) )
		{
		Xyz q1(to_port*p1);
		Xyz q2(to_port*p2);
		bitmap.line(
			(int) q1.x, (int) q1.y,
			(int) q2.x, (int) q2.y,
			index);
		}
	}
	break;
/*...e*/
			}
	return TRUE;
	}
/*...e*/
