//
// wire.h - Support for Wire Frame models
//
// The intention is to allow the renderer to be able to generate a much simpler
// wire frame version of their model, which hopefully can be rendered up in
// much smaller times. Thus the geometry may be checked for consistency and
// asthetic value before paying the price of a full raytrace.
//
// The wireframe renderer seperates definition of the vertexes from the edges,
// so as to be able map them vertexes to the viewpoint once, rather than per
// line (particularly useful for small 3D objects, such as cubes etc., ie:
// most of the images this module is likely to be used to produce).
//
// The rendering engine allows you continually define vertexes in 3D space.
// At render time, the N (typically 32) most recently defined vertexes are
// held in a cache. When a request to render an edge is given in terms of
// two vertex numbers, these identify the vertexes by how recently they were
// defined (0 = last, 1 = one before, 2 = one before that...). eg:
//
//	{
//	OpList ol;
//	ol.vertex(Xyz(0,0,0))	// bl vertex is #3
//	  .vertex(Xyz(1,0,0))	// br vertex is #2
//	  .vertex(Xyz(0,1,0))	// tl vertex is #1
//	  .vertex(Xyz(1,1,0))	// tr vertex is #0
//	  .edge(3,2)		// bl to br
//	  .edge(2,0)		// br to tr
//	  .edge(0,1)		// tr to tl
//	  .edge(1,3);		// tl to bl
//	}

#ifndef WIRE_H
#define	WIRE_H

/*...sincludes:0:*/
#include "types.h"
#include "xyz.h"
#include "rgb.h"
#include "matrix.h"
#include "bitmap.h"
#include "view.h"
#include "oplist.h"

/*...vtypes\46\h:0:*/
/*...vxyz\46\h:0:*/
/*...vrgb\46\h:0:*/
/*...vmatrix\46\h:0:*/
/*...vbitmap\46\h:0:*/
/*...vview\46\h:0:*/
/*...voplist\46\h:0:*/
/*...e*/

class WireOpList : public OpList
	{
public:
	WireOpList();
	WireOpList(const WireOpList & ol);
	WireOpList & operator=(const WireOpList & ol);
	WireOpList & operator+=(const WireOpList & ol);
	~WireOpList();

	// Actual drawing commands
	WireOpList & col(int index);
	WireOpList & vertex(const Xyz & xyz);
	WireOpList & edge(int vertex1, int vertex2);

	// Manipulating coordinates within lists
	WireOpList trans  (Xyz t) const;
	WireOpList trans_x(double t) const;
	WireOpList trans_y(double t) const;
	WireOpList trans_z(double t) const;
	WireOpList rot_x(double angle) const;
	WireOpList rot_y(double angle) const;
	WireOpList rot_z(double angle) const;
	WireOpList scale  (Xyz factor) const;
	WireOpList scale  (double factor) const;
	WireOpList scale_x(double factor) const;
	WireOpList scale_y(double factor) const;
	WireOpList scale_z(double factor) const;
	WireOpList mult_matrix(const Matrix3 & m) const;
	WireOpList mult_matrix(const Matrix4 & m) const;

	// Render to a bitmap
	Boolean render(const View & view, Bitmap & bitmap) const;
	};

#endif
