/**
 @file		main.c
 @brief 		Multicast source ( for W5100 @ Atmega128 )
 			Optimized compiler : WinAVR-20071221
 */
 
#include <avr/io.h>
#include <avr/interrupt.h>
#include <avr/eeprom.h>
#include <stdio.h> 
#include <string.h>


#include "types.h"
#include "w5100.h"

#include "mcu.h"
#include "delay.h"
#include "serial.h"
#include "socket.h"


extern char uart0_getchar(FILE *f);


#define DEFAULT_NET_MEMALLOC	0x55		/**< Default iinchip memory allocation */
#define LOOP_BACK_PORT		5000
#define MULTICAST_PORT		2000
u_char bchannel_start[MAX_SOCK_NUM] = { 0, 0, 0, 0};	/** 0:close, 1:ready, 2:connect */


/**
 @brief	loopback test using TCP(server mode)
 */
void loopback_tcps(
	u_char ch	/**< channel(socket) index */
	)
{
	int len;							
	u_char * data_buf = (u_char*) TX_BUF;

	switch (getSn_SR(ch))
	{
	case SOCK_ESTABLISHED:						/* if connection is established */
		if(bchannel_start[ch]==1)
		{
			//printf("%d : Connected by %s(%u)",ch,inet_ntoa(GetDestAddr(ch)),GetDestPort(ch));
			bchannel_start[ch] = 2;
		}
		if ((len = getSn_RX_RSR(ch)) > 0) 			/* check Rx data */
		{
			if (len > TX_RX_MAX_BUF_SIZE) len = TX_RX_MAX_BUF_SIZE;	/* if Rx data size is lager than TX_RX_MAX_BUF_SIZE */
									/* the data size to read is MAX_BUF_SIZE. */
			len = recv(ch, data_buf, len);			/* read the received data */
			send(ch, data_buf, len);				/* send the received data */
		}
		break;
	case SOCK_CLOSE_WAIT:                           		/* If the client request to close */
		printf("%d : CLOSE_WAIT\r\n", ch);
		disconnect(ch);
		bchannel_start[ch] = 0;
		break;
	case SOCK_CLOSED:                                               /* if a socket is closed */
		if(!bchannel_start[ch]) 
		{
			printf("%d : Loop-Back TCP Server Started.\r\n",ch);
			bchannel_start[ch] = 1;
		}
		if(socket(ch,Sn_MR_TCP,LOOP_BACK_PORT,0x00) == 0)    /* reinitialize the socket */
		{
			printf("\a%d : Fail to create socket.\r\n",ch);
			bchannel_start[ch] = 0;
		}
		else
			listen(ch);
		
		break;
	}
}


/**
 @brief	loopback test using UDP
 */
void loopback_udp(
	u_char ch	/**< channel(socket) index */
	)
{
	int len;
	u_char * data_buf = (u_char*) TX_BUF;
	u_long destip = 0;
	uint16 destport;

	switch (getSn_SR(ch))
	{
	case SOCK_UDP:
		if ((len = getSn_RX_RSR(ch)) > 0) 			/* check Rx data */
		{
			if (len > TX_RX_MAX_BUF_SIZE) len = TX_RX_MAX_BUF_SIZE;	/* if Rx data size is lager than TX_RX_MAX_BUF_SIZE */
									/* the data size to read is MAX_BUF_SIZE. */
			len = recvfrom(ch, data_buf, len,(u_char*)&destip,&destport);			/* read the received data */
#if (SYSTEM_ENDIAN == _ENDIAN_LITTLE_ )
			printf("\r\nMulticast:destip:[%d.%d.%d.%d]:[%d], Data Len[%d]\r\n", 
				(int)(destip & 0xFF),(int)(destip>>8 & 0xFF),(int)(destip>>16 & 0xFF),(int)(destip>>24 & 0xFF),
				destport, len);
#else
			printf("\r\nMulticast:destip:[%d.%d.%d.%d]:[%d], Data Len[%d]\r\n", 
				(int)(destip>>24 & 0xFF),(int)(destip>>16 & 0xFF),(int)(destip>>8 & 0xFF),(int)(destip & 0xFF),
				destport, len);
#endif

#if 0									
			if(sendto(ch, data_buf, len,(u_char*)&destip,destport) == 0)	/* send the received data */
			{
				printf("\a\a\a%d : System Fatal Error.");
				evb_soft_reset();
			}
#endif
		}
		break;
	case SOCK_CLOSED:                                               /* if a socket is closed */
		printf("%d : Loop-Back UDP Started.",ch);
		if(socket(ch,Sn_MR_UDP,MULTICAST_PORT, Sn_MR_MULTI)== 0)    /* reinitialize the socket */
			printf("\a%d : Fail to create socket.",ch);
		break;
	}
}


/**
 @brief 	set the IP, subnet, gateway address
 */
static void SetIP(void)
{
	u_char ip[6];

	// ip address
	ip[0] = 211;ip[1] = 46;ip[2] = 117;ip[3] = 84;
	setSIPR( ip );
	
	// MAC address
	ip[0] = 0x00; ip[1] = 0x08; ip[2] = 0xDC; ip[3] = 0x00; ip[4] = 0x00; ip[5] = 0x4F;
	setSHAR( ip );

	// subnet mask
	ip[0] = 255;ip[1] = 255;ip[2] = 255;ip[3] = 224;
	setSUBR( ip );

	// gateway address
	ip[0] = 211;ip[1] = 46;ip[2] = 117;ip[3] = 94;
	setGAR( ip );


///////////////////////////////
	printf("socket : ");
	

}


/**
	@mainpage	W5100 Firmware Source for Multicast
	@section intro	Introduction
	- Introduction : supported multicast
	@section CREATEINFO	Author
	- author : www.wiznet.co.kr
	- date : 2007.1.2
*/
int main( void )
{
	int i;
	u_char ip[6];
	uint16 port = 0;
	
	mcu_init();
	uart_init(0,7);

	printf("\r\n*** Multicast Test using W5100 ");
#if (__DEF_IINCHIP_BUS__ == __DEF_IINCHIP_DIRECT_MODE__)
	printf("[DIRECT MODE]");
#elif(__DEF_IINCHIP_BUS__ == __DEF_IINCHIP_INDIRECT_MODE__)	/* INDIRECT MODE I/F */
	printf("[INDIRECT MODE]");
#elif (__DEF_IINCHIP_BUS__ == __DEF_IINCHIP_SPI_MODE__)
	printf("[SPI MODE]");
#else
	#error "unknown bus type"
#endif	
	printf(" ***\r\n");
	
	printf("******* Firmware Version : %d.%d.%d.%d *******\r\n",
			(u_char)(FW_VERSION>>24),
			(u_char)(FW_VERSION>>16),
			(u_char)(FW_VERSION>>8),
			(u_char)(FW_VERSION) );
		
	wait_10ms(100);

	iinchip_init();

	// Set IP, subnet, gateway address
	SetIP();
#ifdef __DEF_IINCHIP_INT__
	setIMR(0xEF);
#endif	
	sysinit(DEFAULT_NET_MEMALLOC,DEFAULT_NET_MEMALLOC);

	/*********************************************/
	/* Multicasting Setting */
	/*********************************************/	
	// 1. write multicasting mac address to destination hardware address
	ip[0] = 0x01; ip[1] = 0x00; ip[2] = 0x5E; ip[3] = 0x01; ip[4] = 0x01; ip[5] = 0x01;
	setSn_DHAR(0,ip);
	
	// 2. write multicasting ip address to destination ip address
	ip[0] = 224; ip[1] = 1; ip[2] = 1; ip[3] = 1;
	setSn_DIPR(0,ip);
	
	// 3. write multicasting port number to destination port number
	ip[0] = 0x0F; ip[1] = 0xA0;	//4000
	setSn_DPORT(0,ip);
	
	setSn_TTL(0,1);
	
	// 4. set udp, multicasting on socket Mode register & open socket
	if (socket(0, Sn_MR_UDP, MULTICAST_PORT, Sn_MR_MULTI) == 1) 
	{
		printf("Channel 0 Status : %.2X ok\r\n", getSn_SR(0));
	}

	printf("MULTICASING MAC ADDRESS : ");
	getSn_DHAR(0,ip);
	for (i = 0; i < 5; i++)	
		printf("%.2X:", ip[i]);

	printf("%.2X\r\n", ip[i]);
	
	printf("MULTICASTING IP ADDRESS : ");
	getSn_DIPR(0,ip);
	for (i = 0; i < 3; i++) 
		printf("%d.", ip[i]);
	
	printf("%d\r\n", ip[i]);

	printf("MULTICASTING PORT : ");
	getSn_DPORT(0, ip);
	port = (( ip[0] & 0xFF ) << 8) | (ip[1] & 0xFF );
	printf("%d\r\n", port);
	
	printf("\r\n*** LOOPBACK START ***\r\n");
	
	while(1)
	{
		for (i = 1 ; i < MAX_SOCK_NUM ; i++)
		{
			loopback_tcps(i);
		}

		loopback_udp( 0 );
	}
	
	printf("\r\n*** THANK YOU ***\r\n");
	
	return 1;
	
}

