/*

rawio.c - Raw I/O

A program to do raw read/write on Windows.

I used to use a Windows variant of dd, from http://www.chrysocome.net/dd.
But that stopped working for me on Windows 10.

This code can help find suitable disks and to do block aligned transfers.
Only works if you are running as Administrator.
http://support.microsoft.com/kb/100027

*/

/*...sincludes:0:*/
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <errno.h>

#include <windows.h>
#include <winioctl.h>
/*...e*/

/*...sfatal:0:*/
static void fatal(const char *fmt, ...)
	{
	va_list	vars;
	char s[256+1];
	va_start(vars, fmt);
	vsprintf(s, fmt, vars);
	va_end(vars);
	fprintf(stderr, "rawio: %s\n", s);
	exit(1);
	}
/*...e*/
/*...susage:0:*/
static void usage(void)
	{
	fprintf(stderr, "usage: rawio [flags]\n");
	fprintf(stderr, "flags: -f file    filename\n");
	fprintf(stderr, "       -d device  device filename\n");
	fprintf(stderr, "       -o offset  offset into device (default 0)\n");
	fprintf(stderr, "       -l length  length to read or write (default 8MB)\n");
	fprintf(stderr, "       -R         read from device to file\n");
	fprintf(stderr, "       -W         write to device from file\n");
	fprintf(stderr, "       -L         list available devices\n");
	exit(1);
	}
/*...e*/
/*...sdefines:0:*/
#define	K 1024LL
#define	SS_MAX 4096
/*...e*/
/*...sdevice I\47\O routines:0:*/
typedef struct
	{
	HANDLE hdisk;
	int sector_size;
	} DEV;

/*...sll_seek:0:*/
static BOOL ll_seek(
	HANDLE hdisk,
	long long pos /* must be sector aligned */
	)
	{
	LARGE_INTEGER li;
	li.QuadPart = pos;
	return SetFilePointerEx(
		hdisk,
		li,
		NULL,
		FILE_BEGIN
		);
	}
/*...e*/

/*...svalidBytesPerSector:0:*/
static BOOLEAN validBytesPerSector(int bps)
	{
	return bps ==  128 ||
	       bps ==  256 ||
	       bps ==  512 ||
	       bps == 1024 ||
	       bps == 2048 ||
	       bps == 4096 ;
	}
/*...e*/

/*...sdev_list:0:*/
/*...sshow:0:*/
/*...smediaTypeName \45\ 9 characters:0:*/
static const char *mediaTypeName(enum MEDIA_TYPE mt)
	{
	switch ( mt )
		{
		case Unknown:		return "unknown";
		case RemovableMedia:	return "removable";
		case FixedMedia:	return "fixed";
		}
	if ( mt >= F5_1Pt2_512 && mt <= F3_32M_512 )
		return "floppy";
	return "?";
	}
/*...e*/
/*...sshowSize \45\ usually 6 characters:0:*/
/* 1.23MB */

static void showSize(char *s, long long n)
	{
	if ( n >= K*K*K*K )
		sprintf(s, "%4.2lfTB", (double)n/(K*K*K*K));
	else if ( n >= K*K*K )
		sprintf(s, "%4.2lfGB", (double)n/(K*K*K));
	else if ( n >= K*K )
		sprintf(s, "%4.2lfMB", (double)n/(K*K));
	else if ( n >= K )
		sprintf(s, "%4.2lfKB", (double)n/(K));
	else
		sprintf(s, "%dB ", (int) n);
	}
/*...e*/

static void show(const char *deviceName)
	{
	HANDLE h;
	DISK_GEOMETRY dg;
	DWORD dwResult;
	if ( (h = CreateFileA(
		deviceName,
		GENERIC_READ,
		FILE_SHARE_READ|FILE_SHARE_WRITE,
		NULL,
		OPEN_EXISTING,
		0,
		NULL
		)) == INVALID_HANDLE_VALUE )
		return;
	if ( DeviceIoControl(
		h,
		IOCTL_DISK_GET_DRIVE_GEOMETRY,
		NULL,
		0,
		&dg,
		sizeof(dg),
		&dwResult,
		NULL) &&
	     validBytesPerSector(dg.BytesPerSector) )
		{
		char ss[20+1];
		showSize(ss, dg.Cylinders.QuadPart*dg.TracksPerCylinder*dg.SectorsPerTrack*dg.BytesPerSector);
		fprintf(stdout, "%-40s %-9s %7lld %3d %2d %4d %8s\n",
			deviceName,
			mediaTypeName(dg.MediaType),
			dg.Cylinders.QuadPart,
			dg.TracksPerCylinder,
			dg.SectorsPerTrack,
			dg.BytesPerSector,
			ss
			);
		}
	CloseHandle(h);
	}
/*...e*/

static void dev_list(void)
	{
	int i;
	char s[100+1];
	for ( i = 0; i < 64; i++ )
		{
		sprintf(s, "\\\\.\\PhysicalDrive%d", i); 
		show(s);
		}
	for ( i = 0; i < 26; i++ )
		{
		sprintf(s, "\\\\.\\%c:", 'A'+i);
		show(s);
		}
	}
/*...e*/
/*...sdev_open:0:*/
static const char *dev_open(DEV *d, const char *filename, int w32mode)
	{
	DWORD dwResult;
	DISK_GEOMETRY dg;
	if ( (d->hdisk = CreateFileA(
		filename,
		w32mode,
		FILE_SHARE_READ|FILE_SHARE_WRITE,
		NULL,
		OPEN_EXISTING,
		0,
		NULL)) == INVALID_HANDLE_VALUE )
		return "CreateFile failed";
	if ( ! DeviceIoControl(
		d->hdisk,
		IOCTL_DISK_GET_DRIVE_GEOMETRY,
		NULL,
		0,
		&dg,
		sizeof(dg),
		&dwResult,
		NULL) )
		{
		CloseHandle(d->hdisk);
		return "DeviceIoControl IOCTL_DISK_GET_DRIVE_GEOMETRY failed";
		}
	if ( !validBytesPerSector(dg.BytesPerSector) )
		{
		CloseHandle(d->hdisk);
		return "invalid BytesPerSector";
		}
	if ( ! DeviceIoControl(
		d->hdisk,
		FSCTL_LOCK_VOLUME,
		NULL,
		0,
		NULL,
		0,
		&dwResult,
		NULL) )
		{
		CloseHandle(d->hdisk);
		return "DeviceIoControl FSCTL_LOCK_VOLUME failed";
		}
	d->sector_size = dg.BytesPerSector;
	return NULL;
	}
/*...e*/
/*...sdev_close:0:*/
static void dev_close(DEV *d)
	{
	DWORD dwResult;
	DeviceIoControl(
		d->hdisk,
		FSCTL_UNLOCK_VOLUME,
		NULL,
		0,
		NULL,
		0,
		&dwResult,
		NULL
		);
	DeviceIoControl(
		d->hdisk,
		FSCTL_DISMOUNT_VOLUME,
		NULL,
		0,
		NULL,
		0,
		&dwResult,
		NULL
		);
	CloseHandle(d->hdisk);
	}
/*...e*/
/*...sdev_read:0:*/
static const char *dev_read(const DEV *d, long long offset, char *buf, long long length)
	{
	long long ss = d->sector_size;
	while ( length > 0 )
		{
		long long secaddr = (offset/ss)*ss;
		long long thisgo = min(length, secaddr+ss-offset);
		DWORD dwBytes;
		char buf_sec[SS_MAX];
		if ( !ll_seek(d->hdisk, secaddr) )
			return "bad seek prior to read";
		if ( ! ReadFile(
			d->hdisk,
			buf_sec,
			(DWORD)ss,
			&dwBytes,
			NULL) )
			return "error reading";
		if ( (long long) dwBytes < ss )
			return "error reading full sector";
		memcpy(buf, buf_sec+(offset%ss), (size_t) thisgo);
		buf    += thisgo;
		offset += thisgo;
		length -= thisgo;
		}
	return NULL;
	}
/*...e*/
/*...sdev_write:0:*/
static const char *dev_write(const DEV *d, long long offset, const char *buf, long long length)
	{
	long long ss = d->sector_size;
	while ( length > 0 )
		{
		long long secaddr = (offset/ss)*ss;
		long long thisgo = min(length, secaddr+ss-offset);
		DWORD dwBytes;
		char buf_sec[SS_MAX];
		if ( thisgo < ss )
			/* need to read-modify-write */
			{
			if ( !ll_seek(d->hdisk, secaddr) )
				return "bad seek prior to read";
			if ( ! ReadFile(
				d->hdisk,
				buf_sec,
				(DWORD)ss,
				&dwBytes,
				NULL) )
				return "error reading";
			if ( (long long) dwBytes < ss )
				return "error reading full sector";
			}
		memcpy(buf_sec+(offset%ss), buf, (size_t) thisgo);
		if ( !ll_seek(d->hdisk, secaddr) )
			return "bad seek prior to writed";
		if ( ! WriteFile(
			d->hdisk,
			buf_sec,
			(DWORD)ss,
			&dwBytes,
			NULL) )
			return "error writing";
		if ( (long long) dwBytes < ss )
			return "error writing full sector";
		buf    += thisgo;
		offset += thisgo;
		length -= thisgo;
		}
	return NULL;
	}
/*...e*/
/*...e*/
/*...sdo_read\47\do_write:0:*/
#define	BUF_MAX (1024*1024)

static void do_read(const char *devicename, long long offset, const char *filename, long long length)
	{
	char *buf;
	const char *err;
	DEV dev;
	FILE *fp;
	if ( (buf = malloc(BUF_MAX)) == NULL )
		fatal("out of memory");
	if ( (err = dev_open(&dev, devicename, GENERIC_READ)) != NULL )
		fatal("error opening device %s for reading: %s", devicename, err);
	if ( (fp = fopen(filename, "wb")) == NULL )
		{
		dev_close(&dev);
		fatal("error opening file %s for writing", filename);
		}
	while ( length > 0 )
		{
		long long thisgo = min(length, BUF_MAX);
		if ( (err = dev_read(&dev, offset, buf, thisgo)) != NULL )
			{
			dev_close(&dev);
			fclose(fp);
			remove(filename);
			fatal("error reading device %s: %s", devicename, err);
			}
		if ( fwrite(buf, 1, (size_t) thisgo, fp) != (size_t) thisgo )
			{
			dev_close(&dev);
			fclose(fp);
			remove(filename);
			fatal("error writing file %s", filename);
			}
		offset += thisgo;
		length -= thisgo;
		}
	dev_close(&dev);
	fclose(fp);
	free(buf);
	}

static void do_write(const char *filename, const char *devicename, long long offset, long long length)
	{
	char *buf;
	const char *err;
	FILE *fp;
	DEV dev;
	if ( (buf = malloc(BUF_MAX)) == NULL )
		fatal("out of memory");
	if ( (fp = fopen(filename, "rb")) == NULL )
		fatal("error opening file %s for reading", filename);
	if ( (err = dev_open(&dev, devicename, GENERIC_READ|GENERIC_WRITE)) != NULL )
		{
		fclose(fp);
		fatal("error opening device %s for writing: %s", devicename, err);
		}
	while ( length > 0 )
		{
		long long thisgo = min(length, BUF_MAX);
		if ( fread(buf, 1, (size_t) thisgo, fp) != (size_t) thisgo )
			{
			fclose(fp);
			dev_close(&dev);
			fatal("error reading file %s", filename);
			}
		if ( (err = dev_write(&dev, offset, buf, thisgo)) != NULL )
			{
			fclose(fp);
			dev_close(&dev);
			fatal("error writing device %s: %s", devicename, err);
			}
		offset += thisgo;
		length -= thisgo;
		}
	fclose(fp);
	dev_close(&dev);
	free(buf);
	}
/*...e*/
/*...smain:0:*/
/*...sparse_long_long:0:*/
static long long parse_long_long(const char *s)
	{
	const char *s2 = s;
	long long n = 0LL;
	while ( *s != '\0' && isdigit(*s) )
		n = n * 10LL + (*s++ - '0');
	if ( *s == '\0' || !_strcmpi(s, "B") )
		;
	else if ( !_strcmpi(s, "K") || !_strcmpi(s, "KB") || !_strcmpi(s, "KiB") )
		n *= K;
	else if ( !_strcmpi(s, "M") || !_strcmpi(s, "MB") || !_strcmpi(s, "MiB") )
		n *= K * K;
	else if ( !_strcmpi(s, "G") || !_strcmpi(s, "GB") || !_strcmpi(s, "GiB") )
		n *= K * K * K;
	else if ( !_strcmpi(s, "T") || !_strcmpi(s, "TB") || !_strcmpi(s, "TiB") )
		n *= K * K * K * K;
	else
		fatal("can't parse %s", s2);
	return n;
	}
/*...e*/

int main(int argc, char *argv[])
	{
	long long offset = 0;
	long long length = 8*K*K;
	const char *filename = NULL;
	const char *devicename = NULL;
	int i;

	if ( argc == 1 )
		usage();

	for ( i = 1; i < argc; i++ )
		{
		if ( argv[i][0] != '-' )
			break;
		else if ( argv[i][1] == '-' )
			{ ++i; break; }
		switch ( argv[i][1] )
			{
			case 'f':
				if ( ++i == argc )
					usage();
				filename = argv[i];
				break;
			case 'd':
				if ( ++i == argc )
					usage();
				devicename = argv[i];
				break;
			case 'o':
				if ( ++i == argc )
					usage();
				offset = parse_long_long(argv[i]);
				break;
			case 'l':
				if ( ++i == argc )
					usage();
				length = parse_long_long(argv[i]);
				break;
			case 'R':
				if ( devicename == NULL )
					fatal("devicename not specified");
				if ( filename == NULL )
					fatal("filename not specified");
				do_read(devicename, offset, filename, length);
				break;
			case 'W':
				if ( filename == NULL )
					fatal("filename not specified");
				if ( devicename == NULL )
					fatal("devicename not specified");
				do_write(filename, devicename, offset, length);
				break;
			case 'L':
				dev_list();
				break;
			default:
				usage();
				break;
			}
		}

	if ( i != argc )
		usage();

	return 0;
	}
/*...e*/
