/*

hashmap.c - Something like java.util.HashMap<K,V>

*/

/*...sincludes:0:*/
#include <stdlib.h>
#include <string.h>

#include "hashmap.h"
/*...e*/

/*...svars:0:*/
typedef struct
	{
	void *key;
	void *value;
	} ENTRY;

struct _HASHMAP
	{
	HM_BOOLEAN (*equal)(const void *, const void *);
	unsigned (*hashcode)(const void *);
	void * (*clone)(const void *);
	void (*free)(void *);
	void * (*clone_value)(const void *);
	void (*free_value)(void *);
	int size;
	int used;
	ENTRY *table;
	};

#define	INIT_SIZE 17 /* prime */

static char hm_deleted[1]; /* Special dummy key */
/*...e*/

/*...shm_create:0:*/
HASHMAP *hm_create(
	HM_BOOLEAN (*equal)(const void *, const void *),
	unsigned (*hashcode)(const void *),
	void * (*clone)(const void *),
	void (*free)(void *),
	void * (*clone_value)(const void *),
	void (*free_value)(void *),
	int init_size
	)
	{
	HASHMAP *h;
	int i;
	if ( (h = (HASHMAP *) malloc(sizeof(HASHMAP))) == NULL )
		return NULL;
	h->equal       = equal;
	h->hashcode    = hashcode;
	h->clone       = clone;
	h->free        = free;
	h->clone_value = clone_value;
	h->free_value  = free_value;
	h->size        = ( init_size != 0 ) ? init_size : INIT_SIZE;
	h->used        = 0;
	if ( (h->table = (ENTRY *) malloc(h->size*sizeof(ENTRY))) == NULL )
		{
		free(h);
		return NULL;
		}
	for ( i = 0; i < h->size; i++ )
		h->table[i].key = NULL;
	return h;
	}	
/*...e*/
/*...shm_delete:0:*/
void hm_delete(HASHMAP *h)
	{
	int i;
	for ( i = 0; i < h->size; i++ )
		if ( h->table[i].key != NULL && h->table[i].key != hm_deleted )
			{
			h->free      (h->table[i].key  );
			h->free_value(h->table[i].value);
			}
	free(h->table);
	free(h);
	}
/*...e*/
/*...shm_find:0:*/
/* Return location of key, or -1 */
static int hm_find(HASHMAP *h, const void *k)
	{
	unsigned code = h->hashcode(k);
	int i;
	for ( i = 0; i < h->size; i++ )
		{
		int j = (code+i) % h->size;
		if ( h->table[j].key == NULL )
			return -1;
		if ( h->table[j].key != hm_deleted && h->equal(h->table[j].key, k) )
			return j;
		}
	return -1;
	}
/*...e*/
/*...shm_contains:0:*/
HM_BOOLEAN hm_contains(HASHMAP *h, const void *k)
	{
	return hm_find(h, k) != -1;
	}
/*...e*/
/*...shm_get:0:*/
const void *hm_get(HASHMAP *h, const void *k)
	{
	int i;
	if ( (i = hm_find(h, k)) == -1 )
		return NULL;
	return h->table[i].value;	
	}
/*...e*/
/*...shm_rehash:0:*/
/* Redistribute keys over hashmap.
   Hashmap guaranteed to be empty to start with.
   All keys are distinct.
   Guaranteed to be enough space for all. */
static void hm_rehash(HASHMAP *h, ENTRY *table, int size)
	{
	int p;
	for ( p = 0; p < size; p++ )
		if ( table[p].key != NULL && table[p].key != hm_deleted )
			{
			unsigned code = h->hashcode(table[p].key);
			int i;
			for ( i = 0; i < h->size; i++ )
				{
				int j = (code+i) % h->size;
				if ( h->table[j].key == NULL )
					{
					h->table[j].key   = table[p].key  ;
					h->table[j].value = table[p].value;
					break;
					}
				}
			}
	}
/*...e*/
/*...shm_resize:0:*/
/* Resize the table to a new size.
   As a side effect, weed out the hm_deleted markers. */
static HM_BOOLEAN hm_resize(HASHMAP *h, int new_size)
	{
	int old_size     = h->size;
	ENTRY *old_table = h->table;
	ENTRY *new_table;
	int i;
	if ( (new_table = (ENTRY *) malloc(new_size*sizeof(ENTRY))) == NULL )
		return HM_FALSE;
	for ( i = 0; i < new_size; i++ )
		new_table[i].key = NULL;
	h->table = new_table;
	h->size  = new_size;
	hm_rehash(h, old_table, old_size);
	free(old_table);
	return HM_TRUE;
	}
/*...e*/
/*...shm_put:0:*/
HM_BOOLEAN hm_put(HASHMAP *h, const void *k, const void *v)
	{
	unsigned code = h->hashcode(k);
	int i;
	if ( h->used > (h->size*3)/4 )
		/* Over 75% full, so try to enlarge,
		   resulting in ending up around 37% full.
		   If enlargement fails, keep going anyway. */
		hm_resize(h, h->size*2);
	for ( i	= 0; i < h->size; i++ )
		{
		int j = (code+i) % h->size;
		if ( h->table[j].key == NULL || h->table[j].key == hm_deleted )
			{
			void *kc, *vc;
			if ( (kc = h->clone(k)) == NULL )
				return HM_FALSE;
			if ( (vc = h->clone_value(v)) == NULL )
				{ h->free(kc); return HM_FALSE; }
			h->table[j].key   = kc;
			h->table[j].value = vc;
			++(h->used);
			return HM_TRUE;
			}
		if ( h->equal(h->table[j].key, k) )
			{
			void *vc;
			if ( (vc = h->clone_value(v)) == NULL )
				return HM_FALSE;
			h->free_value(h->table[j].value);
			h->table[j].value = vc;
			return HM_TRUE;
			}
		}
	return HM_FALSE; /* We couldn't enlarge, and we're full */
	}
/*...e*/
/*...shm_remove:0:*/
void hm_remove(HASHMAP *h, const void *k)
	{
	int j;
	if ( (j = hm_find(h, k)) != -1 )
		{
		h->free      (h->table[j].key  );
		h->free_value(h->table[j].value);
		h->table[j].key = hm_deleted; /* So that chaining doesn't break. */
		--(h->used);
		if ( h->used < h->size/4 && h->used > INIT_SIZE )
			/* Less than 25% full, so try to shrink,
			   resulting in ending up around 50% full.
			   If shrinking fails, keep going anyway. */
			hm_resize(h, h->size/2);
		}
	}
/*...e*/
/*...shm_visit:0:*/
void hm_visit(
	HASHMAP *h,
	void (*visit)(const void *, const void *, void *),
	void *context
	)
	{
	int i;
	for ( i = 0; i < h->size; i++ )
		if ( h->table[i].key != NULL && h->table[i].key != hm_deleted )
			visit(h->table[i].key, h->table[i].value, context);
	}
/*...e*/
/*...shm_debug:0:*/
/* Only works if map contains strings mapping to strings. */
void hm_debug(HASHMAP *h)
	{
	int i;
	printf("size=%d, used=%d\n", h->size, h->used);
	for ( i = 0; i < h->size; i++ )
		if ( h->table[i].key == NULL )
			printf("table[%d]=NULL\n", i);
		else if ( h->table[i].key == hm_deleted )
			printf("table[%d]=hm_deleted\n", i);
		else
			printf("table[%d]=\"%s\"\n", i, h->table[i].key, h->table[i].value);
	}
/*...e*/
