/*

linuxsampler.c - Script to sample CPU and memory

Using ps is problematic, as it reports CPU since process inception.
We want a more instantaneous measure.

*/

/*...sincludes:0:*/
#include <stdio.h>
#include <stdlib.h>
#include <stdarg.h>
#include <string.h>
#include <ctype.h>
#include <time.h>
#include <sys/times.h>
#include <unistd.h>	/* for sysconf() */
#include <dirent.h>
#include <pwd.h>

#include "hashmap.h"

/*...vhashmap\46\h:0:*/
/*...e*/

/*...sdefines:0:*/
#define	DEF_INTERVAL  60
#define	DEF_MIN_LINES 1000
#define	DEF_MAX_LINES 1100
#define	MIN_MIN_LINES 10
#define	MAX_MAX_LINES 1000000
#define	MIN_DIF_LINES 10

#define	L_LOG_FN_MAX    500
#define	L_LOG_LINE_MAX  100
#define	L_PROC_LINE_MAX 200
/*...e*/
/*...sfatal:0:*/
static void fatal(const char *fmt, ...)
	{
	va_list	vars;
	char s[256+L_LOG_FN_MAX+1];
	va_start(vars, fmt);
	vsprintf(s, fmt, vars);
	va_end(vars);
	fprintf(stderr, "linuxsampler: %s\n", s);
	exit(1);
	}
/*...e*/
/*...susage:0:*/
static void usage(void)
	{
	fprintf(stderr, "usage: linuxsampler {flags}\n");
	fprintf(stderr, "flags: -i interval  sample every interval seconds (default %d)\n", DEF_INTERVAL);
	fprintf(stderr, "       -f output    output to a file\n");
	fprintf(stderr, "       -s min max   lines in the file (default %d %d)\n", DEF_MIN_LINES, DEF_MAX_LINES);
	fprintf(stderr, "       -m units     memory units (matching [KMGTPE]?i?B or %)\n");
	exit(2);
	}
/*...e*/

/*...svars:0:*/
static int interval = DEF_INTERVAL;

static unsigned long mem_total_kb;
static unsigned long mem_divider = (unsigned long) -1; /* use % */

static int sc_clk_tck;
static int sc_pagesize;

typedef struct
	{
	unsigned long jiffies;
	unsigned long rss;
	} UIDTOT;
/*...e*/

/*...slog_\42\:0:*/
static const char *fn_log = NULL;
static int min_lines = DEF_MIN_LINES, max_lines = DEF_MAX_LINES;
static int n_lines;
static FILE *fp_log;

/*...slog_init:0:*/
static void log_init(void)
	{
	if ( fn_log == NULL )
		fp_log = stdout;
	else
		{
		FILE *fp;
		if ( strlen(fn_log) > L_LOG_FN_MAX )
			fatal("log filename cannot be longer than %d characters", L_LOG_FN_MAX);
		n_lines = 0;
		if ( (fp_log = fopen(fn_log, "r")) != NULL )
			{
			char line[L_LOG_LINE_MAX+1];
			while ( fgets(line, L_LOG_LINE_MAX+1, fp_log) != NULL )
				++n_lines;
			fclose(fp_log);
			}
		}
	}
/*...e*/
/*...slog_begin_batch:0:*/
static void log_begin_batch(void)
	{
	if ( fn_log != NULL )
		if ( (fp_log = fopen(fn_log, "a")) == NULL )
			fatal("can't append to log file %s", fn_log);
	}
/*...e*/
/*...slog_line:0:*/
static void log_line(const char *fmt, ...)
	{
	va_list	vars;
	char s[L_LOG_LINE_MAX+1];
	va_start(vars, fmt);
	vsprintf(s, fmt, vars);
	va_end(vars);
	fprintf(fp_log, "%s\n", s);
	++n_lines;
	}
/*...e*/
/*...slog_end_batch:0:*/
static void log_end_batch(void)
	{
	fflush(fp_log);
	if ( fn_log != NULL )
		fclose(fp_log);
	}
/*...e*/
/*...slog_prune:0:*/
static void log_prune(void)
	{
	if ( fn_log != NULL )
		if ( n_lines > max_lines )
			{
			char fn_log_new[L_LOG_FN_MAX+4+1];
			FILE *fp_log_new;
			char line[L_LOG_LINE_MAX+1];
			sprintf(fn_log_new, "%s.new", fn_log);
			if ( (fp_log = fopen(fn_log, "r")) == NULL )
				fatal("can't read %s", fn_log);
			if ( (fp_log_new = fopen(fn_log_new, "w")) == NULL )
				fatal("can't rotate logs using %s", fn_log_new);
			while ( n_lines > min_lines && fgets(line, L_LOG_LINE_MAX+1, fp_log) != NULL )
				--n_lines;
			n_lines = 0;
			while ( fgets(line, L_LOG_LINE_MAX+1, fp_log) != NULL )
				{
				fputs(line, fp_log_new);
				++n_lines;
				}
			fclose(fp_log_new);
			fclose(fp_log);
			rename(fn_log_new, fn_log);
			}
	}
/*...e*/
/*...e*/

/* Hashmap support methods */
/*...sint_equal:0:*/
static HM_BOOLEAN int_equal(const void *k1, const void *k2)
	{
	const int *i1 = (const int *) k1;
	const int *i2 = (const int *) k2;
	return *i1 == *i2;
	}
/*...e*/
/*...sint_hashcode:0:*/
static unsigned int_hashcode(const void *k)
	{
	const int *i = (const int *) k;
	return (unsigned) *i;
	}
/*...e*/
/*...sint_clone:0:*/
static void *int_clone(const void *k)
	{
	const int *i = (const int *) k;
	int *ic;
	if ( (ic = (int *) malloc(sizeof(int))) == NULL )
		return NULL;
	*ic = *i;
	return (void *) ic;
	}
/*...e*/
/*...sint_free:0:*/
static void int_free(void *k)
	{
	free(k);
	}
/*...e*/
/*...sulong_clone:0:*/
static void *ulong_clone(const void *v)
	{
	const unsigned long *ul = (const unsigned long *) v;
	unsigned long *ulc;
	if ( (ulc = (unsigned long *) malloc(sizeof(unsigned long))) == NULL )
		return NULL;
	*ulc = *ul;
	return (void *) ulc;
	}
/*...e*/
/*...sulong_free:0:*/
static void ulong_free(void *v)
	{
	free(v);
	}
/*...e*/
/*...sUIDTOT_clone:0:*/
static void *UIDTOT_clone(const void *v)
	{
	const UIDTOT *u = (const UIDTOT *) v;
	UIDTOT *uc;
	if ( (uc = (UIDTOT *) malloc(sizeof(UIDTOT))) == NULL )
		return NULL;
	uc->jiffies = u->jiffies;
	uc->rss     = u->rss    ;
	return (void *) uc;
	}
/*...e*/
/*...sUIDTOT_free:0:*/
static void UIDTOT_free(void *v)
	{
	free(v);
	}
/*...e*/

/*...sparse_mem:0:*/
/* Note that kB actually means KiB */

static unsigned long parse_mem(const char *s)
	{
	unsigned long kb;
	char units[10+1];
	sscanf(s, "%lu %s", &kb, units);
	if ( strcmp(units, "kB") )
		fatal("expected /proc/meminfo to be in kB");
	return kb;
	}
/*...e*/
/*...sget_mem_total:0:*/
/* Returns -1 if we can't get info.
   Result is in KiB. */
static unsigned long get_mem_total(void)
	{
	FILE *fp;
	char line[L_PROC_LINE_MAX+1];
	if ( (fp = fopen("/proc/meminfo", "r")) == NULL )
		return (unsigned long) -1;
	while ( fgets(line, L_PROC_LINE_MAX+1, fp) != NULL )
		if ( !memcmp(line, "MemTotal:", 9) )
			return parse_mem(line+9);
	fclose(fp);
	return (unsigned long) -1;
	}
/*...e*/
/*...sget_mem_process_total:0:*/
/* Returns -1 if we can't get info.
   Work out the total memory is used by processes.
   Or more accurately, work out total memory minus memory that isn't.
   Should I be excluding "PageTables", or are these accounted for in proces RSS?
   Result is in KiB. */
static unsigned long get_mem_process_total(void)
	{
	FILE *fp;
	char line[L_PROC_LINE_MAX+1];
	unsigned long mem_total_kb = (unsigned long) -1;
	unsigned long mem_free_kb  = (unsigned long) -1;
	unsigned long buffers      = (unsigned long) -1;
	unsigned long cached       = (unsigned long) -1;
	unsigned long swap_cached  = (unsigned long) -1;
	unsigned long page_tables  = (unsigned long) -1;
	unsigned long non_process;
	if ( (fp = fopen("/proc/meminfo", "r")) == NULL )
		return (unsigned long) -1;
	while ( fgets(line, L_PROC_LINE_MAX+1, fp) != NULL )
		if ( !memcmp(line, "MemTotal:", 9) )
			mem_total_kb = parse_mem(line+9);
		else if ( !memcmp(line, "MemFree:", 8) )
			mem_free_kb = parse_mem(line+8);
		else if ( !memcmp(line, "Buffers:", 8) )
			buffers = parse_mem(line+8);
		else if ( !memcmp(line, "Cached:", 7) )
			cached = parse_mem(line+7);
		else if ( !memcmp(line, "SwapCached:", 11) )
			swap_cached = parse_mem(line+11);
		else if ( !memcmp(line, "PageTables:", 11) )
			page_tables = parse_mem(line+11);
	fclose(fp);
	if ( mem_total_kb == (unsigned long) -1 ||
	     mem_free_kb  == (unsigned long) -1 ||
	     buffers      == (unsigned long) -1 ||
	     cached       == (unsigned long) -1 ||
	     swap_cached  == (unsigned long) -1 ||
	     page_tables  == (unsigned long) -1 )
		return (unsigned long) -1;
	non_process = mem_free_kb + buffers + cached + swap_cached + page_tables;
	if ( mem_total_kb >= non_process )
		return mem_total_kb - non_process;
	else
		return 0;
	}
/*...e*/
/*...sget_huge_info:0:*/
/* Optimistically try to get this information.
   Results are in KiB. */

static void get_huge_info(
	unsigned long *huge_used,
	unsigned long *huge_free
	)
	{
	FILE *fp;
	char line[L_PROC_LINE_MAX+1];
	unsigned long hp_total = (unsigned long) -1;
	unsigned long hp_free  = (unsigned long) -1;
	unsigned long hp_size  = (unsigned long) -1;
	*huge_used = 0;
	*huge_free = 0;
	if ( (fp = fopen("/proc/meminfo", "r")) == NULL )
		return;
	while ( fgets(line, L_PROC_LINE_MAX+1, fp) != NULL )
		if ( !memcmp(line, "HugePages_Total:", 16) )
			sscanf(line+16, "%lu", &hp_total);
		else if ( !memcmp(line, "HugePages_Free:", 15) )
			sscanf(line+15, "%lu", &hp_free);
		else if ( !memcmp(line, "Hugepagesize:", 13) ) 
			hp_size = parse_mem(line+13);
	fclose(fp);
	if ( hp_total != (unsigned long) -1 &&
	     hp_free  != (unsigned long) -1 &&
	     hp_size  != (unsigned long) -1 )
		{
		*huge_used = (hp_total - hp_free) * hp_size;
		*huge_free =             hp_free  * hp_size;
		}
	}
/*...e*/
/*...sget_uptime_jiffies:0:*/
static unsigned long get_uptime_jiffies(void)
	{
	FILE *fp;
	double seconds;
	if ( (fp = fopen("/proc/uptime", "r")) == NULL )
		fatal("can't open /proc/uptime");
	fscanf(fp, "%lf", &seconds);
	fclose(fp);
	return (unsigned long) ( seconds * sc_clk_tck );
	}
/*...e*/
/*...sget_process_uid:0:*/
/* Returns -1 if we can't get info.
   Can happen if the process recently completed. */

static int get_process_uid(int pid)
	{
	FILE *fp;
	char fn[100+1];
	char line[L_PROC_LINE_MAX+1];
	sprintf(fn, "/proc/%d/status", pid);
	if ( (fp = fopen(fn, "r")) == NULL )
		return -1;
	while ( fgets(line, L_PROC_LINE_MAX+1, fp) != NULL )
		if ( !memcmp(line, "Uid:", 4) )
			{
			int real, effective, saved, fs;
			sscanf(line+4, "%d %d %d %d",
				&real, &effective, &saved, &fs);
			fclose(fp);
			return effective;
			}
	fclose(fp);
	return -1;
	}
/*...e*/
/*...sget_process_jiffies:0:*/
/* Returns -1 if we can't get info.
   Can happen if the process recently completed. */
static unsigned long get_process_jiffies(int pid)
	{
	FILE *fp;
	char fn[100+1];
	int pid2, ppid, pgrp, session, tty_nr, tpgid;
	unsigned long flags; /* only unsigned from Linux 2.6.22 onwards */
	unsigned long minflt, cminflt, majflt, cmajflt, utime, stime;
	char comm[L_PROC_LINE_MAX+1];
	char state;
	sprintf(fn, "/proc/%d/stat", pid);
	if ( (fp = fopen(fn, "r")) == NULL )
		return (unsigned long) -1;
	fscanf(fp, "%d %s %c %d %d %d %d %d %lu %lu %lu %lu %lu %lu %lu",
		&pid2, &comm, &state, &ppid,
		&pgrp, &session, &tty_nr, &tpgid,
		&flags,
		&minflt, &cminflt, &majflt, &cmajflt,
		&utime, &stime
		/* ... many more, but we don't need them */
		);
	fclose(fp);
	return utime + stime;
	}
/*...e*/
/*...sget_process_rss:0:*/
/* Returns -1 if we can't get info.
   Can happen if the process recently completed.
   Values are in pages. */
static unsigned long get_process_rss(int pid)
	{
	FILE *fp;
	char fn[100+1];
	unsigned long size, resident, share, text, lib, data;
	sprintf(fn, "/proc/%d/statm", pid);
	if ( (fp = fopen(fn, "r")) == NULL )
		return (unsigned long) -1;
	fscanf(fp, "%lu %lu %lu %lu %lu %lu",
		&size, &resident, &share, &text, &lib, &data);
	fclose(fp);
	return resident;
	}
/*...e*/
/*...sget_non_idle_jiffies_total:0:*/
/* Returns -1 if we can't get info.
   The cpu line reports a total of jiffies over all processors.
   The first 4 cpu values should always be there.
   How many we get afterwards depends on kernel version.
   We could find we are under-reporting with later kernel versions.
   We just don't want to report on the idle. */
static unsigned long get_non_idle_jiffies_total(void)
	{
	FILE *fp;
	char line[L_PROC_LINE_MAX+1];
	unsigned long user=0, userlow=0, system=0, idle=0;
	unsigned long iowait=0, irq=0, softirq=0, steal=0, guest=0;
	if ( (fp = fopen("/proc/stat", "r")) == NULL )
		return (unsigned long) -1;
	while ( fgets(line, L_PROC_LINE_MAX+1, fp) != NULL )
		if ( !memcmp(line, "cpu ", 4) )
			sscanf(line+4, "%lu %lu %lu %lu %lu %lu %lu %lu %lu",
				&user, &userlow, &system, &idle,
				&iowait, &irq, &softirq, &steal, &guest);
	fclose(fp);
	return user+userlow+system+irq+softirq+steal;
	}
/*...e*/

/*...sshow_timestamp:0:*/
static void show_timestamp(void)
	{
	time_t t = time(NULL);
	struct tm *tm = localtime(&t);
	char timestamp[100+1];
	strftime(timestamp, 100, "%Y %m %d %H %M %S", tm);
	log_line("%s", timestamp);
	}
/*...e*/
/*...sformat_memory:0:*/
static void format_memory(unsigned long mem_kb, char *buf)
	{
	if ( mem_divider == -1 )
		/* Show as a percentage of total memory */
		sprintf(buf, "%3.1lf",
			100.0 * (double) mem_kb / (double) mem_total_kb );
	else
		/* Show in absolute terms */
		sprintf(buf, "%3.1lf",
			(double) mem_kb * 1024.0 / (double) mem_divider );
	}
/*...e*/
/*...sscan_processes:0:*/
static unsigned long jiffies_process;
static unsigned long rss_process;

/*...svisit:0:*/
static void visit(const void *k, const void *v, void *context)
	{
	int uid = *( (const int *) k );
	const UIDTOT *uidtot = (const UIDTOT *) v; 
	unsigned long jiffies_delta = *( (unsigned long *) context );
	struct passwd *p;
	char buf[10+1], *pw_name, buf2[20+1];
	if ( (p = getpwuid((uid_t) uid)) != NULL )
		pw_name = p->pw_name;
	else
		{
		sprintf(buf, "%d", (int) uid);
		pw_name = buf;
		}
	format_memory(uidtot->rss * (sc_pagesize/1024), buf2);
	log_line("\t%s %3.1lf %s",
		pw_name,
		(double) uidtot->jiffies / (double) jiffies_delta * 100.0,
		buf2
		);
	jiffies_process += uidtot->jiffies;
	rss_process     += uidtot->rss;
	}	
/*...e*/

static void scan_processes(
	unsigned long uptime_last, unsigned long non_idle_last, HASHMAP *hm_last,
	unsigned long uptime     , unsigned long non_idle     , HASHMAP *hm
	)
	{
	unsigned long uptime_delta = uptime - uptime_last;
	DIR *d;
	HASHMAP *hm_uids;
	if ( (hm_uids = hm_create(
		int_equal,
		int_hashcode,
		int_clone,
		int_free,
		UIDTOT_clone,
		UIDTOT_free,
		0)) == NULL )
		fatal("out of memory");
	struct dirent *de;
	unsigned long huge_used, huge_free;
	unsigned long nonproc_jiffies;
	unsigned long process_rss_kb;
	unsigned long process_total_kb;
	unsigned long nonproc_kb;
	if ( (d = opendir("/proc")) == NULL )
		fatal("can't open /proc");
	while ( (de = readdir(d)) != NULL )
		if ( isdigit(de->d_name[0]) )
			{
			int pid, uid;
			sscanf(de->d_name, "%d", &pid);
			if ( (uid = get_process_uid(pid)) != -1 )
				{
				unsigned long jiffies;
				if ( (jiffies = get_process_jiffies(pid)) != (unsigned long) -1 )
					{
					unsigned long rss;
					if ( (rss = get_process_rss(pid)) != (unsigned long) -1 )
						{
						unsigned long jiffies_last = 0;
						const unsigned long *p;
						UIDTOT uidtot;
						const UIDTOT *q;
						if ( hm_last != NULL &&
						     (p = (const unsigned long *) hm_get(hm_last, &pid)) != NULL )
							jiffies_last = *p;
						uidtot.jiffies = 0;
						uidtot.rss     = 0;
						if ( (q = (const UIDTOT *) hm_get(hm_uids, &uid)) != NULL )
							uidtot = *q;
						if ( jiffies >= jiffies_last )
							uidtot.jiffies += ( jiffies-jiffies_last );
						else
							/* This could be a process that terminated,
							   and a new one created with the same uid.
							   Unlikely, but possible. */
							uidtot.jiffies += jiffies;
						uidtot.rss += rss;
						if ( ! hm_put(hm_uids, &uid, &uidtot) )
							fatal("out of memory");						
						if ( ! hm_put(hm, &pid, &jiffies) )
							fatal("out of memory");
						}
					}
				}
			}
	closedir(d);
	jiffies_process = 0;
	rss_process = 0; /* in pages */
	hm_visit(hm_uids, visit, &uptime_delta);

	/* Include huge pages */
	get_huge_info(&huge_used, &huge_free);
	if ( huge_used != 0 )
		{
		char buf2[20+1];
		format_memory(huge_used, buf2);
		log_line("\thp_used 0 %s", buf2);
		rss_process += huge_used / (sc_pagesize/1024);
		}
	if ( huge_free != 0 )
		{
		char buf2[20+1];
		format_memory(huge_free, buf2);
		log_line("\thp_free 0 %s", buf2);
		rss_process += huge_free / (sc_pagesize/1024);
		}

	/* Now try to work out how much CPU we missed */
	if ( jiffies_process < non_idle-non_idle_last )
		nonproc_jiffies = (non_idle-non_idle_last) - jiffies_process;
	else
		nonproc_jiffies = 0;

	/* Now try to work out how much memory we missed */
	process_rss_kb = rss_process * (sc_pagesize/1024);
	process_total_kb = get_mem_process_total();
	if ( process_total_kb != (unsigned long) -1 &&
	     process_total_kb > process_rss_kb )
		nonproc_kb = process_total_kb - process_rss_kb;
	else
		nonproc_kb = 0;

	if ( nonproc_jiffies != 0 || nonproc_kb != 0 )
		{
		char buf2[20+1];
		format_memory(nonproc_kb, buf2);
		log_line("\tnonproc %3.1lf %s",
			(double) nonproc_jiffies / (double) uptime_delta * 100.0,
			buf2
			);
		}
	hm_delete(hm_uids);
	}
/*...e*/
/*...swait_a_bit:0:*/
static void wait_a_bit(void)
	{
	struct timespec ts_req, ts_rem;		
	ts_req.tv_sec  = interval;
	ts_req.tv_nsec = 0;
	nanosleep(&ts_req, &ts_rem);
	}
/*...e*/

int main(int argc, char *argv[])
	{
	HASHMAP *hm_last = NULL;
	unsigned long uptime_last = 0;
	unsigned long non_idle_last = 0;

/*...sparse command line:8:*/
{
int i;
for ( i = 1; i < argc; i++ )
	if ( argv[i][0] != '-' )
		break;
	else if ( argv[i][1] == '-' )
		{ ++i; break; }
	else
		switch ( argv[i][1] )
			{
/*...si       \45\ interval:32:*/
case 'i':
	if ( ++i == argc )
		usage();
	sscanf(argv[i], "%i", &interval);
	break;
/*...e*/
/*...sf       \45\ filename:32:*/
case 'f':
	if ( ++i == argc )
		usage();
	fn_log = argv[i];
	break;
/*...e*/
/*...ss       \45\ sizes:32:*/
case 's':
	if ( ++i == argc )
		usage();
	sscanf(argv[i], "%i", &min_lines);
	if ( ++i == argc )
		usage();
	sscanf(argv[i], "%i", &max_lines);
	break;
/*...e*/
/*...sm       \45\ memory units:32:*/
case 'm':
	if ( ++i == argc )
		usage();
	if ( !strcmp(argv[i], "B") )
		mem_divider = 1UL;
	else if ( !strcmp(argv[i], "KB") )
		mem_divider = 1000UL;
	else if ( !strcmp(argv[i], "MB") )
		mem_divider = 1000UL*1000UL;
	else if ( !strcmp(argv[i], "GB") )
		mem_divider = 1000UL*1000UL*1000UL;
	else if ( !strcmp(argv[i], "TB") )
		mem_divider = 1000UL*1000UL*1000UL*1000UL;
	else if ( !strcmp(argv[i], "PB") )
		mem_divider = 1000UL*1000UL*1000UL*1000UL*1000UL;
	else if ( !strcmp(argv[i], "EB") )
		mem_divider = 1000UL*1000UL*1000UL*1000UL*1000UL*1000UL;
	else if ( !strcmp(argv[i], "KiB") )
		mem_divider = 1024UL;
	else if ( !strcmp(argv[i], "MiB") )
		mem_divider = 1024UL*1024UL;
	else if ( !strcmp(argv[i], "GiB") )
		mem_divider = 1024UL*1024UL*1024UL;
	else if ( !strcmp(argv[i], "TiB") )
		mem_divider = 1024UL*1024UL*1024UL*1024UL;
	else if ( !strcmp(argv[i], "PiB") )
		mem_divider = 1024UL*1024UL*1024UL*1024UL*1024UL;
	else if ( !strcmp(argv[i], "EiB") )
		mem_divider = 1024UL*1024UL*1024UL*1024UL*1024UL*1024UL;
	else if ( !strcmp(argv[i], "%") )
		mem_divider = (unsigned long) -1;
	else
		usage();
	break;
/*...e*/
/*...sdefault \45\ error:32:*/
default:
	usage();
/*...e*/
			}
if ( i != argc )
	usage();
if ( min_lines < MIN_MIN_LINES )
	fatal("minimum number of lines can't be smaller than %d", MIN_MIN_LINES);
if ( max_lines > MAX_MAX_LINES )
	fatal("maximum number of lines can't be greater than %d", MAX_MAX_LINES);
if ( max_lines - min_lines < MIN_DIF_LINES )
	fatal("maximum number of lines must be at least %d bigger than minimum number of lines", MIN_DIF_LINES);
}
/*...e*/

	log_init();

	sc_clk_tck  = sysconf(_SC_CLK_TCK);
	sc_pagesize = sysconf(_SC_PAGESIZE);

	if ( (mem_total_kb = get_mem_total()) == (unsigned long) -1 )
		fatal("can't determine total memory in the system");

	/* Do this so that we don't have a huge number first time around. */
	non_idle_last = get_non_idle_jiffies_total();
	if ( non_idle_last == (unsigned long) -1 )
		fatal("can't get basic CPU statistics from /proc/stat");

	for ( ;; )
		{
		HASHMAP *hm;
		unsigned long uptime;
		unsigned long non_idle;
		if ( (hm = hm_create(
			int_equal,
			int_hashcode,
			int_clone,
			int_free,
			ulong_clone,
			ulong_free,
			0)) == NULL )
			fatal("out of memory");
		uptime = get_uptime_jiffies();
		non_idle = get_non_idle_jiffies_total();
		log_begin_batch();
		show_timestamp();
		scan_processes(uptime_last, non_idle_last, hm_last,
		               uptime     , non_idle,      hm     );
		uptime_last = uptime;
		non_idle_last = non_idle;
		if ( hm_last != NULL )
			hm_delete(hm_last);
		hm_last = hm;
		log_end_batch();
		log_prune();
		wait_a_bit();
		}

	return 0;
	}
