//
// TrendServlet.java - Trend stuff
//

//...simports:0:
import java.net.*;
import java.io.*;
import java.util.*;
import java.net.URL;
import java.net.HttpURLConnection;
import java.net.MalformedURLException;
import jakarta.servlet.*;
import jakarta.servlet.http.*;

// Use Mini-AWT to draw basic bitmap, and serialise to PNG
import nyangau.miniawt.*;

// Or use AWT to draw it, and Acme code to serialise to GIF
// import java.awt.*;
// import Acme.JPM.Encoders.GifEncoder;
//...e

public class TrendServlet extends HttpServlet
  {
//...sencodeHTML:2:
// For encoding data within HTML output.
// This doesn't try to be clever.
// It could try harder to map to HTML entities with names, such as &cent;
// It just does the common ones, and falls back to numeric for the rest.

public static String encodeHTML(String s)
  {
  StringBuilder sb = new StringBuilder( s.length() );
  for ( int i = 0; i < s.length(); i++ )
    {
    char c = s.charAt(i);
    switch ( c )
      {
      case '<': sb.append("&lt;"); break;
      case '>': sb.append("&gt;"); break;
      case '&': sb.append("&amp;"); break;
      case '"': sb.append("&quot;"); break;
      default:
        if ( c >= ' ' && c <= '~' )
          // regular printable ASCII character
          sb.append(c);
        else
          // use numeric entity notation for the rest
          {
          sb.append("&#");
          sb.append((int)c);
          sb.append(";");
          }
        break;
      }
    }
  return sb.toString();
  }
//...e
//...scloneCalendar:2:
// Work around IBM JDK 1.4 bug with Calendar class - if we clone a Calendar and
// manipulate the clone, the original can be affected. Invoking the get method
// on the original seems to ensure it has its own distinct internal state.

protected Object _o_clone_bug = new Object();
protected int _clone_bug;
protected Calendar cloneCalendar(Calendar c)
  {
  Calendar c2 = (Calendar) c.clone();
  synchronized ( _o_clone_bug )
    {
    _clone_bug = c.get(Calendar.DAY_OF_MONTH);
    }
  return c2;
  }
//...e
//...smatches:2:
// The pattern may contain *, which means zero or more characters of any kind.
// May also contain ?, which means any single character.
// Recursively walk down the pattern, matching the text.

protected static boolean matches(String text, String pattern)
  {
  if ( pattern.equals("") )
    // We've got a match only if there is no more text
    return text.equals("");
  else if ( pattern.charAt(0) == '*' )
    // Consider matching nothing or any character
    return matches(text, pattern.substring(1)) ||
           ( ! text.equals("") && matches(text.substring(1), pattern) );
  else if ( pattern.charAt(0) == '?' )
    // Match any character
    return ! text.equals("") &&
           matches(text.substring(1), pattern.substring(1));
  else
    // There must be at least one character of text and it must match,
    // and whatever follows must match also
    return ! text.equals("") && pattern.charAt(0) == text.charAt(0) &&
           matches(text.substring(1), pattern.substring(1));
  }

// Look through a vector of patterns and return index of first match

protected static int matches(String text, Vector<String> patterns)
  {
  for ( int i = 0; i < patterns.size(); i++ )
    if ( matches(text, patterns.elementAt(i)) )
      return i;
  return -1;
  }
//...e
//...sgraphic attributes:2:
static final int minW =  100;
static final int defW =  800;
static final int maxW = 4000;
static final int minH =   50;
static final int defH =  200;
static final int maxH = 2000;
static final Color   colorBitmap    = Color.white;
static final Color   colorGraph     = Color.lightGray;
static final Color   colorGrid      = Color.gray;
static final Color   colorText      = Color.black;
static final Color   colorTitleBg   = new Color(0xcc, 0xcc, 0xff);
static final Color   colorSqueezed  = Color.red;
static final Color[] colorConsumers =
  {
  // light
  new Color(0xcc,0x00,0x00), // red
  new Color(0x00,0xcc,0xcc), // cyan
  new Color(0x00,0xcc,0x00), // green
  new Color(0xcc,0x00,0xcc), // magenta
  new Color(0x00,0x00,0xcc), // blue
  new Color(0xcc,0xcc,0x00), // yellow
  // dark
  new Color(0x66,0x00,0x00), // red
  new Color(0x00,0x66,0x66), // cyan
  new Color(0x00,0x66,0x00), // green
  new Color(0x66,0x00,0x66), // magenta
  new Color(0x00,0x00,0x66), // blue
  new Color(0x66,0x66,0x00), // yellow
  };
static final Color[] colorConsumersExceed =
  {
  // light tint
  new Color(0xff,0x66,0x66), // red
  new Color(0x66,0xff,0xff), // cyan
  new Color(0x66,0xff,0x66), // green
  new Color(0xff,0x66,0xff), // magenta
  new Color(0x66,0x66,0xff), // blue
  new Color(0xff,0xff,0x66), // yellow
  // dark tint
  new Color(0xff,0x33,0x33), // red
  new Color(0x33,0xff,0xff), // cyan
  new Color(0x33,0xff,0x33), // green
  new Color(0xff,0x33,0xff), // magenta
  new Color(0x33,0x33,0xff), // blue
  new Color(0xff,0xff,0x33), // yellow
  };
//...e
  static final int mintime = 60*1000; // 1 minute
  Properties props;
  String name;
  String doclinkname;
  String doclinkurl;
  Vector<String> ds_all; int nDatasources;
  Vector<String>  r_all; int nResources;
  float multipliers[];
//...sclass Sample:2:
class Sample
  {
  public Calendar cal;
  public float[][] values;
  public Sample(Calendar c, int nConsumers)
    {
    cal = c;
    values = new float[nConsumers][nResources];
      // they'll all be 0.0f to start with
    }
  }
//...e
  Dictionary<String,Vector<Sample>> samplesHash = new Hashtable<String,Vector<Sample>>(); 
//...stoInt:2:
protected int toInt(String s, int def)
  {
  if ( s == null )
    return def;
  try
    {
    return Integer.parseInt(s);
    }
  catch ( NumberFormatException nfe )
    {
    return def;
    }
  }
//...e
//...stoFloat:2:
protected float toFloat(String s, float def)
  {
  if ( s == null )
    return def;
  try
    {
    return Float.parseFloat(s);
    }
  catch ( NumberFormatException nfe )
    {
    return def;
    }
  }
//...e
//...sgetInitParameter:2:
protected String getInitParameter(String p)
  {
  if ( props != null )
    {
    String v;
    if ( (v = props.getProperty(p)) != null )
      return v;
    }
  ServletConfig sc = getServletConfig();
  return sc.getInitParameter(p);
  }
//...e
//...sgetInitParameter:2:
protected String getInitParameter(String p, String def)
  {
  String v;
  if ( (v = getInitParameter(p)) != null )
    return v;
  else
    return def;
  }
//...e
//...sgetInitParameterVector:2:
protected Vector<String> getInitParameterVector(String p)
  {
  String v;
  Vector<String> vec = new Vector<String>();
  if ( (v = getInitParameter(p)) != null )
    {
    StringTokenizer st = new StringTokenizer(v, ",");
    while ( st.hasMoreTokens() )
      vec.add(st.nextToken());
    }
  return vec;
  }
//...e
//...scal_to_str:2:
protected static String cal_to_str(Calendar cal)
  {
  int year   = cal.get(Calendar.YEAR);
  int month  = cal.get(Calendar.MONTH) - Calendar.JANUARY + 1;
  int day    = cal.get(Calendar.DAY_OF_MONTH);
  int hour   = cal.get(Calendar.HOUR_OF_DAY);
  int minute = cal.get(Calendar.MINUTE);
  int second = cal.get(Calendar.SECOND);
  return
    year+"-"+
    ((month <10)?"0":"")+month+"-"+
    ((day   <10)?"0":"")+day+"-"+
    ((hour  <10)?"0":"")+hour+"-"+
    ((minute<10)?"0":"")+minute+"-"+
    ((second<10)?"0":"")+second;
  }
//...e
//...scal_to_str_csv:2:
// YYYY-MM-DD HH:MM:SS

protected static String cal_to_str_csv(Calendar cal)
  {
  int year   = cal.get(Calendar.YEAR);
  int month  = cal.get(Calendar.MONTH) - Calendar.JANUARY + 1;
  int day    = cal.get(Calendar.DAY_OF_MONTH);
  int hour   = cal.get(Calendar.HOUR_OF_DAY);
  int minute = cal.get(Calendar.MINUTE);
  int second = cal.get(Calendar.SECOND);
  return
    year+"-"+
    ((month <10)?"0":"")+month+"-"+
    ((day   <10)?"0":"")+day+" "+
    ((hour  <10)?"0":"")+hour+":"+
    ((minute<10)?"0":"")+minute+":"+
    ((second<10)?"0":"")+second;
  }
//...e
//...sstr_to_cal:2:
protected static Calendar str_to_cal(String s, String delims)
  {
  StringTokenizer st = new StringTokenizer(s, delims);
  Calendar cal = Calendar.getInstance();
  try
    {
    int year   = Integer.parseInt(st.nextToken());
    int month  = Integer.parseInt(st.nextToken()); 
    int day    = Integer.parseInt(st.nextToken()); 
    int hour   = Integer.parseInt(st.nextToken()); 
    int minute = Integer.parseInt(st.nextToken()); 
    int second = Integer.parseInt(st.nextToken()); 
    cal.clear();
    cal.set(year, month-1+Calendar.JANUARY, day, hour, minute, second);
    return cal;
    }
  catch ( NumberFormatException nfe )
    {
    return null;
    }
  catch ( NoSuchElementException nsee )
    {
    return null;
    }
  }
//...e
//...sstr_to_cal:2:
protected static Calendar str_to_cal(String s)
  {
  return str_to_cal(s, "-");
  }
//...e
//...srad_to_cal:2:
protected static Calendar rad_to_cal(Calendar cal, String s)
  {
       if ( s.equals("n") )
    ;
  else if ( s.equals("h") )
    cal.add(Calendar.HOUR_OF_DAY, -1);
  else if ( s.equals("d") )
    cal.add(Calendar.DAY_OF_MONTH, -1);
  else if ( s.equals("w") )
    cal.add(Calendar.DAY_OF_MONTH, -7);
  else if ( s.equals("m") )
    cal.add(Calendar.MONTH, -1);
  else if ( s.equals("q") )
    cal.add(Calendar.MONTH, -3);
  else if ( s.equals("y") )
    cal.add(Calendar.YEAR, -1);
  else
    return null;
  return cal;
  }
//...e
//...sdsStream:2:
protected InputStream dsStream(String ds)
  throws IOException
  {
  String fn = getInitParameter("datasources."+ds);
  if ( fn == null )
    throw new IOException("no datasources."+ds+" property");
  if ( fn.startsWith("http:") )
//...sHTTP fetch:6:
{
try
  {
  URL url = new URL(fn);
  HttpURLConnection uc = (HttpURLConnection) url.openConnection();
  if ( uc.getContentLength() != 0 )
    return uc.getInputStream();
  else
    throw new IOException("No content");
  }
catch ( MalformedURLException ex )
  {
  throw new IOException("Bad URL: " + ex.getMessage());
  }
}
//...e
  else
//...sopen a local file:6:
{
File f = new File(fn);
FileInputStream fis = new FileInputStream(f);
return fis;
}
//...e
  }
//...e
//...ssendHeader:2:
protected void sendHeader(HttpServletResponse resp, String type)
  throws IOException
  {
  resp.setContentType("text/html");
  ServletOutputStream os = resp.getOutputStream();
  String title = "Trend Server";
  if ( name != null )
    title += ( " - " + encodeHTML(name) );
  if ( type != null )
    title += ( " - " + encodeHTML(type) );
  os.println("<HTML>");
  os.println("<HEAD>");
  os.println("<TITLE>"+title+"</TITLE>");
  os.println("<LINK REL=\"stylesheet\" TYPE=\"text/css\" HREF=\"default.css\" MEDIA=\"all\">");
  os.println("</HEAD>");
  os.println("<BODY>");
  os.println("<BASEFONT SIZE=3>");
  os.println("<TABLE><TR><TH><FONT SIZE=+2>"+title+"</FONT></TABLE>");
  os.println();
  }
//...e
//...ssendTrailer:2:
protected static void sendTrailer(HttpServletResponse resp)
  throws IOException
  {
  ServletOutputStream os = resp.getOutputStream();
  os.println();
  os.println("</BODY>");
  os.println("</HTML>");
  }
//...e
//...ssendUsage:2:
protected void sendUsage(
  HttpServletRequest req, HttpServletResponse resp
  )
  throws IOException
  {
  sendHeader(resp, "usage");
  ServletOutputStream os = resp.getOutputStream();
  os.println("<P>TrendServlet can be used to trend information");
  os.println("<P>Usage:");
  os.println("<P>");
  os.println("<PRE>");
  String uri = req.getRequestURI();
  os.println("  http://hostname"+encodeHTML(uri)+"[?options]");
  os.println("</PRE>");
  os.println("<P>Where options include:");
  os.println("<P>");
  os.println("<UL>");
  os.println("<LI><CODE>a</CODE> is the action (main, graph, samples or samples.csv, main is the default)");
  os.println("<LI><CODE>ds</CODE> is the data source(s)");
  os.println("<LI><CODE>r</CODE> is the resource(s)");
  os.println("<LI><CODE>st</CODE> is the start time (default to a day before end time)");
  os.println("<LI><CODE>et</CODE> is the end time (default is now)");
  os.println("<LI><CODE>w</CODE> is graph width");
  os.println("<LI><CODE>h</CODE> is graph height");
  os.println("</UL>");
  sendTrailer(resp);
  }
//...e
//...ssendError:2:
protected void sendError(HttpServletResponse resp, String error)
  throws IOException
  {
  sendHeader(resp, "error");
  ServletOutputStream os = resp.getOutputStream();
  os.println("<P>"+encodeHTML(error));
  sendTrailer(resp);
  }
//...e
//...ssendSamples:2:
// Send samples from a given datasource.
// Just send the file in raw text format.
// This allows folks to serve this content using a regular web server.
// This is useful when we can't put TradeServlet on the monitored system.

protected void sendSamples(
  HttpServletRequest req, HttpServletResponse resp,
  String[] ds_a
  )
  throws IOException
  {
  if ( ds_a == null || ds_a.length != 1 )
    {
    sendError(resp, "ds=datasource parameter required");
    return;
    }
  String ds = ds_a[0];
  try
    {
    InputStream is = dsStream(ds);
    InputStreamReader isr = new InputStreamReader(is);
    BufferedReader br = new BufferedReader(isr);
    ServletOutputStream os = resp.getOutputStream();
    resp.setContentType("text/plain");
    String line;
    while ( (line = br.readLine()) != null )
      os.println(line);
    br.close();
    }
  catch ( IOException ioe )
    {
    resp.sendError(HttpServletResponse.SC_INTERNAL_SERVER_ERROR);
    }
  }
//...e
//...ssendSamplesCsv:2:
//...ssanitize:2:
protected static String sanitize(String s)
  {
  StringBuilder sb = new StringBuilder( s.length() );
  for ( int i = 0; i < s.length(); i++ )
    {
    char c = s.charAt(i);
    if ( Character.isLetter(c) || Character.isDigit(c) )
      sb.append(c);
    }
  return sb.toString();
  }
//...e

protected void sendSamplesCsv(
  HttpServletResponse resp,
  String[] ds_a, String[] r_a,
  Calendar st_cal, Calendar et_cal
  )
  throws IOException
  {
//...sget\47\validate parameters:4:
if ( ds_a == null || ds_a.length != 1 ||
      r_a == null ||  r_a.length != 1 )
  {
  sendError(resp, "ds=datasource and r=resource parameters required");
  return;
  }
String ds = ds_a[0];
String  r =  r_a[0];
int inx = r_all.indexOf(r);
if ( inx == -1 )
  {
  sendError(resp, "Unrecognised resource: "+r);
  return;
  }

long st_m = st_cal.getTime().getTime();
long et_m = et_cal.getTime().getTime();
if ( st_m + mintime > et_m )
  {
  sendError(resp, "Timespan must cover at least one minute");
  return;
  }
//...e

  resp.setContentType("application/csv");
  resp.setHeader("Content-Disposition", "attachment; filename=\""+sanitize(ds)+"_"+sanitize(r)+"_"+cal_to_str(st_cal)+"_"+cal_to_str(et_cal)+".csv\"");
  ServletOutputStream os = resp.getOutputStream();

  // Get consumers applicable to this datasource
  Vector<String> c_all = getInitParameterVector("consumers."+ds);
  int nConsumers = c_all.size();
  // Get consumers we're not interested in
  Vector<String> c_ignore = getInitParameterVector("consumers-ignore."+ds);
  // Sampling interval for this datasource, in milliseconds

  // title row
  os.print("datetime");
  for ( String c : c_all )
    os.print(","+c);
  os.println();

  synchronized ( this )
    {
    Vector<Sample> samples;
    try
      {
      samples = getSamples(ds, c_all, c_ignore);
      }
    catch ( IOException ioe )
      {
      return;
      }
    int sFirst = locateAfter(samples, st_cal);
    int sLast  = locateAfter(samples, et_cal);
    if ( sFirst >                0 ) --sFirst;
    if ( sLast  < samples.size()-2 ) ++sLast ;
    for ( int s = sFirst; s < sLast-1; s++ )
      {
      // Work out time for sample
      Sample s1 = samples.elementAt(s);
      os.print(cal_to_str_csv(s1.cal));
      for ( int c = 0; c < nConsumers; c++ )
        os.print(","+s1.values[c][inx]);
      os.println();
      }
    }
  }
//...e
//...sreadSamples:2:
// Read a samples file/stream producing a Vector of Samples
// We tolerate files which don't start with a timestamp, as this allows the
// scripting that generates the file to be crude when it comes to log
// truncation or rotation, ie: they can simply use something like 'tail -1000'.
// We also tolerate consumer lines which don't cover all the resources.

protected Vector<Sample> readSamples(
  InputStream is,
  Vector<String> c_all,
  Vector<String> c_ignore
  )
  throws IOException
  {
  InputStreamReader isr = new InputStreamReader(is);
  BufferedReader br = new BufferedReader(isr);

  Vector<Sample> samples = new Vector<Sample>(1000, 1000);

  String line;

  // Skip until first timestamp
  line = br.readLine();
  while ( line != null && line.charAt(0) == '\t' )
    line = br.readLine();

  int ic_other = c_all.indexOf("other");
  int nConsumers = c_all.size();

  // Record all samples
  while ( line != null )
    {
    Calendar cal = str_to_cal(line, " \t");
    if ( cal != null )
      {
      Sample sample = new Sample(cal, nConsumers);
      line = br.readLine();
      while ( line != null && line.charAt(0) == '\t' )
        {
        StringTokenizer st = new StringTokenizer(line);
        String c = st.nextToken();
        int ic;
        if ( matches(c, c_ignore) != -1 )
          ; // ignore it
        else if ( (ic = matches(c, c_all)) != -1 || (ic = ic_other) != -1 )
          // We know about this consumer
          for ( int i = 0; i < nResources && st.hasMoreTokens(); i++ )
            sample.values[ic][i] += toFloat(st.nextToken(), 0.0f) * multipliers[i];
        line = br.readLine();
        }
      samples.add(sample);
      }
    else
      // Bad timestamp in input data, skip it and any data
      {
      line = br.readLine();
      while ( line != null && line.charAt(0) == '\t' )
        line = br.readLine();
      }
    }
  return samples;
  }
//...e
//...sgetSamples:2:
// Attempt to locate a Vector of samples for the datasource
// If not got the info cached in memory already, open and read it

protected Vector<Sample> getSamples(
  String ds,
  Vector<String> c_all,
  Vector<String> c_ignore
  )
  throws IOException
  {
  Vector<Sample> samples;
  if ( (samples = samplesHash.get(ds)) == null )
    // Read in some new data, covering the range requested
    try
      (
      InputStream is = dsStream(ds);
      )
      {
      samples = readSamples(is, c_all, c_ignore);
      samplesHash.put(ds, samples);
      }
  return samples;
  }
//...e
//...sdropSamples:2:
protected void dropSamples(String ds)
  {
  samplesHash.remove(ds);
  }
//...e
//...ssendGraph:2:
// Return a graph of the data from a datasource, showing a resource.
// Graph samples within a timerange, producing a graph of size w x h.

//...sdivWidth:2:
protected int divWidth(
  Calendar cal, long st_m, long et_m,
  int field,
  int xs
  )
  {
  Calendar cal2 = cloneCalendar(cal);
  cal2.add(field, 1);
  long m = cal2.getTime().getTime();
  return (int) ( (xs*(m-st_m))/(et_m-st_m) );
  }
//...e
//...sdivSeconds\44\ valSeconds:2:
protected void divSeconds(
  Graphics g,
  int xo, int yo, int xs, int ys,
  Calendar st_cal, long st_m, Calendar et_cal, long et_m,
  int step, int lw
  )
  {
  Calendar cal = cloneCalendar(st_cal);
  cal.add(Calendar.SECOND, 1);
  g.setColor(colorGrid);
  for ( ; cal.before(et_cal); cal.add(Calendar.SECOND, 1) )
    if ( (cal.get(Calendar.SECOND)%step) == 0 )
      {
      long m = cal.getTime().getTime();
      int x = (int) ( (xs*(m-st_m))/(et_m-st_m) );
      g.fillRect(xo+x, yo, lw, ys);
      }
  }

protected void valSeconds(
  Graphics g,
  int xo, int yo, int xs, int ys,
  Calendar st_cal, long st_m, Calendar et_cal, long et_m,
  int step
  )
  {
  Calendar cal = cloneCalendar(st_cal);
  cal.add(Calendar.SECOND, 1);
  g.setColor(colorText);
  for ( ; cal.before(et_cal); cal.add(Calendar.SECOND, 1) )
    if ( (cal.get(Calendar.SECOND)%step) == 0 )
      {
      long m = cal.getTime().getTime();
      int x = (int) ( (xs*(m-st_m))/(et_m-st_m) );
      int s = cal.get(Calendar.SECOND);
      g.drawString(
        ((s<10)?"0":"")+
        Integer.toString(s)+"s",
        xo+x, yo+ys);
      }
  }
//...e
//...sdivMinutes\44\ valMinutes:2:
protected void divMinutes(
  Graphics g,
  int xo, int yo, int xs, int ys,
  Calendar st_cal, long st_m, Calendar et_cal, long et_m,
  int step, int lw
  )
  {
  Calendar cal = cloneCalendar(st_cal);
  cal.add(Calendar.MINUTE, 1);
  cal.add(Calendar.SECOND, -1);
  cal.set(Calendar.SECOND, 0);
  g.setColor(colorGrid);
  for ( ; cal.before(et_cal); cal.add(Calendar.MINUTE, 1) )
    if ( (cal.get(Calendar.MINUTE)%step) == 0 )
      {
      long m = cal.getTime().getTime();
      int x = (int) ( (xs*(m-st_m))/(et_m-st_m) );
      g.fillRect(xo+x, yo, lw, ys);
      }
  }

protected void valMinutes(
  Graphics g,
  int xo, int yo, int xs, int ys,
  Calendar st_cal, long st_m, Calendar et_cal, long et_m,
  int step
  )
  {
  Calendar cal = cloneCalendar(st_cal);
  cal.add(Calendar.MINUTE, 1);
  cal.add(Calendar.SECOND, -1);
  cal.set(Calendar.SECOND, 0);
  g.setColor(colorText);
  for ( ; cal.before(et_cal); cal.add(Calendar.MINUTE, 1) )
    if ( (cal.get(Calendar.MINUTE)%step) == 0 )
      {
      long m = cal.getTime().getTime();
      int x = (int) ( (xs*(m-st_m))/(et_m-st_m) );
      int moh = cal.get(Calendar.MINUTE);
      g.drawString(
        ((moh<10)?"0":"")+
        Integer.toString(moh)+"m",
        xo+x, yo+ys);
      }
  }
//...e
//...sdivHours  \44\ valHours:2:
protected void divHours(
  Graphics g,
  int xo, int yo, int xs, int ys,
  Calendar st_cal, long st_m, Calendar et_cal, long et_m,
  int step, int lw
  )
  {
  Calendar cal = cloneCalendar(st_cal);
  cal.add(Calendar.HOUR_OF_DAY, 1);
  cal.add(Calendar.SECOND, -1);
  cal.set(Calendar.MINUTE, 0);
  cal.set(Calendar.SECOND, 0);
  g.setColor(colorGrid);
  for ( ; cal.before(et_cal); cal.add(Calendar.HOUR_OF_DAY, 1) )
    if ( (cal.get(Calendar.HOUR_OF_DAY)%step) == 0 )
      {
      long m = cal.getTime().getTime();
      int x = (int) ( (xs*(m-st_m))/(et_m-st_m) );
      g.fillRect(xo+x, yo, lw, ys);
      }
  }

protected void valHours(
  Graphics g,
  int xo, int yo, int xs, int ys,
  Calendar st_cal, long st_m, Calendar et_cal, long et_m,
  int step
  )
  {
  Calendar cal = cloneCalendar(st_cal);
  cal.add(Calendar.HOUR_OF_DAY, 1);
  cal.add(Calendar.SECOND, -1);
  cal.set(Calendar.MINUTE, 0);
  cal.set(Calendar.SECOND, 0);
  g.setColor(colorText);
  for ( ; cal.before(et_cal); cal.add(Calendar.HOUR_OF_DAY, 1) )
    if ( (cal.get(Calendar.HOUR_OF_DAY)%step) == 0 )
      {
      long m = cal.getTime().getTime();
      int x = (int) ( (xs*(m-st_m))/(et_m-st_m) );
      int hod = cal.get(Calendar.HOUR_OF_DAY);
      g.drawString(
        ((hod<10)?"0":"")+
        Integer.toString(hod)+"h",
        xo+x, yo+ys);
      }
  }
//...e
//...sdivDays   \44\ valDays:2:
protected void divDays(
  Graphics g,
  int xo, int yo, int xs, int ys,
  Calendar st_cal, long st_m, Calendar et_cal, long et_m,
  int step, int lw
  )
  {
  Calendar cal = cloneCalendar(st_cal);
  cal.add(Calendar.DAY_OF_MONTH, 1);
  cal.add(Calendar.SECOND, -1);
  cal.set(Calendar.HOUR_OF_DAY, 0);
  cal.set(Calendar.MINUTE, 0);
  cal.set(Calendar.SECOND, 0);
  g.setColor(colorGrid);
  for ( ; cal.before(et_cal); cal.add(Calendar.DAY_OF_MONTH, 1) )
    if ( ((cal.get(Calendar.DAY_OF_MONTH)-1)%step) == 0 )
      {
      long m = cal.getTime().getTime();
      int x = (int) ( (xs*(m-st_m))/(et_m-st_m) );
      g.fillRect(xo+x, yo, lw, ys);
      }
  }

protected void valDays(
  Graphics g,
  int xo, int yo, int xs, int ys,
  Calendar st_cal, long st_m, Calendar et_cal, long et_m,
  int step
  )
  {
  Calendar cal = cloneCalendar(st_cal);
  cal.add(Calendar.DAY_OF_MONTH, 1);
  cal.add(Calendar.SECOND, -1);
  cal.set(Calendar.HOUR_OF_DAY, 0);
  cal.set(Calendar.MINUTE, 0);
  cal.set(Calendar.SECOND, 0);
  g.setColor(colorText);
  for ( ; cal.before(et_cal); cal.add(Calendar.DAY_OF_MONTH, 1) )
    if ( ((cal.get(Calendar.DAY_OF_MONTH)-1)%step) == 0 )
      {
      long m = cal.getTime().getTime();
      int x = (int) ( (xs*(m-st_m))/(et_m-st_m) );
      int dom = cal.get(Calendar.DAY_OF_MONTH);
      String suffix;
      switch ( dom )
        {
        case 1: case 21: case 31: suffix = "st"; break;
        case 2: case 22:          suffix = "nd"; break;
        case 3: case 23:          suffix = "rd"; break;
        default:                  suffix = "th"; break;
        }
      g.drawString(
        Integer.toString(dom)+suffix,
        xo+x, yo+ys);
      }
  }
//...e
//...sdivMonths \44\ valMonths:2:
protected void divMonths(
  Graphics g,
  int xo, int yo, int xs, int ys,
  Calendar st_cal, long st_m, Calendar et_cal, long et_m,
  int step, int lw
  )
  {
  Calendar cal = cloneCalendar(st_cal);
  cal.add(Calendar.MONTH, 1);
  cal.add(Calendar.SECOND, -1);
  cal.set(Calendar.DAY_OF_MONTH, 1);
  cal.set(Calendar.HOUR_OF_DAY, 0);
  cal.set(Calendar.MINUTE, 0);
  cal.set(Calendar.SECOND, 0);
  g.setColor(colorGrid);
  for ( ; cal.before(et_cal); cal.add(Calendar.MONTH, 1) )
    if ( ((cal.get(Calendar.MONTH)-Calendar.JANUARY)%step) == 0 )
      {
      long m = cal.getTime().getTime();
      int x = (int) ( (xs*(m-st_m))/(et_m-st_m) );
      g.fillRect(xo+x, yo, lw, ys);
      }
  }

protected void valMonths(
  Graphics g,
  int xo, int yo, int xs, int ys,
  Calendar st_cal, long st_m, Calendar et_cal, long et_m,
  int step
  )
  {
  Calendar cal = cloneCalendar(st_cal);
  cal.add(Calendar.MONTH, 1);
  cal.add(Calendar.SECOND, -1);
  cal.set(Calendar.DAY_OF_MONTH, 1);
  cal.set(Calendar.HOUR_OF_DAY, 0);
  cal.set(Calendar.MINUTE, 0);
  cal.set(Calendar.SECOND, 0);
  g.setColor(colorText);
  String[] months = { "Jan", "Feb", "Mar", "Apr", "May", "Jun",
                      "Jul", "Aug", "Sep", "Oct", "Nov", "Dec" };
  for ( ; cal.before(et_cal); cal.add(Calendar.MONTH, 1) )
    if ( ((cal.get(Calendar.MONTH)-Calendar.JANUARY)%step) == 0 )
      {
      long m = cal.getTime().getTime();
      int x = (int) ( (xs*(m-st_m))/(et_m-st_m) );
      g.drawString(
        months[cal.get(Calendar.MONTH)-Calendar.JANUARY],
        xo+x, yo+ys);
      }
  }
//...e
//...sdivYears  \44\ valYears:2:
protected void divYears(
  Graphics g,
  int xo, int yo, int xs, int ys,
  Calendar st_cal, long st_m, Calendar et_cal, long et_m,
  int step, int lw
  )
  {
  Calendar cal = cloneCalendar(st_cal);
  cal.add(Calendar.YEAR, 1);
  cal.add(Calendar.SECOND, -1);
  cal.set(Calendar.MONTH, Calendar.JANUARY);
  cal.set(Calendar.DAY_OF_MONTH, 1);
  cal.set(Calendar.HOUR_OF_DAY, 0);
  cal.set(Calendar.MINUTE, 0);
  cal.set(Calendar.SECOND, 0);
  g.setColor(colorGrid);
  for ( ; cal.before(et_cal); cal.add(Calendar.YEAR, 1) )
    if ( (cal.get(Calendar.MONTH)%step) == 0 )
      {
      long m = cal.getTime().getTime();
      int x = (int) ( (xs*(m-st_m))/(et_m-st_m) );
      g.fillRect(xo+x, yo, lw, ys);
      }
  }

protected void valYears(
  Graphics g,
  int xo, int yo, int xs, int ys,
  Calendar st_cal, long st_m, Calendar et_cal, long et_m,
  int step
  )
  {
  Calendar cal = cloneCalendar(st_cal);
  cal.add(Calendar.YEAR, 1);
  cal.add(Calendar.SECOND, -1);
  cal.set(Calendar.MONTH, Calendar.JANUARY);
  cal.set(Calendar.DAY_OF_MONTH, 1);
  cal.set(Calendar.HOUR_OF_DAY, 0);
  cal.set(Calendar.MINUTE, 0);
  cal.set(Calendar.SECOND, 0);
  g.setColor(colorText);
  for ( ; cal.before(et_cal); cal.add(Calendar.YEAR, 1) )
    if ( (cal.get(Calendar.MONTH)%step) == 0 )
      {
      long m = cal.getTime().getTime();
      int x = (int) ( (xs*(m-st_m))/(et_m-st_m) );
      g.drawString(
        Integer.toString(cal.get(Calendar.YEAR)),
        xo+x, yo+ys);
      }
  }
//...e

//...slocateAfter:2:
// Use a simple binary search to locate index of first sample in the samples
// vector which has a calendar value >= the cal parameter passed.

protected int locateAfter(Vector<Sample> samples, Calendar cal)
  {
  int iLo = 0;
  int iHi = samples.size();
  while ( iLo < iHi )
    {
    int iMid = (iLo+iHi)/2;
    Sample sMid = samples.elementAt(iMid);
         if ( cal.before(sMid.cal) )
      iHi = iMid;
    else if ( cal.after (sMid.cal) )
      iLo = iMid+1;
    else
      return iMid;
    }
  return iLo;
  }
//...e

protected void sendGraph(
  HttpServletRequest req, HttpServletResponse resp,
  String[] ds_a, String[] r_a,
  Calendar st_cal, Calendar et_cal,
  boolean split, boolean scale, boolean squeeze,
  int w, int h
  )
  throws IOException
  {
//...sget\47\validate parameters:4:
if ( ds_a == null || ds_a.length != 1 ||
      r_a == null ||  r_a.length != 1 )
  {
  sendError(resp, "ds=datasource and r=resource parameters required");
  return;
  }
String ds = ds_a[0];
String  r =  r_a[0];
int inx = r_all.indexOf(r);
if ( inx == -1 )
  {
  sendError(resp, "Unrecognised resource: "+r);
  return;
  }

long st_m = st_cal.getTime().getTime();
long et_m = et_cal.getTime().getTime();
if ( st_m + mintime > et_m )
  {
  sendError(resp, "Timespan must cover at least one minute");
  return;
  }
//...e

  // Get consumers applicable to this datasource
  Vector<String> c_all = getInitParameterVector("consumers."+ds);
  int nConsumers = c_all.size();

  // Get consumers we're not interested in
  Vector<String> c_ignore = getInitParameterVector("consumers-ignore."+ds);

  // How much resources in total
  String tot;
  if ( (tot = getInitParameter("resources."+ds+"."+r+".total")) == null &&
       (tot = getInitParameter("resources."       +r+".total")) == null )
    tot = "";
  float total = toFloat(tot, 100.0f);

  // The units
  String units;
  if ( (units = getInitParameter("resources."+ds+"."+r+".units")) == null &&
       (units = getInitParameter("resources."       +r+".units")) == null )
    units = "";

  // Sampling interval for this datasource, in milliseconds
  int interval = toInt(getInitParameter("interval."+ds), 65) * 1000;

  Frame fr = new Frame();
  fr.addNotify();
  Image image = fr.createImage(w, h);
  Graphics g = image.getGraphics();

  // Border sizes
  int bw = 4, bh = 4;

  // We need to get a rough estimate of how wide each character will be
  // We'll use this for our ad-hoc titling
  FontMetrics fm = g.getFontMetrics();
  String adhoc = "0123456789abcdefghijklmnopqrstuvwxyzABCDEFGHIJKLMNOPQRSTUVWXYZ";
  int cw = fm.stringWidth(adhoc)/(10+26+26);
  int ch = fm.getHeight();

  // Title size
  int th = ch+bh;

  // Work out width of margin needed for consumer names
  int nw = 0;
  for ( int c = 0; c < nConsumers; c++ )
    {
    int nw2 = fm.stringWidth(c_all.elementAt(c));
    if ( nw2 > nw )
      nw = nw2;
    }

  synchronized ( this )
    {
    Vector<Sample> samples;
    try
      {
      samples = getSamples(ds, c_all, c_ignore);
      }
    catch ( IOException ioe )
      {
      sendError(resp, ioe.getMessage());
      return;
      }
    int sFirst = locateAfter(samples, st_cal);
    int sLast  = locateAfter(samples, et_cal);
    if ( sFirst >                0 ) --sFirst;
    if ( sLast  < samples.size()-2 ) ++sLast ;

    if ( scale || total < 0.0f )
//...sdetermine a suitable scale\44\ by looking at the samples:8:
{
total = 0.0f;
for ( int s = sFirst; s < sLast-1; s++ )
  {
  Sample s1 = samples.elementAt(s);
  float y = 0.0f;
  for ( int c = 0; c < nConsumers; c++ )
    y += s1.values[c][inx];
  total = Math.max(total, y);
  }
}
//...e
//...sentitlement calculations\44\ which may adjust the total:6:
// Determine entitlements for each consumer, given datasource and resource
float[] ents = new float[nConsumers];
for ( int c = 0; c < nConsumers; c++ )
  {
  String c2 = c_all.elementAt(c);
  String e;
  if ( (e = getInitParameter("entitlement."+ds+"."+r+"."+c2)) == null &&
       (e = getInitParameter("entitlement."       +r+"."+c2)) == null )
    e = "";
  ents[c] = toFloat(e, -1.0f);
  }

// Sanity check that entitlements do not exceed total available
float sum = 0.0f;
int nEnts = 0;
for ( int c = 0; c < nConsumers; c++ )
  if ( ents[c] >= 0.0f )
    {
    sum += ents[c];
    ++nEnts;
    }
if ( sum > total )
  // Raise the total to at least include the entitlements
  total = sum;

// Give consumers without explicit entitlements an equal share of whats left
float[] yents = new float[nConsumers];
for ( int c = 0; c < nConsumers; c++ )
  if ( ents[c] >= 0.0f )
    yents[c] = ents[c];
  else
    yents[c] = ( total - sum ) / ( nConsumers - nEnts );
//...e

    // Work out width of margin needed for vertical division labels
    int mw = fm.stringWidth(Integer.toString((int) total)+units);
    int mh = 2*ch;

    // Size left for the graph
    int gw = w - bw - nw - bw - mw - bw;
    int gh = h - bh - th - bh - mh - bh;

    // Fill in entire graphic background
    g.setColor(colorBitmap);
    g.fillRect(0, 0, w, h);

    // Color graph background
    g.setColor(colorGraph);
    g.fillRect(bw+nw+bw+mw+bw, bh+th+bh, gw, gh);

//...sdo vertical scale:6:
// Draw names of consumers
for ( int c = 0; c < nConsumers; c++ )
  {
  g.setColor(colorConsumers[c%colorConsumers.length]);
  g.drawString(c_all.elementAt(c), bw, bh+th+bh+gh-(gh*c)/nConsumers);
  }

// Work out size of a vertical division
int divsMax = gh / ch;
float vdiv = 1.0f;
for ( ;; )
  {
  if ( total / vdiv <= divsMax )
    break;
  vdiv *= 2.0;
  if ( total / vdiv <= divsMax )
    break;
  vdiv *= 2.5;
  if ( total / vdiv <= divsMax )
    break;
  vdiv *= 2.0;
  }

// Draw vertical divisions and lines
for ( float v = 0.0f; v <= total; v += vdiv )
  {
  int y = (int) ( gh*(v/total) );
  g.setColor(colorText);
  String vs = Integer.toString((int) v) + units;
  int sw = fm.stringWidth(vs);
  g.drawString(vs, bw+nw+bw+mw-sw, bh+th+bh+gh-y-1);
  g.setColor(colorGrid);
  g.fillRect(bw+nw+bw+mw+bw, bh+th+bh+gh-y-1, gw, 1);
  }
//...e
//...sdo time division lines and values:6:
// How many pixels needed for one division of each size
int tdwYear   = divWidth(st_cal, st_m, et_m, Calendar.YEAR        , gw);
int tdwMonth  = divWidth(st_cal, st_m, et_m, Calendar.MONTH       , gw);
int tdwDay    = divWidth(st_cal, st_m, et_m, Calendar.DAY_OF_MONTH, gw);
int tdwHour   = divWidth(st_cal, st_m, et_m, Calendar.HOUR_OF_DAY , gw);
int tdwMinute = divWidth(st_cal, st_m, et_m, Calendar.MINUTE      , gw);
int tdwSecond = divWidth(st_cal, st_m, et_m, Calendar.SECOND      , gw);

// Steps that make sense
int[] stepSeconds = { 5, 15, 30 };
int[] stepMinutes = { 1, 5, 15, 30 };
int[] stepHours   = { 1, 3, 6, 12 };
int[] stepDays    = { 1, 7 };
int[] stepMonths  = { 1, 3, 6 };
int[] stepYears   = { 1, 10, 100, 1000 };

// We can show any divisions of more than 20 pixels apart
int tdwMin = 20;
int lw = 1;
for ( int i = 0; i < stepSeconds.length; i++ )
  {
  int step = stepSeconds[i];
  if ( tdwSecond*step >= tdwMin )
    {
    divSeconds(g, bw+nw+bw+mw+bw, bh+th+bh, gw, gh, st_cal, st_m, et_cal, et_m, step, lw++);
    break;
    }
  }
for ( int i = 0; i < stepMinutes.length; i++ )
  {
  int step = stepMinutes[i];
  if ( tdwMinute*step >= tdwMin )
    {
    divMinutes(g, bw+nw+bw+mw+bw, bh+th+bh, gw, gh, st_cal, st_m, et_cal, et_m, step, lw++);
    break;
    }
  }
for ( int i = 0; i < stepHours.length; i++ )
  {
  int step = stepHours[i];
  if ( tdwHour*step >= tdwMin )
    {
    divHours(g, bw+nw+bw+mw+bw, bh+th+bh, gw, gh, st_cal, st_m, et_cal, et_m, step, lw++);
    break;
    }
  }
for ( int i = 0; i < stepDays.length; i++ )
  {
  int step = stepDays[i];
  if ( tdwDay*step >= tdwMin )
    {
    divDays(g, bw+nw+bw+mw+bw, bh+th+bh, gw, gh, st_cal, st_m, et_cal, et_m, step, lw++);
    break;
    }
  }
for ( int i = 0; i < stepMonths.length; i++ )
  {
  int step = stepMonths[i];
  if ( tdwMonth*step >= tdwMin )
    {
    divMonths(g, bw+nw+bw+mw+bw, bh+th+bh, gw, gh, st_cal, st_m, et_cal, et_m, step, lw++);
    break;
    }
  }
for ( int i = 0; i < stepYears.length; i++ )
  {
  int step = stepYears[i];
  if ( tdwYear*step >= tdwMin )
    {
    divYears(g, bw+nw+bw+mw+bw, bh+th+bh, gw, gh, st_cal, st_m, et_cal, et_m, step, lw++);
    break;
    }
  }

// We can show any labels that are sufficiently apart
int vy = ch;
if ( vy <= mh )
  for ( int i = 0; i < stepSeconds.length; i++ )
    {
    int step = stepSeconds[i];
    if ( tdwSecond*step >= 4*cw ) // "SSs "
      {
      valSeconds(g, bw+nw+bw+mw+bw, bh+th+bh, gw, gh+vy, st_cal, st_m, et_cal, et_m, step);
      vy += ch;
      break;
      }
    }
if ( vy <= mh )
  for ( int i = 0; i < stepMinutes.length; i++ )
    {
    int step = stepMinutes[i];
    if ( tdwMinute*step >= 4*cw ) // "MMm "
      {
      valMinutes(g, bw+nw+bw+mw+bw, bh+th+bh, gw, gh+vy, st_cal, st_m, et_cal, et_m, step);
      vy += ch;
      break;
      }
    }
if ( vy <= mh )
  for ( int i = 0; i < stepHours.length; i++ )
    {
    int step = stepHours[i];
    if ( tdwHour*step >= 4*cw ) // "HHh "
      {
      valHours(g, bw+nw+bw+mw+bw, bh+th+bh, gw, gh+vy, st_cal, st_m, et_cal, et_m, step);
      vy += ch;
      break;
      }
    }
if ( vy <= mh )
  for ( int i = 0; i < stepDays.length; i++ )
    {
    int step = stepDays[i];
    if ( tdwDay*step >= 5*cw ) // "DDth "
      {
      valDays(g, bw+nw+bw+mw+bw, bh+th+bh, gw, gh+vy, st_cal, st_m, et_cal, et_m, step);
      vy += ch;
      break;
      }
    }
if ( vy <= mh )
  for ( int i = 0; i < stepMonths.length; i++ )
    {
    int step = stepMonths[i];
    if ( tdwMonth*step >= 4*cw ) // "MON "
      {
      valMonths(g, bw+nw+bw+mw+bw, bh+th+bh, gw, gh+vy, st_cal, st_m, et_cal, et_m, step);
      vy += ch;
      break;
      }
    }
if ( vy <= mh )
  for ( int i = 0; i < stepYears.length; i++ )
    {
    int step = stepYears[i];
    if ( tdwYear*step >= 5*cw ) // "YYYY "
      {
      valYears(g, bw+nw+bw+mw+bw, bh+th+bh, gw, gh+vy, st_cal, st_m, et_cal, et_m, step);
      break;
      }
    }
//...e

    float[] ys = new float[nConsumers];
    float[] hs = new float[nConsumers];
    for ( int s = sFirst; s < sLast-1; s++ )
      {
      // Work out time range and x pixel range for sample
      Sample s1 = samples.elementAt(s);
      long m1 = s1.cal.getTime().getTime();
      int x1 = (int) ( (gw*(m1-st_m))/(et_m-st_m) );
      if ( x1 < 0 ) x1 = 0;
      Sample s2 = samples.elementAt(s+1);    
      long m2 = s2.cal.getTime().getTime();
      if ( m2 > m1 + interval )
        m2 = m1 + interval;
      int x2 = (int) ( (gw*(m2-st_m))/(et_m-st_m) );
      if ( x1 == x2 )
        ++x2;
      if ( x2 > gw ) x2 = gw;

      // Work out y pixel range desired
      if ( split )
        // Split out bars with gaps
        {
        float y = 0.0f, y2 = 0.0f;
        for ( int c = 0; c < nConsumers; c++ )
          {
          ys[c] = Math.max(y, y2);
          hs[c] = s1.values[c][inx];
          y2 = ys[c] + hs[c];
          y += yents[c];
          }
        if ( nConsumers >= 2 )
          // At least two consumers implies at least one place a gap could be
          while ( ys[nConsumers-1]+hs[nConsumers-1] > total )
            // All bars don't fit within the range
            {
            float gap = 0.0f;
            int c;
            for ( c = nConsumers-1; c > 0; c-- )
              {
              gap = ys[c] - ( ys[c-1]+hs[c-1] );
              if ( gap > 0.0f )
                break; // Found a gap
              }
            if ( c == 0 )
              break; // Found no gaps, give up
            for ( ; c < nConsumers; c++ )
              ys[c] -= gap;
            }
        }
      else
        {
        float y = 0.0f;
        for ( int c = 0; c < nConsumers; c++ )
          {
          ys[c] = y;
          hs[c] = s1.values[c][inx];
          y += hs[c];
          }
        }

      float factor = 1.0f;
      if ( nConsumers > 0 )
        {
        float highest = ys[nConsumers-1]+hs[nConsumers-1];
        // If resources overflow the scale, and we can squeeze to fit
        if ( highest > total && squeeze )
          // then scale all the values down
          // and give a visual clue we have done this
          {
          factor = total/highest;
          g.setColor(colorSqueezed);
          g.fillRect(bw+nw+bw+mw+bw+x1, bh+th+bh-4, x2-x1, 2);
          }
        }

      // Draw y pixel ranges
      for ( int c = 0; c < nConsumers; c++ )
        {
        int ytop = (int) ( gh*((ys[c]+hs[c])*factor/total) );
        int ybot = (int) ( gh*((ys[c]      )*factor/total) );
        g.setColor(colorConsumers[c%colorConsumers.length]);
        g.fillRect(bw+nw+bw+mw+bw+x1, bh+th+bh+gh-ytop, x2-x1, ytop-ybot);
        if ( ents[c] >= 0.0f && hs[c] > ents[c] )
          // Have exceeded entitlement
          // Give visual clue this has happened
          {
          int y = (int) ( gh*((ys[c]+ents[c])*factor/total) );
          g.setColor(colorConsumersExceed[c%colorConsumers.length]);
          g.fillRect(bw+nw+bw+mw+bw+x1, bh+th+bh+gh-y-1, x2-x1, 2);
          }
        }
      }
//...sgraph title \40\last\44\ to ensure visible\41\:6:
// Display title
int gtx = bw+nw+bw+mw+bw;
int gtw;
int gth = ch+fm.getDescent();
gtw = fm.stringWidth(ds);
g.setColor(colorTitleBg);
g.fillRect(gtx, bh, bw+gtw+bw, gth);
g.setColor(colorText);
g.drawString(ds, gtx+bw, bh+ch);
gtx += (bw+gtw+bw+bw);
gtw = fm.stringWidth(r);
g.setColor(colorTitleBg);
g.fillRect(gtx, bh, bw+gtw+bw, gth);
g.setColor(colorText);
g.drawString(r, gtx+bw, bh+ch);
gtx += (bw+gtw+bw+bw);
String ival = cal_to_str(st_cal)+" to "+cal_to_str(et_cal);
gtw = fm.stringWidth(ival);
g.setColor(colorTitleBg);
g.fillRect(gtx, bh, bw+gtw+bw, gth);
g.setColor(colorText);
g.drawString(ival, gtx+bw, bh+ch);
String label = getInitParameter("datasources."+ds+".label");
if ( label != null )
  {
  gtx += (bw+gtw+bw+bw);
  gtw = fm.stringWidth(label);
  g.setColor(colorTitleBg);
  g.fillRect(gtx, bh, bw+gtw+bw, gth);
  g.setColor(colorText);
  g.drawString(label, gtx+bw, bh+ch);
  }
//...e
    }

  ServletOutputStream os = resp.getOutputStream();

  // Use Mini-AWTs serialise to PNG capability
  resp.setContentType("image/png");
  new PngEncoder(image).encode(os);

  // Use Acmes serialise to GIF
  // resp.setContentType("image/gif");
  // new GifEncoder(image, os).encode();
  }
//...e
//...ssendStats:2:
//...sstatsRow:2:
protected String statsRow(Color col, String consumer, float min, float max, float range, float avg)
  {
  StringBuilder sb = new StringBuilder();
  Formatter f = new Formatter(sb, Locale.US);
  f.format("<TR>"+
    "<TD CLASS=\"stats-consumer\" STYLE=\"color:#%02X%02X%02X\">%s"+
    "<TD CLASS=\"stats-value\">%4.2f"+
    "<TD CLASS=\"stats-value\">%4.2f"+
    "<TD CLASS=\"stats-value\">%4.2f"+
    "<TD CLASS=\"stats-value\">%4.2f",
    col.getRed(), col.getGreen(), col.getBlue(),
    encodeHTML(consumer), min, max, range, avg);
  return f.toString();
  }
//...e

protected void sendStats(
  HttpServletResponse resp,
  String ds, String r,
  Calendar st_cal, Calendar et_cal
  )
  throws IOException
  {
  // Get consumers applicable to this datasource
  Vector<String> c_all = getInitParameterVector("consumers."+ds);
  int nConsumers = c_all.size();
  // Get consumers we're not interested in
  Vector<String> c_ignore = getInitParameterVector("consumers-ignore."+ds);
  // Sampling interval for this datasource, in milliseconds
  int interval = toInt(getInitParameter("interval."+ds), 65) * 1000;
  // Determine resource
  int inx = r_all.indexOf(r);
  if ( inx == -1 )
    return;
  long st_m = st_cal.getTime().getTime();
  long et_m = et_cal.getTime().getTime();
  if ( st_m + mintime > et_m )
    return;
  float[] mins   = new float[nConsumers];
  float[] maxs   = new float[nConsumers];
  float[] totals = new float[nConsumers];
  float summin = Float.POSITIVE_INFINITY;
  float summax = 0.0f;
  float sumtotal = 0.0f;
  float total_time = 0.0f;
  for ( int c = 0; c < nConsumers; c++ )
    mins[c] = Float.POSITIVE_INFINITY;
  synchronized ( this )
    {
    Vector<Sample> samples;
    try
      {
      samples = getSamples(ds, c_all, c_ignore);
      }
    catch ( IOException ioe )
      {
      return;
      }
    int sFirst = locateAfter(samples, st_cal);
    int sLast  = locateAfter(samples, et_cal);
    if ( sFirst >                0 ) --sFirst;
    if ( sLast  < samples.size()-2 ) ++sLast ;
    for ( int s = sFirst; s < sLast-1; s++ )
      {
      // Work out time range for sample
      Sample s1 = samples.elementAt(s);
      long m1 = s1.cal.getTime().getTime();
      Sample s2 = samples.elementAt(s+1);    
      long m2 = s2.cal.getTime().getTime();
      if ( m2 > m1 + interval )
        m2 = m1 + interval;
      if ( m1 < st_m ) m1 = st_m; else if ( m1 > et_m ) m1 = et_m;
      if ( m2 < st_m ) m2 = st_m; else if ( m2 > et_m ) m2 = et_m;
      float sumv = 0.0f;
      for ( int c = 0; c < nConsumers; c++ )
        {
        float v = s1.values[c][inx];
        if ( v < mins[c] )
          mins[c] = v;
        if ( v > maxs[c] )
          maxs[c] = v;
        totals[c] += ( v * (m2-m1) );
        sumv += v;
        }
      if ( sumv < summin )
        summin = sumv;
      if ( sumv > summax )
        summax = sumv;
      sumtotal += ( sumv * (m2-m1) );
      total_time += (m2-m1);
      }
    }

  if ( total_time == 0.0f )
    return; // avoid /0 bug

  ServletOutputStream os = resp.getOutputStream();
  os.println("<TABLE>");
  os.println("<TR><TH CLASS=\"stats-consumer\">Consumer<TH CLASS=\"stats-value\">Min<TH CLASS=\"stats-value\">Max<TH CLASS=\"stats-value\">Range<TH CLASS=\"stats-value\">Avg");
  for ( int c = nConsumers-1; c >= 0; c-- )
    os.println(statsRow(
      colorConsumers[c%colorConsumers.length],
      c_all.elementAt(c),
      mins[c], maxs[c], maxs[c]-mins[c], totals[c]/total_time));
  os.println(statsRow(Color.black, "Total", summin, summax, summax-summin,sumtotal/total_time));
  os.println("</TABLE>");
  }
//...e
//...ssendMain:2:
// Provide user interface to application, ie: act as controller.
// Maintain choices via parameters in URL string.
// Embed graphs corresponding to current user choices.
// Use a form to allow modifications to choices.

protected void sendMain(
  HttpServletRequest req, HttpServletResponse resp,
  String[] ds_a, String[] r_a,
  Calendar st_cal, Calendar et_cal,
  boolean split, boolean scale, boolean squeeze, boolean stats, boolean samples,
  int w, int h,
  boolean refresh
  )
  throws IOException
  {
  sendHeader(resp, null);
  ServletOutputStream os = resp.getOutputStream();

  String uri = req.getRequestURI();

  Vector<String> ds_show = new Vector<String>();
  if ( ds_a != null )
    for ( int i = 0; i < ds_a.length; i++ )
      ds_show.add(ds_a[i]);

  Vector<String> r_show = new Vector<String>();
  if ( r_a != null )
    for ( int i = 0; i < r_a.length; i++ )
      r_show.add(r_a[i]);

  String st_str = cal_to_str(st_cal);
  String et_str = cal_to_str(et_cal);

  if ( refresh )
    for ( int i = 0; i < ds_show.size(); i++ )
      {
      String ds = ds_show.elementAt(i);
        synchronized ( this )
          {
          dropSamples(ds);
          }
      }

  String ua,t1,t2,t3,t4;
  if ( (ua = req.getHeader("User-Agent")) != null &&
       ua.indexOf("Mozilla/4") != -1 &&
       ua.indexOf("MSIE") == -1 )
    // Netscape 4.x doesn't do tables within tables right
    // So we'll end up with the table below the graph
    { t1 = ""; t2 = "<BR>"; t3 = ""; t4 = ""; }
  else
    // Internet Explorer 6 and Opera 6 do nested tables ok
    // We'll assume other browsers do too
    // So we'll get the table to the right of the graph, as we'd like
    { t1 = "<TABLE>"; t2 = "<TR><TD>"; t3 = "<TD>"; t4 = "</TABLE>"; };

  os.println(t1);
  for ( int i = 0; i < ds_show.size(); i++ )
    {
    String ds = ds_show.elementAt(i);
    for ( int j = 0; j < r_show.size(); j++ )
      {
      String r = r_show.elementAt(j);
      os.println(t2+"<IMG SRC=\""+encodeHTML(uri)+"?a=graph"
        +"&ds="+encodeHTML(ds)
        +"&r=" +encodeHTML(r)
        +"&st="+st_str
        +"&et="+et_str
        + ( split   ? "&split=on"   : "" )
        + ( scale   ? "&scale=on"   : "" )
        + ( squeeze ? "&squeeze=on" : "" ) 
        +"&w="+w
        +"&h="+h
        +"\" WIDTH=\""+w+"\" HEIGHT=\""+h+"\">");
      if ( stats )
        {
        os.println(t3);
        sendStats(resp, ds, r, st_cal, et_cal);
        }
      if ( samples )
        {
        os.println(t3);
        os.println("<FORM ACTION=\""+encodeHTML(uri)+"\">");
        os.println("<INPUT TYPE=\"HIDDEN\" NAME=\"a\" VALUE=\"samples\">");
        os.println("<INPUT TYPE=\"HIDDEN\" NAME=\"ds\" VALUE=\""+encodeHTML(ds)+"\">");
        os.println("<INPUT CLASS=\"BUTTON\" TYPE=\"SUBMIT\" VALUE=\"Source\">");
        os.println("</FORM>");
        os.println("<FORM ACTION\""+encodeHTML(uri)+"\">");
        os.println("<INPUT TYPE=\"HIDDEN\" NAME=\"a\" VALUE=\"samples.csv\">");
        os.println("<INPUT TYPE=\"HIDDEN\" NAME=\"ds\" VALUE=\""+encodeHTML(ds)+"\">");
        os.println("<INPUT TYPE=\"HIDDEN\" NAME=\"r\" VALUE=\""+encodeHTML(r)+"\">");
        os.println("<INPUT TYPE=\"HIDDEN\" NAME=\"st\" VALUE=\""+st_str+"\">");
        os.println("<INPUT TYPE=\"HIDDEN\" NAME=\"et\" VALUE=\""+et_str+"\">");
        os.println("<INPUT CLASS=\"BUTTON\" TYPE=\"SUBMIT\" VALUE=\"CSV\">");
        os.println("</FORM>");
        }
      }
    }
  os.println(t4);

  if ( ds_show.size() == 0 || r_show.size() == 0 )
    os.println("<P>Please select the graphs you'd like to see from the form below.</P>");

  os.println("<FORM NAME=\"choiceForm\" ACTION=\""+encodeHTML(uri)+"\">");
  os.println("  <TABLE>");
  os.println("    <TR><TH>Datasource(s)<TH>Resource(s)<TH>From<TH>To<TH>Options<TH>Width<TH>Height");
  os.println("    <TR>");
  os.println("      <TD VALIGN=TOP>");
  os.println("        <SELECT NAME=\"ds\" MULTIPLE SIZE=\""+Math.min(nDatasources,10)+"\">");
  for ( int i = 0; i < ds_all.size(); i++ )
    {
    String ds = ds_all.elementAt(i);
    os.print("          <OPTION VALUE=\""+encodeHTML(ds)+"\"");
    if ( ds_show.contains(ds) )
      os.print(" SELECTED");
    os.println(">"+encodeHTML(ds));
    }
  os.println("        </SELECT>");
  os.println("      <TD VALIGN=TOP>");
  os.println("        <SELECT NAME=\"r\" MULTIPLE SIZE=\""+Math.min(nResources,10)+"\">");
  for ( int i = 0; i < r_all.size(); i++ )
    {
    String r = r_all.elementAt(i);
    os.print("          <OPTION VALUE=\""+encodeHTML(r)+"\"");
    if ( r_show.contains(r) )
      os.print(" SELECTED");
    os.println(">"+encodeHTML(r));
    }
  os.println("        </SELECT>");
  os.println("      <TD VALIGN=TOP>");
  os.println("                     <INPUT TYPE=\"RADIO\" NAME=\"str\" VALUE=\"\" onClick=\"Est()\" CHECKED>");
  os.println("                     <INPUT TYPE=\"TEXT\" NAME=\"st\" VALUE=\""+encodeHTML(st_str)+"\">");
  os.println("                 <BR><INPUT TYPE=\"RADIO\" NAME=\"str\" VALUE=\"h\" onClick=\"Dst()\"> hour before");
  os.println("                 <BR><INPUT TYPE=\"RADIO\" NAME=\"str\" VALUE=\"d\" onClick=\"Dst()\"> day before");
  os.println("                 <BR><INPUT TYPE=\"RADIO\" NAME=\"str\" VALUE=\"w\" onClick=\"Dst()\"> week before");
  os.println("                 <BR><INPUT TYPE=\"RADIO\" NAME=\"str\" VALUE=\"m\" onClick=\"Dst()\"> month before");
  os.println("                 <BR><INPUT TYPE=\"RADIO\" NAME=\"str\" VALUE=\"q\" onClick=\"Dst()\"> quarter before");
  os.println("                 <BR><INPUT TYPE=\"RADIO\" NAME=\"str\" VALUE=\"y\" onClick=\"Dst()\"> year before");
  os.println("      <TD VALIGN=TOP>");
  os.println("                     <INPUT TYPE=\"RADIO\" NAME=\"etr\" VALUE=\"\" onClick=\"Eet()\" CHECKED>");
  os.println("                     <INPUT TYPE=\"TEXT\" NAME=\"et\" VALUE=\""+encodeHTML(et_str)+"\">");
  os.println("                 <BR><INPUT TYPE=\"RADIO\" NAME=\"etr\" VALUE=\"n\" onClick=\"Det()\"> now");
  os.println("      <TD VALIGN=TOP><INPUT TYPE=\"CHECKBOX\" NAME=\"split\""   + ( split   ? " CHECKED" : "" ) + ">Split");
  os.println("                 <BR><INPUT TYPE=\"CHECKBOX\" NAME=\"scale\""   + ( scale   ? " CHECKED" : "" ) + ">Scale");
  os.println("                 <BR><INPUT TYPE=\"CHECKBOX\" NAME=\"squeeze\"" + ( squeeze ? " CHECKED" : "" ) + ">Squeeze");
  os.println("                 <BR><INPUT TYPE=\"CHECKBOX\" NAME=\"stats\""   + ( stats   ? " CHECKED" : "" ) + ">Stats");
  os.println("                 <BR><INPUT TYPE=\"CHECKBOX\" NAME=\"samples\"" + ( samples ? " CHECKED" : "" ) + ">Samples");
  os.println("      <TD VALIGN=TOP><INPUT TYPE=\"TEXT\" NAME=\"w\" VALUE=\""+w+"\" SIZE=\"5\">");
  os.println("      <TD VALIGN=TOP><INPUT TYPE=\"TEXT\" NAME=\"h\" VALUE=\""+h+"\" SIZE=\"5\">");
  os.println("  </TABLE>");
  os.println("  <P>");
  os.println("  <INPUT CLASS=\"button\" TYPE=\"SUBMIT\" VALUE=\"Redraw\">");
  os.println("  <INPUT CLASS=\"button\" TYPE=\"SUBMIT\" VALUE=\"Refresh and Redraw\" NAME=\"refresh\">");
  os.println("  <INPUT CLASS=\"button\" TYPE=\"RESET\" onClick=\"Est();Eet()\">");
  os.println("</FORM>");

  String help = "<P>Trend Server <A HREF=\"trendserver.htm\">documentation</A> is available";
  if ( doclinkname != null && doclinkurl != null )
    help += ( " and also a link to <A HREF=\""+encodeHTML(doclinkurl)+"\">"+encodeHTML(doclinkname)+"</A>" );
  os.println(help+".");

  os.println("<SCRIPT LANGUAGE=\"JavaScript\"><!--");
  os.println("function Dst() { window.document.choiceForm.st.disabled=true; }");
  os.println("function Est() { window.document.choiceForm.st.disabled=false; }");
  os.println("function Det() { window.document.choiceForm.et.disabled=true; }");
  os.println("function Eet() { window.document.choiceForm.et.disabled=false; }");
  os.println("--></SCRIPT>");

  sendTrailer(resp);
  }
//...e
//...sdoGet:2:
// Validate parameters we've been passed (to a degree)
// Select actual action to be performed.

protected void doGet(HttpServletRequest req, HttpServletResponse resp)
  throws ServletException, IOException
  {
  ServletConfig sc = getServletConfig();

  // Work out action
  String a = req.getParameter("a");

  // Work out datasources, or null if none specified
  String[] ds_a = req.getParameterValues("ds");

  // Work out resources, or null if none specified
  String[] r_a = req.getParameterValues("r");

  // Work out time to end at
  Calendar et_cal = null;
  String et_str;
  if ( (et_str = req.getParameter("etr")) != null && ! et_str.equals("") )
    et_cal = rad_to_cal(Calendar.getInstance(), et_str);
  else if ( (et_str = req.getParameter("et")) != null )
    et_cal = str_to_cal(et_str);
  if ( et_cal == null )
    et_cal = Calendar.getInstance();

  // Work out time to start at
  Calendar st_cal = null;
  String st_str;
  if ( (st_str = req.getParameter("str")) != null && ! st_str.equals("") )
    st_cal = rad_to_cal(cloneCalendar(et_cal), st_str);
  else if ( (st_str = req.getParameter("st")) != null )
    st_cal = str_to_cal(st_str);
  if ( st_cal == null )
    {
    st_cal = cloneCalendar(et_cal);
    st_cal.add(Calendar.DATE, -1);
    }

  boolean split   = ( req.getParameter("split"  ) != null );
  boolean scale   = ( req.getParameter("scale"  ) != null );
  boolean squeeze = ( req.getParameter("squeeze") != null );
  boolean stats   = ( req.getParameter("stats"  ) != null );
  boolean samples = ( req.getParameter("samples") != null );
  boolean refresh = ( req.getParameter("refresh") != null );

  // Work out graphic size, using defaults if not specified
  String w_str;
  if ( (w_str = req.getParameter("w")) == null )
    w_str = getInitParameter("graph.w");
  int w = toInt(w_str, defW);
  String h_str;
  if ( (h_str = req.getParameter("h")) == null )
    h_str = getInitParameter("graph.h");
  int h = toInt(h_str, defH);

  if ( ! st_cal.before(et_cal) )
    sendError(resp, "Start time must be before end time");
  else if ( w < minW || w > maxW || h < minH || h > maxH )
    sendError(resp, "Bad graphic size, pick between "+minW+"x"+minH+" to "+maxW+"x"+maxH);
  else
    {
         if ( a == null || a.equals("main") )
      sendMain(req, resp, ds_a, r_a, st_cal, et_cal, split, scale, squeeze, stats, samples, w, h, refresh);
    else if ( a.equals("graph") )
      sendGraph(req, resp, ds_a, r_a, st_cal, et_cal, split, scale, squeeze, w, h);
    else if ( a.equals("samples") )
      sendSamples(req, resp, ds_a);
    else if ( a.equals("samples.csv") )
      sendSamplesCsv(resp, ds_a, r_a, st_cal, et_cal);
    else
      sendUsage(req, resp);
    }
  }
//...e
//...sinit:2:
public void init()
  throws ServletException
  {
  super.init();
  String propsfn;
  if ( (propsfn = getServletConfig().getInitParameter("trend.propsfn")) != null )
    try
      {
      FileInputStream fis = new FileInputStream(propsfn);
      props = new Properties();
      props.load(fis);
      }
    catch ( IOException e )
      {
      throw new ServletException(
        "Can't load properties from "+propsfn+": "+e.getMessage());
      }
  name         = getInitParameter("name");
  doclinkname  = getInitParameter("doclink.name");
  doclinkurl   = getInitParameter("doclink.url");
  ds_all       = getInitParameterVector("datasources");
  nDatasources = ds_all.size();
  r_all        = getInitParameterVector("resources");
  nResources   = r_all.size();
  multipliers = new float[nResources];
  for ( int i = 0; i < nResources; i++ )
    multipliers[i] = toFloat(getInitParameter("resources."+r_all.elementAt(i)+".multiplier"), 1.0f);
  }
//...e
  }
