#!/bin/ksh
#
# TrendServerControl.sh - TrendServer control script
#
# Can be used to control a single default instance,
# or a selection of named instances.
#
# Has crude log management.
#

trendHome=`dirname $0`/..
cd $trendHome
fqTrendHome=`pwd -P`

mkdir -p $fqTrendHome/logs

#...splatform specifics:0:
case `uname -s` in
	Linux)
		ps='ps auxwww'
		awk=awk
		;;
	SunOS)
		ps='/usr/ucb/ps auxwww'
		awk=nawk
		;;
	AIX)
		ps='ps -ef'
		awk=awk
		;;
	Darwin)
		ps='ps auxwww'
		awk=awk
		;;
	*)
		ps='ps -ef'
		awk=awk
		;;
esac
#...e
#...sstart\44\ stop and status:0:
# Start, stop and status of background commands

get_pid()
	# $1=command
	{
	$ps | grep -F "$1" | grep -F -v grep | $awk ' { print $2 } '
	}

start_bg()
	# $1=start_command
	# $2=match_command
	# $3=log
	{
	pid=`get_pid "$2"`
	if [[ "$pid" = "" ]] ; then
		$1 < /dev/null >> $log 2>&1 &
		pid=`get_pid "$1"`
		wait=
		while [[ "$pid" = "" && "$wait" != "xxxxx" ]] ; do
			sleep 1
			wait="x$wait"
			pid=`get_pid "$2"`
		done
	fi
	if [[ "$pid" = "" ]] ; then
		return 1
	else
		return 0
	fi
	}

stop_bg()
	# $1=match_command
	{
	pid=`get_pid "$1"`
	if [[ "$pid" != "" ]] ; then
		kill $pid
		pid=`get_pid "$1"`
		wait=
		while [[ "$pid" != "" && "$wait" != "xxxxx" ]] ; do
			sleep 1
			wait="x$wait"
			pid=`get_pid "$1"`
		done
	fi
	if [[ "$pid" = "" ]] ; then
		return 0
	else
		return 1
	fi
	}

status_bg()
	# $1=match_command
	{
	if [[ `get_pid "$1"` != "" ]] ; then
		return 0
	else
		return 1
	fi
	}
#...e
#...slogging:0:
# Make an attempt to keep log files from growing indefinitely.

log_limit()
	# $1=logfile
	# $2=if_more_than
	# $3=then_trim_to
	{
	if [[ -r $1 && `wc -l < $1` -gt $2 ]] ; then
		tail -$3 $1 > $1-limited
		mv $1-limited $1
	fi
	}
#...e

#...sselect_instance:0:
select_instance()
	# $1=instance
	{
	# It might seem that $fqTabName/ at the start of these variables
	# is redundant, but by having it here we ensure that we can tell
	# process apart in the output of ps, allowing us to match and kill
	# only the right one.
	if [[ "$1" != "default" ]] ; then
		propsServer=$fqTrendHome/config/TrendServer-$1.props
		propsServlet=$fqTrendHome/config/TrendServlet-$1.props
		log=$fqTrendHome/logs/TrendServer-$1.log
	else
		propsServer=$fqTrendHome/config/TrendServer.props
		propsServlet=$fqTrendHome/config/TrendServlet.props
		log=$fqTrendHome/logs/TrendServer.log
	fi
	if [[ ! -r $propsServer || ! -r $propsServlet ]] ; then
		print "$0: $instance is not a valid instance"
		exit 2
	fi
	# Note: we need to get lib/* to the java command without expansion,
	# lib/* gets expanded by the shell, but luckily .:lib/* doesn't,
	# variants like 'lib/*' "lib/*" or lib\* don't work as required here.
	cmd="java -cp .:lib/* -Xmx256m TrendServer -propsServer $propsServer -propsServlet $propsServlet"
	}
#...e

cmd="$1"
shift

instances="$*"
if [[ "$instances" = "" ]] ; then
	if [[ -r config/TrendServer.props ]] ; then
		instances="$instances default"
	fi
	for props in `ls -1 config/TrendServer-*.props 2>/dev/null` ; do
		x=${props#config/TrendServer-}
		x=${x%.props}
		instances="$instances $x"
	done
fi

rc=0

case "$cmd" in
	start)
		for instance in $instances ; do
			select_instance "$instance"
			status_bg "$cmd"
			if [[ $? -eq 0 ]] ; then
				print "TrendServer $instance already started"
			else
				log_limit $log 1100000 1000000
				start_bg "$cmd" "$cmd" "$log"
				if [[ $? -eq 0 ]] ; then
					print "TrendServer $instance started"
				else
					print "TrendServer $instance failed to start"
					rc=1
				fi
			fi
		done
		;;
	stop)
		for instance in $instances ; do
			select_instance "$instance"
			status_bg "$cmd"
			if [[ $? -ne 0 ]] ; then
				print "TrendServer $instance already stopped"
			else
				stop_bg "$cmd"
				if [[ $? -eq 0 ]] ; then
					print "TrendServer $instance stopped"
				else
					print "TrendServer $instance failed to stop"
					rc=1
				fi
			fi
		done
		;;
	restart)
		for instance in $instances ; do
			select_instance "$instance"
			$0 stop "$instance"
			$0 start "$instance"
		done
		;;
	status)
		for instance in $instances ; do
			select_instance "$instance"
			status_bg "$cmd"
			if [[ $? -eq 0 ]] ; then
				print "TrendServer $instance running"
			else
				print "TrendServer $instance not running"
			fi
		done
		;;
	monitor)
		rc=110
		for instance in $instances ; do
			select_instance "$instance"
			status_bg "$cmd"
			if [[ $? -ne 0 ]] ; then
				rc=100
			fi
		done
		;;
	log)
		for instance in $instances ; do
			select_instance "$instance"
			tail -f $log
		done
		;;
	*)
		echo "usage: $0 (start|stop|restart|status|monitor|log) {instance}"
		;;
esac

exit $rc
